"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaRuntime = exports.LambdaFunction = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const case_1 = require("case");
const internal_1 = require("./internal");
const component_1 = require("../component");
const javascript_1 = require("../javascript");
const source_code_1 = require("../source-code");
/**
 * Generates a pre-bundled AWS Lambda function construct from handler code.
 *
 * To use this, create an AWS Lambda handler file under your source tree with
 * the `.lambda.ts` extension and add a `LambdaFunction` component to your
 * typescript project pointing to this entrypoint.
 *
 * This will add a task to your "compile" step which will use `esbuild` to
 * bundle the handler code into the build directory. It will also generate a
 * file `src/foo-function.ts` with a custom AWS construct called `FooFunction`
 * which extends `@aws-cdk/aws-lambda.Function` which is bound to the bundled
 * handle through an asset.
 *
 * @example
 *
 * new LambdaFunction(myProject, {
 *   srcdir: myProject.srcdir,
 *   entrypoint: 'src/foo.lambda.ts',
 * });
 */
class LambdaFunction extends component_1.Component {
    /**
     * Defines a pre-bundled AWS Lambda function construct from handler code.
     *
     * @param project The project to use
     * @param options Options
     */
    constructor(project, options) {
        super(project);
        const cdkDeps = options.cdkDeps;
        const bundler = javascript_1.Bundler.of(project);
        if (!bundler) {
            throw new Error("No bundler found. Please add a Bundler component to your project.");
        }
        const runtime = options.runtime ?? LambdaRuntime.NODEJS_14_X;
        // allow Lambda handler code to import dev-deps since they are only needed
        // during bundling
        const eslint = javascript_1.Eslint.of(project);
        eslint?.allowDevDeps(options.entrypoint);
        const entrypoint = options.entrypoint;
        if (!entrypoint.endsWith(internal_1.TYPESCRIPT_LAMBDA_EXT) &&
            !entrypoint.endsWith(internal_1.TYPESCRIPT_EDGE_LAMBDA_EXT)) {
            throw new Error(`${entrypoint} must have a ${internal_1.TYPESCRIPT_LAMBDA_EXT} or ${internal_1.TYPESCRIPT_EDGE_LAMBDA_EXT} extension`);
        }
        const basePath = path.posix.join(path.dirname(entrypoint), path.basename(entrypoint, options.edgeLambda ? internal_1.TYPESCRIPT_EDGE_LAMBDA_EXT : internal_1.TYPESCRIPT_LAMBDA_EXT));
        const constructFile = options.constructFile ?? `${basePath}-function.ts`;
        if (path.extname(constructFile) !== ".ts") {
            throw new Error(`Construct file name "${constructFile}" must have a .ts extension`);
        }
        // type names
        const constructName = options.constructName ?? case_1.pascal(path.basename(basePath)) + "Function";
        const propsType = `${constructName}Props`;
        const bundle = bundler.addBundle(entrypoint, {
            target: runtime.esbuildTarget,
            platform: runtime.esbuildPlatform,
            externals: ["aws-sdk"],
            ...options.bundlingOptions,
            tsconfigPath: project?.tsconfigDev?.fileName,
        });
        // calculate the relative path between the directory containing the
        // generated construct source file to the directory containing the bundle
        // index.js by resolving them as absolute paths first.
        // e.g:
        //  - outfileAbs => `/project-outdir/assets/foo/bar/baz/foo-function/index.js`
        //  - constructAbs => `/project-outdir/src/foo/bar/baz/foo-function.ts`
        const outfileAbs = path.join(project.outdir, bundle.outfile);
        const constructAbs = path.join(project.outdir, constructFile);
        const relativeOutfile = path.relative(path.dirname(constructAbs), path.dirname(outfileAbs));
        const src = new source_code_1.SourceCode(project, constructFile);
        if (src.marker) {
            src.line(`// ${src.marker}`);
        }
        src.line("import * as path from 'path';");
        if (cdkDeps.cdkMajorVersion === 1) {
            if (options.edgeLambda) {
                src.line("import * as cloudfront from '@aws-cdk/aws-cloudfront';");
                cdkDeps.addV1Dependencies("@aws-cdk/aws-cloudfront");
            }
            src.line("import * as lambda from '@aws-cdk/aws-lambda';");
            src.line("import { Construct } from '@aws-cdk/core';");
            cdkDeps.addV1Dependencies("@aws-cdk/aws-lambda");
            cdkDeps.addV1Dependencies("@aws-cdk/core");
        }
        else {
            if (options.edgeLambda) {
                src.line("import * as cloudfront from 'aws-cdk-lib/aws-cloudfront';");
            }
            src.line("import * as lambda from 'aws-cdk-lib/aws-lambda';");
            src.line("import { Construct } from 'constructs';");
        }
        src.line();
        src.line("/**");
        src.line(` * Props for ${constructName}`);
        src.line(" */");
        if (options.edgeLambda) {
            src.open(`export interface ${propsType} extends cloudfront.experimental.EdgeFunctionProps {`);
        }
        else {
            src.open(`export interface ${propsType} extends lambda.FunctionOptions {`);
        }
        src.close("}");
        src.line();
        src.line("/**");
        src.line(` * An AWS Lambda function which executes ${internal_1.convertToPosixPath(basePath)}.`);
        src.line(" */");
        if (options.edgeLambda) {
            src.open(`export class ${constructName} extends cloudfront.experimental.EdgeFunction {`);
        }
        else {
            src.open(`export class ${constructName} extends lambda.Function {`);
        }
        src.open(`constructor(scope: Construct, id: string, props?: ${propsType}) {`);
        src.open("super(scope, id, {");
        src.line(`description: '${internal_1.convertToPosixPath(entrypoint)}',`);
        src.line("...props,");
        src.line(`runtime: new lambda.Runtime('${runtime.functionRuntime}', lambda.RuntimeFamily.NODEJS),`);
        src.line("handler: 'index.handler',");
        src.line(`code: lambda.Code.fromAsset(path.join(__dirname, '${internal_1.convertToPosixPath(relativeOutfile)}')),`);
        src.close("});");
        if ((options.awsSdkConnectionReuse ?? true) && !options.edgeLambda) {
            src.line("this.addEnvironment('AWS_NODEJS_CONNECTION_REUSE_ENABLED', '1', { removeInEdge: true });");
        }
        src.close("}");
        src.close("}");
        this.project.logger.verbose(`${basePath}: construct "${constructName}" generated under "${constructFile}"`);
        this.project.logger.verbose(`${basePath}: bundle task "${bundle.bundleTask.name}"`);
        if (bundle.watchTask) {
            this.project.logger.verbose(`${basePath}: bundle watch task "${bundle.watchTask.name}"`);
        }
    }
}
exports.LambdaFunction = LambdaFunction;
_a = JSII_RTTI_SYMBOL_1;
LambdaFunction[_a] = { fqn: "projen.awscdk.LambdaFunction", version: "0.67.0" };
/**
 * The runtime for the AWS Lambda function.
 */
class LambdaRuntime {
    constructor(
    /**
     * The Node.js runtime to use
     */
    functionRuntime, 
    /**
     * The esbuild setting to use.
     */
    esbuildTarget) {
        this.functionRuntime = functionRuntime;
        this.esbuildTarget = esbuildTarget;
        this.esbuildPlatform = "node";
    }
}
exports.LambdaRuntime = LambdaRuntime;
_b = JSII_RTTI_SYMBOL_1;
LambdaRuntime[_b] = { fqn: "projen.awscdk.LambdaRuntime", version: "0.67.0" };
/**
 * Node.js 10.x
 * @deprecated NodeJS10 has been deprecated
 */
LambdaRuntime.NODEJS_10_X = new LambdaRuntime("nodejs10.x", "node10");
/**
 * Node.js 12.x
 */
LambdaRuntime.NODEJS_12_X = new LambdaRuntime("nodejs12.x", "node12");
/**
 * Node.js 14.x
 */
LambdaRuntime.NODEJS_14_X = new LambdaRuntime("nodejs14.x", "node14");
/**
 * Node.js 16.x
 */
LambdaRuntime.NODEJS_16_X = new LambdaRuntime("nodejs16.x", "node16");
/**
 * Node.js 18.x
 */
LambdaRuntime.NODEJS_18_X = new LambdaRuntime("nodejs18.x", "node18");
//# sourceMappingURL=data:application/json;base64,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