"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NpmJs = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_events_1 = require("@aws-cdk/aws-events");
const aws_events_targets_1 = require("@aws-cdk/aws-events-targets");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_lambda_event_sources_1 = require("@aws-cdk/aws-lambda-event-sources");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const aws_sqs_1 = require("@aws-cdk/aws-sqs");
const core_1 = require("@aws-cdk/core");
const deep_link_1 = require("../deep-link");
const metric_utils_1 = require("../metric-utils");
const runbook_url_1 = require("../runbook-url");
const canary_1 = require("./npmjs/canary");
const constants_lambda_shared_1 = require("./npmjs/constants.lambda-shared");
const npm_js_follower_1 = require("./npmjs/npm-js-follower");
const stage_and_notify_1 = require("./npmjs/stage-and-notify");
/**
 * (experimental) A package source that gets package data from the npmjs.com package registry.
 *
 * @experimental
 */
class NpmJs {
    /**
     * @experimental
     */
    constructor(props = {}) {
        this.props = props;
    }
    /**
     * (experimental) Binds the package source to a scope and target queue.
     *
     * @experimental
     */
    bind(scope, { baseUrl, denyList, ingestion, licenseList, monitoring, queue, repository }) {
        var _b, _c, _d;
        repository === null || repository === void 0 ? void 0 : repository.addExternalConnection('public:npmjs');
        const bucket = this.props.stagingBucket || new aws_s3_1.Bucket(scope, 'NpmJs/StagingBucket', {
            blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL,
            enforceSSL: true,
            lifecycleRules: [{ prefix: "staged/" /* STAGED_KEY_PREFIX */, expiration: core_1.Duration.days(30) }],
        });
        bucket.grantRead(ingestion);
        const stager = new stage_and_notify_1.StageAndNotify(scope, 'NpmJs-StageAndNotify', {
            deadLetterQueue: new aws_sqs_1.Queue(scope, 'StagerDLQ', {
                encryption: aws_sqs_1.QueueEncryption.KMS_MANAGED,
                retentionPeriod: core_1.Duration.days(14),
                visibilityTimeout: core_1.Duration.minutes(15),
            }),
            description: `[${scope.node.path}/NpmJS-StageAndNotify] Stages tarballs to S3 and notifies ConstructHub`,
            environment: {
                AWS_EMF_ENVIRONMENT: 'Local',
                BUCKET_NAME: bucket.bucketName,
                QUEUE_URL: queue.queueUrl,
            },
            memorySize: 10024,
            retryAttempts: 2,
            timeout: core_1.Duration.minutes(5),
            tracing: aws_lambda_1.Tracing.ACTIVE,
        });
        bucket.grantReadWrite(stager);
        denyList === null || denyList === void 0 ? void 0 : denyList.grantRead(stager);
        queue.grantSendMessages(stager);
        stager.addEventSource(new aws_lambda_event_sources_1.SqsEventSource(stager.deadLetterQueue, { batchSize: 1, enabled: false }));
        const follower = new npm_js_follower_1.NpmJsFollower(scope, 'NpmJs', {
            description: `[${scope.node.path}/NpmJs] Periodically query npmjs.com index for new packages`,
            environment: {
                AWS_EMF_ENVIRONMENT: 'Local',
                BUCKET_NAME: bucket.bucketName,
                FUNCTION_NAME: stager.functionName,
            },
            memorySize: 10024,
            reservedConcurrentExecutions: 1,
            timeout: core_1.Duration.minutes(5),
            tracing: aws_lambda_1.Tracing.ACTIVE,
        });
        bucket.grantReadWrite(follower, constants_lambda_shared_1.MARKER_FILE_NAME);
        denyList === null || denyList === void 0 ? void 0 : denyList.grantRead(follower);
        licenseList.grantRead(follower);
        stager.grantInvoke(follower);
        const rule = new aws_events_1.Rule(scope, 'NpmJs/Schedule', {
            description: `${scope.node.path}/NpmJs/Schedule`,
            schedule: aws_events_1.Schedule.rate(core_1.Duration.minutes(5)),
            targets: [new aws_events_targets_1.LambdaFunction(follower)],
        });
        this.registerAlarms(scope, follower, stager, monitoring, rule);
        return {
            name: follower.node.path,
            links: [
                { name: 'NpmJs Follower', url: deep_link_1.lambdaFunctionUrl(follower), primary: true },
                { name: 'Marker Object', url: deep_link_1.s3ObjectUrl(bucket, constants_lambda_shared_1.MARKER_FILE_NAME) },
                { name: 'Stager', url: deep_link_1.lambdaFunctionUrl(stager) },
                { name: 'Stager DLQ', url: deep_link_1.sqsQueueUrl(stager.deadLetterQueue) },
            ],
            dashboardWidgets: [
                [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Follower Health',
                        left: [
                            metric_utils_1.fillMetric(follower.metricInvocations({ label: 'Invocations' })),
                            metric_utils_1.fillMetric(follower.metricErrors({ label: 'Errors' })),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            this.metricRemainingTime({ label: 'Remaining Time' }),
                        ],
                        rightYAxis: { min: 0 },
                        period: core_1.Duration.minutes(5),
                    }),
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Stager Health',
                        left: [
                            metric_utils_1.fillMetric(stager.metricInvocations({ label: 'Invocations' })),
                            metric_utils_1.fillMetric(stager.metricErrors({ label: 'Errors' })),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            stager.metricDuration({ label: 'Duration' }),
                        ],
                        rightYAxis: { min: 0 },
                        period: core_1.Duration.minutes(5),
                    }),
                ], [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'CouchDB Follower',
                        left: [
                            metric_utils_1.fillMetric(this.metricChangeCount({ label: 'Change Count' }), 0),
                            metric_utils_1.fillMetric(this.metricUnprocessableEntity({ label: 'Unprocessable' }), 0),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            metric_utils_1.fillMetric(this.metricNpmJsChangeAge({ label: 'Lag to npmjs.com' }), 'REPEAT'),
                            metric_utils_1.fillMetric(this.metricPackageVersionAge({ label: 'Package Version Age' }), 'REPEAT'),
                        ],
                        rightYAxis: { label: 'Milliseconds', min: 0, showUnits: false },
                        period: core_1.Duration.minutes(5),
                    }),
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'CouchDB Changes',
                        left: [
                            metric_utils_1.fillMetric(this.metricLastSeq({ label: 'Last Sequence Number' }), 'REPEAT'),
                        ],
                        period: core_1.Duration.minutes(5),
                    }),
                ],
                [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Stager Dead-Letter Queue',
                        left: [
                            metric_utils_1.fillMetric(stager.deadLetterQueue.metricApproximateNumberOfMessagesVisible({ label: 'Visible Messages' }), 0),
                            metric_utils_1.fillMetric(stager.deadLetterQueue.metricApproximateNumberOfMessagesNotVisible({ label: 'Invisible Messages' }), 0),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            stager.deadLetterQueue.metricApproximateAgeOfOldestMessage({ label: 'Oldest Message' }),
                        ],
                        rightYAxis: { min: 0 },
                        period: core_1.Duration.minutes(1),
                    }),
                    ...(((_b = this.props.enableCanary) !== null && _b !== void 0 ? _b : true)
                        ? this.registerCanary(follower, (_c = this.props.canaryPackage) !== null && _c !== void 0 ? _c : 'construct-hub-probe', (_d = this.props.canarySla) !== null && _d !== void 0 ? _d : core_1.Duration.minutes(5), bucket, baseUrl, monitoring)
                        : []),
                ],
            ],
        };
    }
    /**
     * (experimental) The average time it took to process a changes batch.
     *
     * @experimental
     */
    metricBatchProcessingTime(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(1),
            statistic: aws_cloudwatch_1.Statistic.AVERAGE,
            ...opts,
            metricName: "BatchProcessingTime" /* BATCH_PROCESSING_TIME */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * (experimental) The total count of changes that were processed.
     *
     * @experimental
     */
    metricChangeCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(1),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "ChangeCount" /* CHANGE_COUNT */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * (experimental) The last sequence number that was processed.
     *
     * This metric can be used to
     * discover when a sequence reset has happened in the CouchDB instance.
     *
     * @experimental
     */
    metricLastSeq(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(1),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "LastSeq" /* LAST_SEQ */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * @experimental
     */
    metricNpmJsChangeAge(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(1),
            statistic: aws_cloudwatch_1.Statistic.MINIMUM,
            ...opts,
            metricName: "NpmJsChangeAge" /* NPMJS_CHANGE_AGE */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * (experimental) The age of the oldest package version that was processed.
     *
     * @experimental
     */
    metricPackageVersionAge(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(1),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "PackageVersionAge" /* PACKAGE_VERSION_AGE */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * (experimental) The total count of package versions that were inspected.
     *
     * @experimental
     */
    metricPackageVersionCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(1),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "PackageVersionCount" /* PACKAGE_VERSION_COUNT */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * (experimental) The total count of package versions that were deemed relevant.
     *
     * @experimental
     */
    metricRelevantPackageVersions(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(1),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "RelevantPackageVersions" /* RELEVANT_PACKAGE_VERSIONS */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * (experimental) The amount of time that was remaining when the lambda returned in order to avoid hitting a timeout.
     *
     * @experimental
     */
    metricRemainingTime(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MINIMUM,
            ...opts,
            metricName: "RemainingTime" /* REMAINING_TIME */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * (experimental) The amount of changes that were not processed due to having an invalid format.
     *
     * @experimental
     */
    metricUnprocessableEntity(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(1),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "UnprocessableEntity" /* UNPROCESSABLE_ENTITY */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    registerAlarms(scope, follower, stager, monitoring, schedule) {
        const failureAlarm = follower.metricErrors().createAlarm(scope, 'NpmJs/Follower/Failures', {
            alarmName: `${scope.node.path}/NpmJs/Follower/Failures`,
            alarmDescription: [
                'The NpmJs follower function failed!',
                '',
                `RunBook: ${runbook_url_1.RUNBOOK_URL}`,
                '',
                `Direct link to Lambda function: ${deep_link_1.lambdaFunctionUrl(follower)}`,
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 3,
            threshold: 1,
            treatMissingData: aws_cloudwatch_1.TreatMissingData.MISSING,
        });
        monitoring.addHighSeverityAlarm('NpmJs/Follower Failures', failureAlarm);
        const notRunningAlarm = follower.metricInvocations().createAlarm(scope, 'NpmJs/Follower/NotRunning', {
            alarmName: `${scope.node.path}/NpmJs/Follower/NotRunning`,
            alarmDescription: [
                'The NpmJs follower function is not running!',
                '',
                `RunBook: ${runbook_url_1.RUNBOOK_URL}`,
                '',
                `Direct link to Lambda function: ${deep_link_1.lambdaFunctionUrl(follower)}`,
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.LESS_THAN_THRESHOLD,
            evaluationPeriods: 2,
            threshold: 1,
            treatMissingData: aws_cloudwatch_1.TreatMissingData.BREACHING,
        });
        monitoring.addHighSeverityAlarm('NpmJs/Follower Not Running', notRunningAlarm);
        const noChangeAlarm = this.metricChangeCount().createAlarm(scope, 'NpmJs/Follower/NoChanges', {
            alarmName: `${scope.node.path}/NpmJs/Follower/NoChanges`,
            alarmDescription: [
                'The NpmJs follower function is no discovering any changes from CouchDB!',
                '',
                `RunBook: ${runbook_url_1.RUNBOOK_URL}`,
                '',
                `Direct link to Lambda function: ${deep_link_1.lambdaFunctionUrl(follower)}`,
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.LESS_THAN_THRESHOLD,
            evaluationPeriods: 2,
            threshold: 1,
            // If the metric is not emitted, it can be assumed to be zero.
            treatMissingData: aws_cloudwatch_1.TreatMissingData.BREACHING,
        });
        monitoring.addLowSeverityAlarm('Np npmjs.com changes discovered', noChangeAlarm);
        const dlqNotEmptyAlarm = new aws_cloudwatch_1.MathExpression({
            expression: 'mVisible + mHidden',
            usingMetrics: {
                mVisible: stager.deadLetterQueue.metricApproximateNumberOfMessagesVisible({ period: core_1.Duration.minutes(1) }),
                mHidden: stager.deadLetterQueue.metricApproximateNumberOfMessagesNotVisible({ period: core_1.Duration.minutes(1) }),
            },
        }).createAlarm(scope, `${scope.node.path}/NpmJs/Stager/DLQNotEmpty`, {
            alarmName: `${scope.node.path}/NpmJs/Stager/DLQNotEmpty`,
            alarmDescription: [
                'The NpmJS package stager is failing - its dead letter queue is not empty',
                '',
                `Link to the lambda function: ${deep_link_1.lambdaFunctionUrl(stager)}`,
                `Link to the dead letter queue: ${deep_link_1.sqsQueueUrl(stager.deadLetterQueue)}`,
                '',
                `Runbook: ${runbook_url_1.RUNBOOK_URL}`,
            ].join('/n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 1,
            treatMissingData: aws_cloudwatch_1.TreatMissingData.NOT_BREACHING,
        });
        monitoring.addHighSeverityAlarm('NpmJs/Stager DLQ Not Empty', dlqNotEmptyAlarm);
        // Finally - the "not running" alarm depends on the schedule (it won't run until the schedule
        // exists!), and the schedule depends on the failure alarm existing (we don't want it to run
        // before we can know it is failing). This means the returned `IDependable` effectively ensures
        // all alarms have been provisionned already! Isn't it nice!
        notRunningAlarm.node.addDependency(schedule);
        schedule.node.addDependency(failureAlarm);
    }
    registerCanary(scope, packageName, visibilitySla, bucket, constructHubBaseUrl, monitoring) {
        const canary = new canary_1.NpmJsPackageCanary(scope, 'Canary', { bucket, constructHubBaseUrl, packageName });
        const alarm = new aws_cloudwatch_1.MathExpression({
            expression: 'MAX([mDwell, mTTC])',
            period: core_1.Duration.minutes(1),
            usingMetrics: {
                mDwell: canary.metricDwellTime(),
                mTTC: canary.metricTimeToCatalog(),
            },
        }).createAlarm(canary, 'Alarm', {
            alarmName: `${canary.node.path}/SLA-Breached`,
            alarmDescription: [
                `New versions of ${packageName} have been published over ${visibilitySla.toHumanString()} ago and are still not visible in construct hub`,
                `Runbook: ${runbook_url_1.RUNBOOK_URL}`,
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_THRESHOLD,
            evaluationPeriods: 2,
            // If there is no data, the canary might not be running, so... *Chuckles* we're in danger!
            treatMissingData: aws_cloudwatch_1.TreatMissingData.BREACHING,
            threshold: visibilitySla.toSeconds(),
        });
        monitoring.addLowSeverityAlarm('New version visibility SLA breached', alarm);
        return [
            new aws_cloudwatch_1.GraphWidget({
                height: 6,
                width: 12,
                title: 'Package Canary',
                left: [
                    canary.metricDwellTime({ label: 'Dwell Time' }),
                    canary.metricTimeToCatalog({ label: 'Time to Catalog' }),
                ],
                leftAnnotations: [{
                        color: '#ff0000',
                        label: `SLA (${visibilitySla.toHumanString()})`,
                        value: visibilitySla.toSeconds(),
                    }],
                leftYAxis: { min: 0 },
                right: [
                    canary.metricTrackedVersionCount({ label: 'Tracked Version Count' }),
                ],
                rightYAxis: { min: 0 },
            }),
        ];
    }
}
exports.NpmJs = NpmJs;
_a = JSII_RTTI_SYMBOL_1;
NpmJs[_a] = { fqn: "construct-hub.sources.NpmJs", version: "0.3.117" };
//# sourceMappingURL=data:application/json;base64,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