"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
const semver_1 = require("semver");
const aws = require("../shared/aws.lambda-shared");
const compress_content_lambda_shared_1 = require("../shared/compress-content.lambda-shared");
const constants = require("../shared/constants");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
const language_1 = require("../shared/language");
const constants_1 = require("./constants");
aws_embedded_metrics_1.Configuration.namespace = constants_1.METRICS_NAMESPACE;
async function handler(event, context) {
    var _a;
    console.log('Event:', JSON.stringify(event, null, 2));
    const indexedPackages = new Map();
    const packageNames = new Set();
    const packageMajorVersions = new Set();
    const perLanguage = new Map();
    /**
     * Records the status of a particular package, package major version, package
     * version, and package version submodule in the per-language state storage.
     * Whenever a new entry is added, a `MISSING` entry is automatically inserted
     * for the other languages (unless another entry already exists).
     *
     * If a submodule is provided, only that submodule's availability is updated.
     */
    function recordPerLanguage(language, status, pkgName, pkgMajor, pkgVersion, submodule) {
        for (const lang of language_1.DocumentationLanguage.ALL) {
            doRecordPerLanguage(perLanguage, lang, 
            // If the language is NOT the registered one, then we insert "MISSING".
            lang === language ? status : "Missing" /* MISSING */, pkgName, pkgMajor, pkgVersion, submodule);
        }
    }
    const bucket = env_lambda_shared_1.requireEnv('BUCKET_NAME');
    for await (const key of relevantObjectKeys(bucket)) {
        const [, name, version] = constants.STORAGE_KEY_FORMAT_REGEX.exec(key);
        packageNames.add(name);
        const majorVersion = `${name}@${new semver_1.SemVer(version).major}`;
        packageMajorVersions.add(majorVersion);
        const fullName = `${name}@${version}`;
        // Ensure the package is fully registered for per-language status, even if no doc exists yet.
        for (const language of language_1.DocumentationLanguage.ALL) {
            recordPerLanguage(language, "Missing" /* MISSING */, name, majorVersion, fullName);
        }
        if (!indexedPackages.has(fullName)) {
            indexedPackages.set(fullName, {});
        }
        const status = indexedPackages.get(fullName);
        if (key.endsWith(constants.METADATA_KEY_SUFFIX)) {
            status.metadataPresent = true;
        }
        else if (key.endsWith(constants.PACKAGE_KEY_SUFFIX)) {
            status.tarballPresent = true;
        }
        else if (key.endsWith(constants.ASSEMBLY_KEY_SUFFIX)) {
            status.assemblyPresent = true;
        }
        else if (key.endsWith(constants.UNINSTALLABLE_PACKAGE_SUFFIX)) {
            status.uninstallable = true;
        }
        else {
            let identified = false;
            for (const language of language_1.DocumentationLanguage.ALL) {
                const match = submoduleKeyRegexp(language).exec(key);
                if (match != null) {
                    const [, submodule, isUnsupported] = match;
                    if (status.submodules == null) {
                        status.submodules = new Set();
                    }
                    status.submodules.add(`${fullName}.${submodule}`);
                    recordPerLanguage(language, isUnsupported ? "Unsupported" /* UNSUPPORTED */ : "Supported" /* SUPPORTED */, name, majorVersion, fullName, submodule);
                    identified = true;
                }
                else if (key.endsWith(constants.docsKeySuffix(language))) {
                    recordPerLanguage(language, "Supported" /* SUPPORTED */, name, majorVersion, fullName);
                    identified = true;
                }
                else if (key.endsWith(constants.docsKeySuffix(language) + constants.NOT_SUPPORTED_SUFFIX)) {
                    recordPerLanguage(language, "Unsupported" /* UNSUPPORTED */, name, majorVersion, fullName);
                    identified = true;
                }
                else if (key.endsWith(constants.docsKeySuffix(language) + constants.CORRUPT_ASSEMBLY_SUFFIX)) {
                    recordPerLanguage(language, "CorruptAssembly" /* CORRUPT_ASSEMBLY */, name, majorVersion, fullName);
                    identified = true;
                }
            }
            if (!identified) {
                status.unknownObjects = (_a = status.unknownObjects) !== null && _a !== void 0 ? _a : [];
                status.unknownObjects.push(key);
            }
        }
    }
    const reports = [];
    function createReport(reportKey, packageVersions) {
        const report = JSON.stringify(packageVersions, null, 2);
        const { buffer, contentEncoding } = compress_content_lambda_shared_1.compressContent(Buffer.from(report));
        console.log(`Uploading list to s3://${bucket}/${reportKey}`);
        reports.push(aws.s3().putObject({
            Body: buffer,
            Bucket: bucket,
            ContentEncoding: contentEncoding,
            ContentType: 'application/json',
            Expires: new Date(Date.now() + 300000),
            Key: reportKey,
            Metadata: {
                'Lambda-Run-Id': context.awsRequestId,
                'Lambda-Log-Group-Name': context.logGroupName,
                'Lambda-Log-Stream-Name': context.logStreamName,
            },
        }).promise());
    }
    await aws_embedded_metrics_1.metricScope((metrics) => () => {
        var _a, _b, _c;
        // Clear out default dimensions as we don't need those. See https://github.com/awslabs/aws-embedded-metrics-node/issues/73.
        metrics.setDimensions();
        const missingMetadata = new Array();
        const missingAssembly = new Array();
        const missingTarball = new Array();
        const uninstallable = new Array();
        const unknownObjects = new Array();
        const submodules = new Array();
        for (const [name, status] of indexedPackages.entries()) {
            if (!status.metadataPresent) {
                missingMetadata.push(name);
            }
            if (!status.assemblyPresent) {
                missingAssembly.push(name);
            }
            if (!status.tarballPresent) {
                missingTarball.push(name);
            }
            if (status.uninstallable) {
                uninstallable.push(name);
            }
            if ((_b = (_a = status.unknownObjects) === null || _a === void 0 ? void 0 : _a.length) !== null && _b !== void 0 ? _b : 0 > 0) {
                unknownObjects.push(...status.unknownObjects);
            }
            for (const submodule of (_c = status.submodules) !== null && _c !== void 0 ? _c : []) {
                submodules.push(submodule);
            }
        }
        metrics.setProperty('detail', { missingMetadata, missingAssembly, missingTarball, unknownObjects });
        metrics.putMetric("UninstallablePackageCount" /* UNINSTALLABLE_PACKAGE_COUNT */, uninstallable.length, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("MissingPackageMetadataCount" /* MISSING_METADATA_COUNT */, missingMetadata.length, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("MissingAssemblyCount" /* MISSING_ASSEMBLY_COUNT */, missingAssembly.length, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("MissingPackageTarballCount" /* MISSING_TARBALL_COUNT */, missingTarball.length, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("PackageCount" /* PACKAGE_COUNT */, packageNames.size, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("PackageMajorVersionCount" /* PACKAGE_MAJOR_COUNT */, packageMajorVersions.size, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("PackageVersionCount" /* PACKAGE_VERSION_COUNT */, indexedPackages.size, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("SubmoduleCount" /* SUBMODULE_COUNT */, submodules.length, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("UnknownObjectCount" /* UNKNOWN_OBJECT_COUNT */, unknownObjects.length, aws_embedded_metrics_1.Unit.Count);
        createReport(constants.UNINSTALLABLE_PACKAGES_REPORT, uninstallable);
    })();
    for (const entry of Array.from(perLanguage.entries())) {
        await aws_embedded_metrics_1.metricScope((metrics) => async (language, data) => {
            console.log('');
            console.log('##################################################');
            console.log(`### Start of data for ${language}`);
            metrics.setDimensions({ [constants_1.LANGUAGE_DIMENSION]: language.toString() });
            for (const forStatus of [
                "Supported" /* SUPPORTED */,
                "Unsupported" /* UNSUPPORTED */,
                "Missing" /* MISSING */,
                "CorruptAssembly" /* CORRUPT_ASSEMBLY */,
            ]) {
                for (const [key, statuses] of Object.entries(data)) {
                    let filtered = Array.from(statuses.entries()).filter(([, status]) => forStatus === status);
                    let metricName = METRIC_NAME_BY_STATUS_AND_GRAIN[forStatus][key];
                    if ((forStatus === "Missing" /* MISSING */ && metricName === "MissingPackageVersionCount" /* PER_LANGUAGE_MISSING_VERSIONS */)
                        || (forStatus === "CorruptAssembly" /* CORRUPT_ASSEMBLY */ && metricName === "CorruptAssemblyPackageVersionCount" /* PER_LANGUAGE_CORRUPT_ASSEMBLY_VERSIONS */)) {
                        // generate reports for missing/corrupt only for package versions granularity
                        const reportKey = forStatus === "Missing" /* MISSING */ ?
                            constants.missingDocumentationReport(language) :
                            constants.corruptAssemblyReport(language);
                        createReport(reportKey, filtered.map(([name]) => name).sort());
                    }
                    console.log(`${forStatus} ${key} for ${language}: ${filtered.length} entries`);
                    metrics.putMetric(metricName, filtered.length, aws_embedded_metrics_1.Unit.Count);
                }
            }
            console.log(`### End of data for ${language}`);
            console.log('##################################################');
            console.log('');
        })(...entry);
    }
    for (const report of reports) {
        await report;
    }
}
exports.handler = handler;
async function* relevantObjectKeys(bucket) {
    var _a;
    const request = {
        Bucket: bucket,
        Prefix: constants.STORAGE_KEY_PREFIX,
    };
    do {
        const response = await aws.s3().listObjectsV2(request).promise();
        for (const { Key } of (_a = response.Contents) !== null && _a !== void 0 ? _a : []) {
            if (Key == null) {
                continue;
            }
            yield Key;
        }
        request.ContinuationToken = response.NextContinuationToken;
    } while (request.ContinuationToken != null);
}
/**
 * This function obtains a regular expression with a capture group that allows
 * determining the submodule name from a submodule documentation key, and
 * another to determine whether the object is an "unsupported beacon" or not.
 */
function submoduleKeyRegexp(language) {
    // We use a placeholder to be able to insert the capture group once we have
    // fully quoted the key prefix for Regex safety.
    const placeholder = '<SUBMODULENAME>';
    // We obtain the standard key prefix.
    const keyPrefix = constants.docsKeySuffix(language, placeholder);
    // Finally, assemble the regular expression with the capture group.
    return new RegExp(`.*${reQuote(keyPrefix).replace(placeholder, '(.+)')}(${reQuote(constants.NOT_SUPPORTED_SUFFIX)})?$`);
    /**
     * Escapes all "speacial meaning" characters in a string, so it can be used as
     * part of a regular expression.
     */
    function reQuote(str) {
        return str.replace(/([+*.()?$[\]])/g, '\\$1');
    }
}
const METRIC_NAME_BY_STATUS_AND_GRAIN = {
    ["Missing" /* MISSING */]: {
        ["packages" /* PACKAGES */]: "MissingPackageCount" /* PER_LANGUAGE_MISSING_PACKAGES */,
        ["package major versions" /* PACKAGE_MAJOR_VERSIONS */]: "MissingMajorVersionCount" /* PER_LANGUAGE_MISSING_MAJORS */,
        ["package versions" /* PACKAGE_VERSIONS */]: "MissingPackageVersionCount" /* PER_LANGUAGE_MISSING_VERSIONS */,
        ["package version submodules" /* PACKAGE_VERSION_SUBMODULES */]: "MissingSubmoduleCount" /* PER_LANGUAGE_MISSING_SUBMODULES */,
    },
    ["Unsupported" /* UNSUPPORTED */]: {
        ["packages" /* PACKAGES */]: "UnsupportedPackageCount" /* PER_LANGUAGE_UNSUPPORTED_PACKAGES */,
        ["package major versions" /* PACKAGE_MAJOR_VERSIONS */]: "UnsupportedMajorVersionCount" /* PER_LANGUAGE_UNSUPPORTED_MAJORS */,
        ["package versions" /* PACKAGE_VERSIONS */]: "UnsupportedPackageVersionCount" /* PER_LANGUAGE_UNSUPPORTED_VERSIONS */,
        ["package version submodules" /* PACKAGE_VERSION_SUBMODULES */]: "UnsupportedSubmoduleCount" /* PER_LANGUAGE_UNSUPPORTED_SUBMODULES */,
    },
    ["Supported" /* SUPPORTED */]: {
        ["packages" /* PACKAGES */]: "SupportedPackageCount" /* PER_LANGUAGE_SUPPORTED_PACKAGES */,
        ["package major versions" /* PACKAGE_MAJOR_VERSIONS */]: "SupportedMajorVersionCount" /* PER_LANGUAGE_SUPPORTED_MAJORS */,
        ["package versions" /* PACKAGE_VERSIONS */]: "SupportedPackageVersionCount" /* PER_LANGUAGE_SUPPORTED_VERSIONS */,
        ["package version submodules" /* PACKAGE_VERSION_SUBMODULES */]: "SupportedSubmoduleCount" /* PER_LANGUAGE_SUPPORTED_SUBMODULES */,
    },
    ["CorruptAssembly" /* CORRUPT_ASSEMBLY */]: {
        ["packages" /* PACKAGES */]: "CorruptAssemblyPackageCount" /* PER_LANGUAGE_CORRUPT_ASSEMBLY_PACKAGES */,
        ["package major versions" /* PACKAGE_MAJOR_VERSIONS */]: "CorruptAssemblyMajorVersionCount" /* PER_LANGUAGE_CORRUPT_ASSEMBLY_MAJORS */,
        ["package versions" /* PACKAGE_VERSIONS */]: "CorruptAssemblyPackageVersionCount" /* PER_LANGUAGE_CORRUPT_ASSEMBLY_VERSIONS */,
        ["package version submodules" /* PACKAGE_VERSION_SUBMODULES */]: "CorruptAssemblySubmoduleCount" /* PER_LANGUAGE_CORRUPT_ASSEMBLY_SUBMODULES */,
    },
};
/**
 * Registers the information for the provided language. A "MISSING" status
 * will be ignored if another status was already registered for the same
 * entity. An "UNSUPPORTED" status will be ignored if a "SUPPORTED" status
 * was already registered for the same entity.
 *
 * If a submodule is provided, only that submodule's availability is updated.
 */
function doRecordPerLanguage(perLanguage, language, status, pkgName, pkgMajor, pkgVersion, submodule) {
    if (!perLanguage.has(language)) {
        perLanguage.set(language, {
            ["package major versions" /* PACKAGE_MAJOR_VERSIONS */]: new Map(),
            ["packages" /* PACKAGES */]: new Map(),
            ["package version submodules" /* PACKAGE_VERSION_SUBMODULES */]: new Map(),
            ["package versions" /* PACKAGE_VERSIONS */]: new Map(),
        });
    }
    const data = perLanguage.get(language);
    // If there is a submodule, only update the submodule domain.
    const outputDomains = submodule
        ? [
            [data["package version submodules" /* PACKAGE_VERSION_SUBMODULES */], `${pkgVersion}.${submodule}`],
        ]
        : [
            [data["package major versions" /* PACKAGE_MAJOR_VERSIONS */], pkgMajor],
            [data["package versions" /* PACKAGE_VERSIONS */], pkgVersion],
            [data["packages" /* PACKAGES */], pkgName],
        ];
    for (const [map, name] of outputDomains) {
        switch (status) {
            case "Missing" /* MISSING */:
                // If we already have a status, don't override it with "MISSING".
                if (!map.has(name)) {
                    map.set(name, status);
                }
                break;
            case "Supported" /* SUPPORTED */:
                // If thr package is "supported", this always "wins"
                map.set(name, status);
                break;
            case "Unsupported" /* UNSUPPORTED */:
            case "CorruptAssembly" /* CORRUPT_ASSEMBLY */:
                // If we already have a status, only override with if it was "MISSING".
                if (!map.has(name) || map.get(name) === "Missing" /* MISSING */) {
                    map.set(name, status);
                }
                break;
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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