import { IConstruct } from "constructs";
/**
 * Functions for resolving polycons (polymorphic constructs) into
 * specific constructs.
 */
export declare class Polycons {
    /**
     * Adds a factory at given scope. This factory will be used for resolving
     * polycons under this scope into constructs.
     */
    static register(scope: IConstruct, factory: IPolyconFactory): void;
    /**
     * Creates a new instance of a polycon. The polycon is resolved using the
     * polycon factory that is registered nearest to it in the tree.
     *
     * For example, if a construct tree has Root -> Parent -> MyPoly, and FactoryA
     * is registered to Root while FactoryB is registered to Parent, then
     * FactoryB will be used to resolve MyPoly.
     *
     * @param type The type identifier
     * @param scope The construct scope
     * @param id The construct identifier
     * @param args The rest of the construct's arguments
     * @returns The resolved construct
     */
    static newInstance(type: string, scope: IConstruct, id: string, ...args: any[]): IConstruct;
    private constructor();
}
/**
 * A factory that determines how to turn polycons into concrete constructs.
 */
export interface IPolyconFactory {
    /**
     * Resolve the parameters needed for creating a specific polycon into a
     * concrete construct.
     *
     * @param type The type identifier
     * @param scope The construct scope
     * @param id The construct identifier
     * @param args The rest of the construct's arguments
     * @returns The resolved construct
     */
    resolve(type: string, scope: IConstruct, id: string, ...args: any[]): IConstruct;
}
