# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['gdbundle']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'gdbundle',
    'version': '0.0.4',
    'description': 'Plugin Manager for GDB and LLDB',
    'long_description': '# GDBundle - Plugin Manager for GDB and LLDB\n\n[![PyPI version](https://badge.fury.io/py/gdbundle.svg)](https://badge.fury.io/py/gdbundle)\n\ngdbundle is short for GDB bundle and is a plugin manager for GDB and LLDB.\n\nFor detailed info about the origin story of gdbundle, read the announcement on Interrupt: \n[gdbundle - GDB\'s Missing Plugin Manager](https://interrupt.memfault.com/blog/gdbundle-plugin-manager)\n\n## Benefits\n\nThere are a handful of indisputable benefits of using gdbundle over manually modifying `.gdbinit` and `.lldbinit` files.\n\n1. Just `pip install gdbundle-<plugin-name>`. No more manually editing your\n   `~/.gdbinit` in specific ways depending on the extension.\n2. It enables developers to use virtual environments (and encourages it!).\n3. Personal projects and team projects can have project-specific\n   `requirements.txt` and `.gdbinit` files. With these two in place, a new\n   developer would just need to `pip install -r requirements.txt`.\n4. Discoverability. Check out this\n   [PyPi filter](https://pypi.org/search/?q=gdbundle) to find new plugins.\n5. Dependency management and version tracking is now done automatically by\n   Python\'s packaging infrastructure.\n\n## Published Plugins\n\nPublished ones can be found on [PyPi](https://pypi.org/search/?q=gdbundle)\n\nFor now, a list is kept here as well:\n\n- [example](https://github.com/memfault/gdbundle-example)\n- [gdb-dashboard](https://github.com/memfault/gdbundle-gdb-dashboard)\n- [PyCortexMDebug](https://github.com/memfault/gdbundle-PyCortexMDebug)\n- [voltron](https://github.com/memfault/gdbundle-voltron)\n\n## Quickstart\n\n> NOTE: The Python compiled with GDB should be the same major version as \n> the Python in the local virtual, Conda, or local environment that \n> you are installing gdbundle and plugins into. \n\nTo check both versions of Python, you can run the following:\n\n```\n$ gdb\n(gdb) pi\n>>> import sys; sys.version\n\'3.6.9 (default, Nov  7 2019, 10:44:02) \\n[GCC 8.3.0]\'\n\n$ python --version\nPython 3.6.9\n```\n\n### Install gdbundle\n\nInstall `gdbundle` from PyPi first. Using a\n[virtual environment](https://docs.python.org/3/library/venv.html) is\nrecommended.\n\n```\n$ pip install gdbundle\n```\n\nIf you\'d rather not use a virtual environment, it\'s advised to use `--user` when\ninstalling the package.\n\n```\n$ pip install --user gdbundle\n```\n\nJust, whatever you do, don\'t use `sudo ...`. And don\'t let your friends either.\n\n### Amend Init Scripts\n\nPlacing the following in the appropriate file will load all installed gdbundle\nplugins by default. If you would like to selectively load certain ones, please\nrefer to the [Configuration](#Configuration) section.\n\n#### GDB\'s `.gdbinit`\n\nAppend the following to your `~/.gdbinit` or to a project-specific `.gdbinit`\nthat is loaded with `gdb --command .gdbinit`\n\n```\n# -- gdbundle_BEGIN\npython\nimport os,sys,subprocess\n# Execute a Python using the user\'s shell and pull out the sys.path (for site-packages)\npaths = subprocess.check_output(\'python -c "import os,sys;print(os.linesep.join(sys.path).strip())"\',shell=True).decode("utf-8").split()\n# Extend GDB\'s Python search path\nsys.path.extend(paths)\n\n# Initialize gdbundle\nimport gdbundle\ngdbundle.init()\nend\n# -- gdbundle_END\n```\n\n#### LLDB\'s `.lldbinit`\n\nCopy the `sample_lldbinit.py` somewhere. For now, let\'s assume it\'s at\n`/path/to/gdbundle_lldbinit.py`\n\n```\n# /path/to/gdbundle_lldbinit.py\n\n# -- gdbundle_BEGIN\nimport os,subprocess,sys\n# Execute a Python using the user\'s shell and pull out the sys.path (for site-packages)\npaths = subprocess.check_output(\'python -c "import os,sys;print(os.linesep.join(sys.path).strip())"\',shell=True).decode("utf-8").split()\n# Extend LLDB\'s Python search path\nsys.path.extend(paths)\n\n# Initialize gdbundle\nimport gdbundle\ngdbundle.init()\n# -- gdbundle_END\n```\n\nAppend the following to your `~/.lldbinit` or to project-specific `.lldbinit`\nfile that is loaded with `lldb -s .lldbinit`.\n\n```\n# .lldbinit\n\ncommand script import \'/path/to/gdbundle_lldbinit.py\'\n```\n\n## Background\n\nGDB has\n[built-in support for extensions](https://sourceware.org/gdb/current/onlinedocs/gdb/Extending-GDB.html#Extending-GDB)\nwritten in in Python, Guile, or GDB\'s command language. However, there is no\nconvenient way to package, distribute, and install these scripts.\n\nGDB does provide a few mechanisms:\n\n- Place scripts in /usr/local/share/gdb/auto-load or similar\n- Use `gdb --command ...` to load the scripts on invocation\n-\n\nGDB\'s documentation gives us the following:\n\n> Python scripts used by GDB should be installed in data-directory/python, where\n> data-directory is the data directory as determined at GDB startup\n\nThis leaves a lot to be desired, because a user would have to copy-paste script\nfiles to this directory. There are many drawbacks to this method, including:\n\n- Requires manual intervention to install scripts to `data-directory`\n- Scripts become out of date since they were copy-pasted\n- `data-directory` is usually in a directory next to GDB\'s installation path,\n  usually in `/usr/local`. Everyone should do their best not to manually edit\n  files there.\n\nThere _has_ to be a better way, and thankfully, there is! It\'s called\n`gdbundle`.\n\n## Installing Packages\n\nPackages are prefixed with `gdbundle-` and are installing through `pip` from\nPyPi.\n\n```\n$ pip install gdbundle-example\n```\n\n## Configuration\n\n`gdbundle` today is simple and has only a few configuration knobs. Configuration\nis passed into the `gdbundle.init(...)` function call.\n\n> NOTE: The names passed into `include` and `exclude` should be the package name\n> minus `gdbundle_`, and hyphens are underscores. (e.g. `gdbundle-gdb-dashboard`\n> becomes `gdb_dashboard`)\n\n```py\n# .gdbinit\n\nimport gdbundle\n\n# Configure which packages to include\ninclude = [\n    # List of packages to load.\n    # e.g. "example"\n]\n\nexclude = [\n    # List of packages to exclude.\n    # Useful if the same virtual environment is used for multiple\n    #  projects and not all packages should be loaded\n    # e.g. "example"\n]\n# Load the configured packages\ngdbundle.init(include=include, exclude=exclude)\n```\n\nIf you\'ve created a Python package that has the necessary gdbundle hooks (e.g.\n`gdbundle.gdb_loader.gdbundle_load()`), you can manually load it using\n`gdbundle.load_module()`.\n\n```\n# gdbundle will import this module and attempt to run `my_package.gdb_loader.gdbundle_load()`\ngdbundle.load_module("my_package")\n```\n\n## Creating and Distributing Packages\n\nThe goal was to keep `gdbundle` packages as simple as possible. A package only\nhas a few responsibilities:\n\n1. Define dependencies in the `pyproject.toml` so they are automatically\n   installed.\n2. Provide a version so users can easily pin and upgrade to specific versions.\n3. Provide a hook `gdb_loader.gdbundle_load()` and/or\n   `lldb_loader.gdbundle_load()` which is called by gdbundle, which should\n   either:\n   - Load the script into the debugger context by calling\n     `gdb.execute("source <file>")` or\n     `lldb.debugger.HandleCommand("command script import <file>"`\n   - Import the Python module which does the sourcing itself:\n     `from mypackage import HelloWorld; HelloWorld()`\n\nTo investigate a real package that works with both GDB and LLDB, check out the\n[gdbundle-example plugin](https://github.com/memfault/gdbundle-example).\n\nLet\'s go over each piece quickly.\n\n### `pyproject.toml`\n\nA standard `setup.py` file. The main thing to note here is that our package\n`name` will be `gdbundle-<something>`, but our Python package will be called\n`gdbundle_<something>`.\n\n```python\n[tool.poetry]\nname = "gdbundle-example"\nversion = "0.0.1"\ndescription = ""\nauthors = ["Tyler Hoffman <tyler@memfault.com>"]\nreadme = "README.md"\ninclude = ["gdbundle_example/scripts/*"]\nlicense = "MIT"\nclassifiers = [\n    [...]\n]\n\n[tool.poetry.dependencies]\npython = "*"\n\n[build-system]\nrequires = ["poetry>=0.12"]\nbuild-backend = "poetry.masonry.api"\n\n```\n\n### `gdbundle_example/gdb_loader.py`\n\n```python\nimport gdb\nimport os\n\nPACKAGE_DIR = os.path.dirname(__file__)\n\nSCRIPT_PATHS = [\n    [PACKAGE_DIR, \'scripts\', \'example_gdb.gdb\'],\n    [PACKAGE_DIR, \'scripts\', \'example_gdb.py\']\n]\n\ndef _abs_path(path):\n    return os.path.abspath(os.path.join(*path))\n\ndef gdbundle_load():\n    for script_path in SCRIPT_PATHS:\n        gdb.execute("source {}".format(_abs_path(script_path)))\n```\n\n### `gdbundle_example/scripts/`\n\nPlace the script files here!\n\n## How It Works\n\nIf the GDB/LLDB executable you are using was downloaded rather than compiled\nfrom source, it\'s likely that it is linked against a system Python library and\n`site-packages` directory, rather than against a virtual environment, Conda\nenvironment, or another user installed version of Python.\n\nFor example, if we start `arm-none-eabi-gdb-py` downloaded from\n[ARM](https://developer.arm.com/tools-and-software/open-source-software/developer-tools/gnu-toolchain),\nwe\'ll see that it is searching within the system Python\'s `site-packages` folder\nfor packages:\n\n```\n$ arm-none-eabi-gdb-py\n(gdb) python-interactive\n>>> import sysconfig\n>>> sysconfig.get_paths()[\'purelib\']\n\'/System/Library/Frameworks/Python.framework/Versions/2.7/lib/python2.7/site-packages\'\n```\n\nSimilar with LLDB:\n\n```\n$ lldb\n(lldb) script\nPython Interactive Interpreter. To exit, type \'quit()\', \'exit()\' or Ctrl-D.\n>>> import sysconfig; sysconfig.get_paths()[\'purelib\']\n\'/Applications/Xcode.app/Contents/Developer/Library/Frameworks/Python3.framework/Versions/3.7/lib/python3.7/site-packages\'\n\n```\n\nWhen a user installs packages with an `apt` or `brew` Python, or one from a\nConda or virtual environment, GDB will **not** automatically find those packages\nsince Python\'s configured `sys.path` is only looking in the system\'s\n`site-packages` folder.\n\nThere are three recommended (but _very bad_) ways to handle this:\n\n- Override `PYTHONPATH` on init or in the users shell.\n- Create sym-links between the virtual environment and the system python\n  installation.\n- Deal with it, as said in LLDB\'s\n  [Python Caveat](https://lldb.llvm.org/resources/caveats.html) docs. Install\n  the packages in the same installation anyways.\n\nWe want something that edits _only_ GDB\'s Python context, and only appends to\nPython\'s `sys.path`, not overrides it.\n\nThe best solution I\'ve found and have been using (along with 100 other engineers\nat my previous employer) is to shell out from within GDB, ask the local shell\nenvironment what the configured Python executable is, get its `sys.path`\nentries, and then append those to our current GDB session\'s Python environment.\nThis allows GDB to find and use packages that are installed into a Conda or\nvirtualenv environment, as well as any other user installed Python environment.\n\n## Who uses GDB Python Scripts\n\nMany companies and projects use and include GDB scripts with their large\nopen-source projects. They are usually buried in the repo, difficult to source,\nand built and used in various ways. They usually require the developer to\ndiscover that they exist, manually source them, and then look at the source code\nto figure out how they work.\n\nCheck out the gdbundle announcement for a list of ones that are public and\nopen-source.\n\n[Neat GDB Script Repositories](https://interrupt.memfault.com/blog/gdbundle-plugin-manager#neat-gdb-script-repositories)\n',
    'author': 'Tyler Hoffman',
    'author_email': 'tyler@memfault.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/memfault/gdbundle',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=2.7, !=3.0.*, !=3.1.*, !=3.2.*, !=3.3.*',
}


setup(**setup_kwargs)
