"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.directorySnapshot = exports.synthSnapshot = void 0;
const os = require("os");
const path = require("path");
const fs = require("fs-extra");
const glob_1 = require("glob");
/**
 * Creates a snapshot of the files generated by a project. Ignores any non-text
 * files so that the snapshots are human readable.
 */
function synthSnapshot(project, options = {}) {
    // defensive: verify that "outdir" is actually in a temporary directory
    if (!path.resolve(project.outdir).startsWith(os.tmpdir()) &&
        !project.outdir.includes("project-temp-dir")) {
        throw new Error("Trying to capture a snapshot of a project outside of tmpdir, which implies this test might corrupt an existing project");
    }
    const synthed = Symbol.for("synthed");
    if (synthed in project) {
        throw new Error("duplicate synth()");
    }
    project[synthed] = true;
    const ENV_PROJEN_DISABLE_POST = process.env.PROJEN_DISABLE_POST;
    try {
        process.env.PROJEN_DISABLE_POST = "true";
        project.synth();
        const ignoreExts = ["png", "ico"];
        return directorySnapshot(project.outdir, {
            ...options,
            excludeGlobs: ignoreExts.map((ext) => `**/*.${ext}`),
        });
    }
    finally {
        fs.removeSync(project.outdir);
        // values assigned to process.env.XYZ are automatically converted to strings
        if (ENV_PROJEN_DISABLE_POST === undefined) {
            delete process.env.PROJEN_DISABLE_POST;
        }
        else {
            process.env.PROJEN_DISABLE_POST = ENV_PROJEN_DISABLE_POST;
        }
    }
}
exports.synthSnapshot = synthSnapshot;
function directorySnapshot(root, options = {}) {
    const output = {};
    const files = glob_1.glob.sync("**", {
        ignore: [".git/**", ...(options.excludeGlobs ?? [])],
        cwd: root,
        nodir: true,
        dot: true,
    }); // returns relative file paths with POSIX separators
    const parseJson = options.parseJson ?? true;
    for (const file of files) {
        const filePath = path.join(root, file);
        let content;
        if (!options.onlyFileNames) {
            if (parseJson && path.extname(filePath) === ".json") {
                content = fs.readJsonSync(filePath);
            }
            else {
                content = fs.readFileSync(filePath, "utf-8");
            }
        }
        else {
            content = true;
        }
        output[file] = content;
    }
    return output;
}
exports.directorySnapshot = directorySnapshot;
//# sourceMappingURL=data:application/json;base64,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