"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Task = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * A task that can be performed on the project. Modeled as a series of shell
 * commands and subtasks.
 */
class Task {
    constructor(name, props = {}) {
        this.name = name;
        this._description = props.description;
        this.condition = props.condition;
        this.cwd = props.cwd;
        this._locked = false;
        this._env = props.env ?? {};
        this._steps = props.steps ?? [];
        this.requiredEnv = props.requiredEnv;
        if (props.exec && props.steps) {
            throw new Error("cannot specify both exec and steps");
        }
        if (props.exec) {
            this.exec(props.exec);
        }
    }
    /**
     * Forbid additional changes to this task.
     */
    lock() {
        this._locked = true;
    }
    /**
     * Returns the description of this task.
     */
    get description() {
        return this._description;
    }
    /**
     * Sets the description of this task.
     */
    set description(desc) {
        this._description = desc;
    }
    /**
     * Reset the task so it no longer has any commands.
     * @param command the first command to add to the task after it was cleared.
     */
    reset(command, options = {}) {
        this.assertUnlocked();
        while (this._steps.length) {
            this._steps.shift();
        }
        if (command) {
            this.exec(command, options);
        }
    }
    /**
     * Executes a shell command
     * @param command Shell command
     * @param options Options
     */
    exec(command, options = {}) {
        this.assertUnlocked();
        this._steps.push({ exec: command, ...options });
    }
    /**
     * Execute a builtin task.
     *
     * Builtin tasks are programs bundled as part of projen itself and used as
     * helpers for various components.
     *
     * In the future we should support built-in tasks from external modules.
     *
     * @param name The name of the builtin task to execute (e.g.
     * `release/resolve-version`).
     */
    builtin(name) {
        this.assertUnlocked();
        this._steps.push({ builtin: name });
    }
    /**
     * Say something.
     * @param message Your message
     * @param options Options
     */
    say(message, options = {}) {
        this.assertUnlocked();
        this._steps.push({ say: message, ...options });
    }
    /**
     * Adds a command at the beginning of the task.
     * @param shell The command to add.
     *
     * @deprecated use `prependExec()`
     */
    prepend(shell, options = {}) {
        this.assertUnlocked();
        this.prependExec(shell, options);
    }
    /**
     * Spawns a sub-task.
     * @param subtask The subtask to execute.
     */
    spawn(subtask, options = {}) {
        this.assertUnlocked();
        this._steps.push({ spawn: subtask.name, ...options });
    }
    /**
     * Adds a command at the beginning of the task.
     * @param shell The command to add.
     */
    prependExec(shell, options = {}) {
        this.assertUnlocked();
        this._steps.unshift({
            exec: shell,
            ...options,
        });
    }
    /**
     * Adds a spawn instruction at the beginning of the task.
     * @param subtask The subtask to execute.
     */
    prependSpawn(subtask, options = {}) {
        this.assertUnlocked();
        this._steps.unshift({
            spawn: subtask.name,
            ...options,
        });
    }
    /**
     * Says something at the beginning of the task.
     * @param message Your message
     */
    prependSay(message, options = {}) {
        this.assertUnlocked();
        this._steps.unshift({
            say: message,
            ...options,
        });
    }
    /**
     * Adds an environment variable to this task.
     * @param name The name of the variable
     * @param value The value. If the value is surrounded by `$()`, we will
     * evaluate it within a subshell and use the result as the value of the
     * environment variable.
     */
    env(name, value) {
        this.assertUnlocked();
        this._env[name] = value;
    }
    /**
     * Returns an immutable copy of all the step specifications of the task.
     */
    get steps() {
        // If the list of steps is a Lazy value, we can't know what the steps
        // are until synthesis occurs, so just return an empty array.
        if (!Array.isArray(this._steps)) {
            return [];
        }
        return [...this._steps];
    }
    /**
     * Renders a task spec into the manifest.
     *
     * @internal
     */
    _renderSpec() {
        return {
            name: this.name,
            description: this.description,
            env: this._env,
            requiredEnv: this.requiredEnv,
            steps: this._steps,
            condition: this.condition,
            cwd: this.cwd,
        };
    }
    assertUnlocked() {
        if (this._locked) {
            throw new Error(`Task "${this.name}" is locked for changes`);
        }
    }
}
exports.Task = Task;
_a = JSII_RTTI_SYMBOL_1;
Task[_a] = { fqn: "projen.Task", version: "0.63.29" };
//# sourceMappingURL=data:application/json;base64,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