"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TaskRuntime = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const child_process_1 = require("child_process");
const fs_1 = require("fs");
const os_1 = require("os");
const path_1 = require("path");
const path = require("path");
const util_1 = require("util");
const chalk = require("chalk");
const common_1 = require("./common");
const logging = require("./logging");
const ENV_TRIM_LEN = 20;
/**
 * The runtime component of the tasks engine.
 */
class TaskRuntime {
    constructor(workdir) {
        this.workdir = path_1.resolve(workdir);
        const manifestPath = path_1.join(this.workdir, TaskRuntime.MANIFEST_FILE);
        this.manifest = fs_1.existsSync(manifestPath)
            ? JSON.parse(fs_1.readFileSync(manifestPath, "utf-8"))
            : { tasks: {} };
    }
    /**
     * The tasks in this project.
     */
    get tasks() {
        return Object.values(this.manifest.tasks ?? {});
    }
    /**
     * Find a task by name, or `undefined` if not found.
     */
    tryFindTask(name) {
        if (!this.manifest.tasks) {
            return undefined;
        }
        return this.manifest.tasks[name];
    }
    /**
     * Runs the task.
     * @param name The task name.
     */
    runTask(name, parents = []) {
        const task = this.tryFindTask(name);
        if (!task) {
            throw new Error(`cannot find command ${task}`);
        }
        new RunTask(this, task, parents);
    }
}
exports.TaskRuntime = TaskRuntime;
_a = JSII_RTTI_SYMBOL_1;
TaskRuntime[_a] = { fqn: "projen.TaskRuntime", version: "0.63.29" };
/**
 * The project-relative path of the tasks manifest file.
 */
TaskRuntime.MANIFEST_FILE = path.posix.join(common_1.PROJEN_DIR, "tasks.json");
class RunTask {
    constructor(runtime, task, parents = []) {
        this.runtime = runtime;
        this.task = task;
        this.env = {};
        this.workdir = task.cwd ?? this.runtime.workdir;
        this.parents = parents;
        if (!task.steps || task.steps.length === 0) {
            this.logDebug(chalk.gray("No actions have been specified for this task."));
            return;
        }
        this.env = this.resolveEnvironment(parents);
        const envlogs = [];
        for (const [k, v] of Object.entries(this.env)) {
            const vv = v ?? "";
            const trimmed = vv.length > ENV_TRIM_LEN ? vv.substr(0, ENV_TRIM_LEN) + "..." : vv;
            envlogs.push(`${k}=${trimmed}`);
        }
        if (envlogs.length) {
            this.logDebug(chalk.gray(`${chalk.underline("env")}: ${envlogs.join(" ")}`));
        }
        // evaluate condition
        if (!this.evalCondition(task)) {
            this.log("condition exited with non-zero - skipping");
            return;
        }
        // verify we required environment variables are defined
        const merged = { ...process.env, ...this.env };
        const missing = new Array();
        for (const name of task.requiredEnv ?? []) {
            if (!(name in merged)) {
                missing.push(name);
            }
        }
        if (missing.length > 0) {
            throw new Error(`missing required environment variables: ${missing.join(",")}`);
        }
        for (const step of task.steps) {
            if (step.say) {
                logging.info(this.fmtLog(step.say));
            }
            if (step.spawn) {
                this.runtime.runTask(step.spawn, [...this.parents, this.task.name]);
            }
            const execs = step.exec ? [step.exec] : [];
            if (step.builtin) {
                execs.push(this.renderBuiltin(step.builtin));
            }
            for (const exec of execs) {
                let command = "";
                let hasError = false;
                const cmd = exec.split(" ")[0];
                if (os_1.platform() == "win32" &&
                    ["mkdir", "mv", "rm", "cp"].includes(cmd)) {
                    command = `shx ${exec}`;
                }
                else {
                    command = exec;
                }
                const cwd = step.cwd;
                try {
                    const result = this.shell({
                        command,
                        cwd,
                    });
                    hasError = result.status !== 0;
                }
                catch (e) {
                    // This is the error 'shx' will throw
                    if (e?.message?.startsWith("non-zero exit code:")) {
                        hasError = true;
                    }
                    throw e;
                }
                if (hasError) {
                    throw new Error(`Task "${this.fullname}" failed when executing "${command}" (cwd: ${path_1.resolve(cwd ?? this.workdir)})`);
                }
            }
        }
    }
    /**
     * Determines if a task should be executed based on "condition".
     *
     * @returns true if the task should be executed or false if the condition
     * evaluates to false (exits with non-zero), indicating that the task should
     * be skipped.
     */
    evalCondition(task) {
        // no condition, carry on
        if (!task.condition) {
            return true;
        }
        this.log(chalk.gray(`${chalk.underline("condition")}: ${task.condition}`));
        const result = this.shell({
            command: task.condition,
            logprefix: "condition: ",
            quiet: true,
        });
        if (result.status === 0) {
            return true;
        }
        else {
            return false;
        }
    }
    /**
     * Renders the runtime environment for a task. Namely, it supports this syntax
     * `$(xx)` for allowing environment to be evaluated by executing a shell
     * command and obtaining its result.
     */
    resolveEnvironment(parents) {
        let env = this.runtime.manifest.env ?? {};
        // add env from all parent tasks one by one
        for (const parent of parents) {
            env = {
                ...env,
                ...(this.runtime.tryFindTask(parent)?.env ?? {}),
            };
        }
        // apply the task's environment last
        env = {
            ...env,
            ...(this.task.env ?? {}),
        };
        const output = {};
        for (const [key, value] of Object.entries(env ?? {})) {
            if (value.startsWith("$(") && value.endsWith(")")) {
                const query = value.substring(2, value.length - 1);
                const result = this.shellEval({ command: query });
                if (result.status !== 0) {
                    const error = result.error
                        ? result.error.stack
                        : result.stderr?.toString() ?? "unknown error";
                    throw new Error(`unable to evaluate environment variable ${key}=${value}: ${error}`);
                }
                output[key] = result.stdout.toString("utf-8").trim();
            }
            else {
                output[key] = value;
            }
        }
        return output;
    }
    /**
     * Returns the "full name" of the task which includes all it's parent task names concatenated by chevrons.
     */
    get fullname() {
        return [...this.parents, this.task.name].join(" » ");
    }
    log(...args) {
        logging.verbose(this.fmtLog(...args));
    }
    logDebug(...args) {
        logging.debug(this.fmtLog(...args));
    }
    fmtLog(...args) {
        return util_1.format(`${chalk.underline(this.fullname)} |`, ...args);
    }
    shell(options) {
        const quiet = options.quiet ?? false;
        if (!quiet) {
            const log = new Array();
            if (options.logprefix) {
                log.push(options.logprefix);
            }
            log.push(options.command);
            if (options.cwd) {
                log.push(`(cwd: ${options.cwd})`);
            }
            this.log(log.join(" "));
        }
        const cwd = options.cwd ?? this.workdir;
        if (!fs_1.existsSync(cwd) || !fs_1.statSync(cwd).isDirectory()) {
            throw new Error(`invalid workdir (cwd): ${cwd} must be an existing directory`);
        }
        return child_process_1.spawnSync(options.command, {
            ...options,
            cwd,
            shell: true,
            stdio: "inherit",
            env: {
                ...process.env,
                ...this.env,
            },
            ...options.spawnOptions,
        });
    }
    shellEval(options) {
        return this.shell({
            quiet: true,
            ...options,
            spawnOptions: {
                stdio: ["inherit", "pipe", "inherit"],
            },
        });
    }
    renderBuiltin(builtin) {
        const moduleRoot = path_1.dirname(require.resolve("../package.json"));
        const program = require.resolve(path_1.join(moduleRoot, "lib", `${builtin}.task.js`));
        return `${process.execPath} ${program}`;
    }
}
//# sourceMappingURL=data:application/json;base64,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