"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Release = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const component_1 = require("../component");
const github_1 = require("../github");
const constants_1 = require("../github/constants");
const workflows_model_1 = require("../github/workflows-model");
const version_1 = require("../version");
const publisher_1 = require("./publisher");
const release_trigger_1 = require("./release-trigger");
const BUILD_JOBID = "release";
const GIT_REMOTE_STEPID = "git_remote";
const LATEST_COMMIT_OUTPUT = "latest_commit";
/**
 * Manages releases (currently through GitHub workflows).
 *
 * By default, no branches are released. To add branches, call `addBranch()`.
 */
class Release extends component_1.Component {
    constructor(project, options) {
        super(project);
        this._branches = new Array();
        this.jobs = {};
        if (Array.isArray(options.releaseBranches)) {
            throw new Error('"releaseBranches" is no longer an array. See type annotations');
        }
        this.github = project.github;
        this.buildTask = options.task;
        this.preBuildSteps = options.releaseWorkflowSetupSteps ?? [];
        this.postBuildSteps = options.postBuildSteps ?? [];
        this.artifactsDirectory = options.artifactsDirectory ?? "dist";
        this.versionFile = options.versionFile;
        this.releaseTrigger = options.releaseTrigger ?? release_trigger_1.ReleaseTrigger.continuous();
        this.containerImage = options.workflowContainerImage;
        this.workflowRunsOn = options.workflowRunsOn;
        this._branchHooks = [];
        /**
         * Use manual releases with no changelog if releaseEveryCommit is explicitly
         * disabled and no other trigger is set.
         *
         * TODO: Remove this when releaseEveryCommit and releaseSchedule are removed
         */
        if (!((options.releaseEveryCommit ?? true) ||
            options.releaseSchedule ||
            options.releaseTrigger)) {
            this.releaseTrigger = release_trigger_1.ReleaseTrigger.manual({ changelog: false });
        }
        if (options.releaseSchedule) {
            this.releaseTrigger = release_trigger_1.ReleaseTrigger.scheduled({
                schedule: options.releaseSchedule,
            });
        }
        this.version = new version_1.Version(project, {
            versionInputFile: this.versionFile,
            artifactsDirectory: this.artifactsDirectory,
            versionrcOptions: options.versionrcOptions,
            tagPrefix: options.releaseTagPrefix,
        });
        this.publisher = new publisher_1.Publisher(project, {
            artifactName: this.artifactsDirectory,
            condition: `needs.${BUILD_JOBID}.outputs.${LATEST_COMMIT_OUTPUT} == github.sha`,
            buildJobId: BUILD_JOBID,
            jsiiReleaseVersion: options.jsiiReleaseVersion,
            failureIssue: options.releaseFailureIssue,
            failureIssueLabel: options.releaseFailureIssueLabel,
            workflowRunsOn: options.workflowRunsOn,
            publishTasks: options.publishTasks,
            dryRun: options.publishDryRun,
            workflowNodeVersion: options.workflowNodeVersion,
        });
        const githubRelease = options.githubRelease ?? true;
        if (githubRelease) {
            this.publisher.publishToGitHubReleases({
                changelogFile: path.posix.join(this.artifactsDirectory, this.version.changelogFileName),
                versionFile: path.posix.join(this.artifactsDirectory, this.version.versionFileName),
                releaseTagFile: path.posix.join(this.artifactsDirectory, this.version.releaseTagFileName),
            });
        }
        // add the default branch (we need the internal method which does not require majorVersion)
        this.defaultBranch = this._addBranch(options.branch, {
            prerelease: options.prerelease,
            majorVersion: options.majorVersion,
            minMajorVersion: options.minMajorVersion,
            workflowName: options.releaseWorkflowName ?? "release",
            tagPrefix: options.releaseTagPrefix,
            npmDistTag: options.npmDistTag,
        });
        for (const [name, opts] of Object.entries(options.releaseBranches ?? {})) {
            this.addBranch(name, opts);
        }
    }
    /**
     * Returns the `Release` component of a project or `undefined` if the project
     * does not have a Release component.
     */
    static of(project) {
        const isRelease = (c) => c instanceof Release;
        return project.components.find(isRelease);
    }
    /**
     * Add a hook that should be run for every branch (including those that will
     * be added by future `addBranch` calls).
     * @internal
     */
    _forEachBranch(hook) {
        for (const branch of this._branches) {
            hook(branch.name);
        }
        this._branchHooks.push(hook);
    }
    /**
     * Adds a release branch.
     *
     * It is a git branch from which releases are published. If a project has more than one release
     * branch, we require that `majorVersion` is also specified for the primary branch in order to
     * ensure branches always release the correct version.
     *
     * @param branch The branch to monitor (e.g. `main`, `v2.x`)
     * @param options Branch definition
     */
    addBranch(branch, options) {
        this._addBranch(branch, options);
        // run all branch hooks
        for (const hook of this._branchHooks) {
            hook(branch);
        }
    }
    /**
     * Adds a release branch.
     *
     * It is a git branch from which releases are published. If a project has more than one release
     * branch, we require that `majorVersion` is also specified for the primary branch in order to
     * ensure branches always release the correct version.
     *
     * @param branch The branch to monitor (e.g. `main`, `v2.x`)
     * @param options Branch definition
     */
    _addBranch(branch, options) {
        if (this._branches.find((b) => b.name === branch)) {
            throw new Error(`The release branch ${branch} is already defined`);
        }
        // if we add a branch, we require that the default branch will also define a
        // major version.
        if (this.defaultBranch &&
            options.majorVersion &&
            this.defaultBranch.majorVersion === undefined) {
            throw new Error('you must specify "majorVersion" for the default branch when adding multiple release branches');
        }
        const releaseBranch = {
            name: branch,
            ...options,
            workflow: this.createWorkflow(branch, options),
        };
        this._branches.push(releaseBranch);
        return releaseBranch;
    }
    preSynthesize() {
        for (const branch of this._branches) {
            if (!branch.workflow) {
                continue;
            }
            branch.workflow.addJobs(this.publisher._renderJobsForBranch(branch.name, branch));
            branch.workflow.addJobs(this.jobs);
        }
    }
    /**
     * Adds jobs to all release workflows.
     * @param jobs The jobs to add (name => job)
     */
    addJobs(jobs) {
        for (const [name, job] of Object.entries(jobs)) {
            this.jobs[name] = job;
        }
    }
    /**
     * Retrieve all release branch names
     */
    get branches() {
        return this._branches.map((b) => b.name);
    }
    /**
     * @returns a workflow or `undefined` if github integration is disabled.
     */
    createWorkflow(branchName, branch) {
        const workflowName = branch.workflowName ?? `release-${branchName}`;
        // to avoid race conditions between two commits trying to release the same
        // version, we check if the head sha is identical to the remote sha. if
        // not, we will skip the release and just finish the build.
        const noNewCommits = `\${{ steps.${GIT_REMOTE_STEPID}.outputs.${LATEST_COMMIT_OUTPUT} == github.sha }}`;
        // The arrays are being cloned to avoid accumulating values from previous branches
        const preBuildSteps = [...this.preBuildSteps];
        const env = {
            RELEASE: "true",
        };
        if (branch.majorVersion !== undefined) {
            env.MAJOR = branch.majorVersion.toString();
        }
        if (branch.minMajorVersion !== undefined) {
            if (branch.majorVersion !== undefined) {
                throw new Error(`minMajorVersion and majorVersion cannot be used together.`);
            }
            env.MIN_MAJOR = branch.minMajorVersion.toString();
        }
        if (branch.prerelease) {
            env.PRERELEASE = branch.prerelease;
        }
        if (branch.tagPrefix) {
            env.RELEASE_TAG_PREFIX = branch.tagPrefix;
        }
        // the "release" task prepares a release but does not publish anything. the
        // output of the release task is: `dist`, `.version.txt`, and
        // `.changelog.md`. this is what publish tasks expect.
        // if this is the release for "main" or "master", just call it "release".
        // otherwise, "release:BRANCH"
        const releaseTaskName = branchName === "main" || branchName === "master"
            ? "release"
            : `release:${branchName}`;
        const releaseTask = this.project.addTask(releaseTaskName, {
            description: `Prepare a release from "${branchName}" branch`,
            env,
        });
        releaseTask.exec(`rm -fr ${this.artifactsDirectory}`);
        releaseTask.spawn(this.version.bumpTask);
        releaseTask.spawn(this.buildTask);
        releaseTask.spawn(this.version.unbumpTask);
        // anti-tamper check (fails if there were changes to committed files)
        // this will identify any non-committed files generated during build (e.g. test snapshots)
        releaseTask.exec(Release.ANTI_TAMPER_CMD);
        if (this.releaseTrigger.isManual) {
            const publishTask = this.publisher.publishToGit({
                changelogFile: path.posix.join(this.artifactsDirectory, this.version.changelogFileName),
                versionFile: path.posix.join(this.artifactsDirectory, this.version.versionFileName),
                releaseTagFile: path.posix.join(this.artifactsDirectory, this.version.releaseTagFileName),
                projectChangelogFile: this.releaseTrigger.changelogPath,
                gitBranch: branchName,
                gitPushCommand: this.releaseTrigger.gitPushCommand,
            });
            releaseTask.spawn(publishTask);
        }
        const postBuildSteps = [...this.postBuildSteps];
        // check if new commits were pushed to the repo while we were building.
        // if new commits have been pushed, we will cancel this release
        postBuildSteps.push({
            name: "Check for new commits",
            id: GIT_REMOTE_STEPID,
            run: `echo ::set-output name=${LATEST_COMMIT_OUTPUT}::"$(git ls-remote origin -h \${{ github.ref }} | cut -f1)"`,
        });
        postBuildSteps.push({
            name: "Backup artifact permissions",
            if: noNewCommits,
            continueOnError: true,
            run: `cd ${this.artifactsDirectory} && getfacl -R . > ${constants_1.PERMISSION_BACKUP_FILE}`,
        }, {
            name: "Upload artifact",
            if: noNewCommits,
            uses: "actions/upload-artifact@v2.1.1",
            with: {
                name: constants_1.BUILD_ARTIFACT_NAME,
                path: this.artifactsDirectory,
            },
        });
        if (this.github && !this.releaseTrigger.isManual) {
            return new github_1.TaskWorkflow(this.github, {
                name: workflowName,
                jobId: BUILD_JOBID,
                outputs: {
                    latest_commit: {
                        stepId: GIT_REMOTE_STEPID,
                        outputName: LATEST_COMMIT_OUTPUT,
                    },
                },
                triggers: {
                    schedule: this.releaseTrigger.schedule
                        ? [{ cron: this.releaseTrigger.schedule }]
                        : undefined,
                    push: this.releaseTrigger.isContinuous
                        ? { branches: [branchName] }
                        : undefined,
                },
                container: this.containerImage
                    ? { image: this.containerImage }
                    : undefined,
                env: {
                    CI: "true",
                },
                permissions: {
                    contents: workflows_model_1.JobPermission.WRITE,
                },
                checkoutWith: {
                    // we must use 'fetch-depth=0' in order to fetch all tags
                    // otherwise tags are not checked out
                    "fetch-depth": 0,
                },
                preBuildSteps,
                task: releaseTask,
                postBuildSteps,
                runsOn: this.workflowRunsOn,
            });
        }
        else {
            return undefined;
        }
    }
}
exports.Release = Release;
_a = JSII_RTTI_SYMBOL_1;
Release[_a] = { fqn: "projen.release.Release", version: "0.63.29" };
Release.ANTI_TAMPER_CMD = "git diff --ignore-space-at-eol --exit-code";
//# sourceMappingURL=data:application/json;base64,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