"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GithubWorkflow = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const case_1 = require("case");
const _resolve_1 = require("../_resolve");
const component_1 = require("../component");
const util_1 = require("../util");
const yaml_1 = require("../yaml");
/**
 * Workflow for GitHub.
 *
 * A workflow is a configurable automated process made up of one or more jobs.
 *
 * @see https://docs.github.com/en/actions/reference/workflow-syntax-for-github-actions
 */
class GithubWorkflow extends component_1.Component {
    constructor(github, name, options = {}) {
        super(github.project);
        this.events = {};
        this.jobs = {};
        this.name = name;
        this.concurrency = options.concurrency;
        this.projenCredentials = github.projenCredentials;
        const workflowsEnabled = github.workflowsEnabled || options.force;
        if (workflowsEnabled) {
            this.file = new yaml_1.YamlFile(this.project, `.github/workflows/${name.toLocaleLowerCase()}.yml`, {
                obj: () => this.renderWorkflow(),
                // GitHub needs to read the file from the repository in order to work.
                committed: true,
            });
        }
    }
    /**
     * Add events to triggers the workflow.
     *
     * @param events The event(s) to trigger the workflow.
     */
    on(events) {
        this.events = {
            ...this.events,
            ...events,
        };
    }
    /**
     * Adds a single job to the workflow.
     * @param id The job name (unique within the workflow)
     * @param job The job specification
     */
    addJob(id, job) {
        this.addJobs({ [id]: job });
    }
    /**
     * Add jobs to the workflow.
     *
     * @param jobs Jobs to add.
     */
    addJobs(jobs) {
        // verify that job has a "permissions" statement to ensure workflow can
        // operate in repos with default tokens set to readonly
        for (const [id, job] of Object.entries(jobs)) {
            if (!job.permissions) {
                throw new Error(`${id}: all workflow jobs must have a "permissions" clause to ensure workflow can operate in restricted repositories`);
            }
        }
        // verify that job has a "runsOn" statement to ensure a worker can be selected appropriately
        for (const [id, job] of Object.entries(jobs)) {
            if (!("uses" in job)) {
                if ("runsOn" in job && job.runsOn.length === 0) {
                    throw new Error(`${id}: at least one runner selector labels must be provided in "runsOn" to ensure a runner instance can be selected`);
                }
            }
        }
        this.jobs = {
            ...this.jobs,
            ...jobs,
        };
    }
    renderWorkflow() {
        return {
            name: this.name,
            on: snakeCaseKeys(this.events),
            concurrency: this.concurrency,
            jobs: renderJobs(this.jobs),
        };
    }
}
exports.GithubWorkflow = GithubWorkflow;
_a = JSII_RTTI_SYMBOL_1;
GithubWorkflow[_a] = { fqn: "projen.github.GithubWorkflow", version: "0.63.29" };
function snakeCaseKeys(obj) {
    if (typeof obj !== "object" || obj == null) {
        return obj;
    }
    if (Array.isArray(obj)) {
        return obj.map(snakeCaseKeys);
    }
    const result = {};
    for (let [k, v] of Object.entries(obj)) {
        if (typeof v === "object" && v != null) {
            v = snakeCaseKeys(v);
        }
        result[case_1.snake(k)] = v;
    }
    return result;
}
function renderJobs(jobs) {
    const result = {};
    for (const [name, job] of Object.entries(jobs)) {
        result[name] = renderJob(job);
    }
    return result;
    /** @see https://docs.github.com/en/actions/reference/workflow-syntax-for-github-actions */
    function renderJob(job) {
        const steps = new Array();
        // https://docs.github.com/en/actions/using-workflows/reusing-workflows#supported-keywords-for-jobs-that-call-a-reusable-workflow
        if ("uses" in job) {
            return {
                name: job.name,
                needs: arrayOrScalar(job.needs),
                if: job.if,
                permissions: util_1.kebabCaseKeys(job.permissions),
                concurrency: job.concurrency,
                uses: job.uses,
                with: job.with,
                secrets: job.secrets,
            };
        }
        if (job.tools) {
            steps.push(...setupTools(job.tools));
        }
        const userDefinedSteps = util_1.kebabCaseKeys(_resolve_1.resolve(job.steps), false);
        steps.push(...userDefinedSteps);
        return {
            name: job.name,
            needs: arrayOrScalar(job.needs),
            "runs-on": arrayOrScalar(job.runsOn),
            permissions: util_1.kebabCaseKeys(job.permissions),
            environment: job.environment,
            concurrency: job.concurrency,
            outputs: renderJobOutputs(job.outputs),
            env: job.env,
            defaults: util_1.kebabCaseKeys(job.defaults),
            if: job.if,
            steps: steps.map(renderStep),
            "timeout-minutes": job.timeoutMinutes,
            strategy: renderJobStrategy(job.strategy),
            "continue-on-error": job.continueOnError,
            container: job.container,
            services: job.services,
        };
    }
    function renderJobOutputs(output) {
        if (output == null) {
            return undefined;
        }
        const rendered = {};
        for (const [name, { stepId, outputName }] of Object.entries(output)) {
            rendered[name] = `\${{ steps.${stepId}.outputs.${outputName} }}`;
        }
        return rendered;
    }
    function renderJobStrategy(strategy) {
        if (strategy == null) {
            return undefined;
        }
        const rendered = {
            "max-parallel": strategy.maxParallel,
            "fail-fast": strategy.failFast,
        };
        if (strategy.matrix) {
            const matrix = {
                include: strategy.matrix.include,
                exclude: strategy.matrix.exclude,
            };
            for (const [key, values] of Object.entries(strategy.matrix.domain ?? {})) {
                if (key in matrix) {
                    // A domain key was set to `include`, or `exclude`:
                    throw new Error(`Illegal job strategy matrix key: ${key}`);
                }
                matrix[key] = values;
            }
            rendered.matrix = matrix;
        }
        return rendered;
    }
    function renderStep(step) {
        return {
            name: step.name,
            id: step.id,
            if: step.if,
            uses: step.uses,
            env: step.env,
            run: step.run,
            with: step.with,
            "continue-on-error": step.continueOnError,
            "timeout-minutes": step.timeoutMinutes,
        };
    }
}
function arrayOrScalar(arr) {
    if (arr == null || arr.length === 0) {
        return arr;
    }
    if (arr.length === 1) {
        return arr[0];
    }
    return arr;
}
function setupTools(tools) {
    const steps = [];
    if (tools.java) {
        steps.push({
            uses: "actions/setup-java@v3",
            with: { distribution: "temurin", "java-version": tools.java.version },
        });
    }
    if (tools.node) {
        steps.push({
            uses: "actions/setup-node@v3",
            with: { "node-version": tools.node.version },
        });
    }
    if (tools.python) {
        steps.push({
            uses: "actions/setup-python@v3",
            with: { "python-version": tools.python.version },
        });
    }
    if (tools.go) {
        steps.push({
            uses: "actions/setup-go@v3",
            with: { "go-version": tools.go.version },
        });
    }
    if (tools.dotnet) {
        steps.push({
            uses: "actions/setup-dotnet@v2",
            with: { "dotnet-version": tools.dotnet.version },
        });
    }
    return steps;
}
//# sourceMappingURL=data:application/json;base64,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