"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.WorkflowActions = void 0;
const GIT_PATCH_FILE = ".repo.patch";
const RUNNER_TEMP = "${{ runner.temp }}";
/**
 * A set of utility functions for creating GitHub actions in workflows.
 */
class WorkflowActions {
    /**
     * Creates a .patch file from the current git diff and uploads it as an
     * artifact. Use `checkoutWithPatch` to download and apply in another job.
     *
     * If a patch was uploaded,
     *
     * @param options Options
     * @returns Job steps
     */
    static createUploadGitPatch(options) {
        const MUTATIONS_FOUND = `steps.${options.stepId}.outputs.${options.outputName}`;
        const steps = [
            {
                id: options.stepId,
                name: "Find mutations",
                run: [
                    "git add .",
                    `git diff --staged --patch --exit-code > ${GIT_PATCH_FILE} || echo "::set-output name=${options.outputName}::true"`,
                ].join("\n"),
            },
            {
                if: MUTATIONS_FOUND,
                name: "Upload patch",
                uses: "actions/upload-artifact@v2",
                with: { name: GIT_PATCH_FILE, path: GIT_PATCH_FILE },
            },
        ];
        if (options.mutationError) {
            steps.push({
                name: "Fail build on mutation",
                if: MUTATIONS_FOUND,
                run: [
                    `echo "::error::${options.mutationError}"`,
                    `cat ${GIT_PATCH_FILE}`,
                    "exit 1",
                ].join("\n"),
            });
        }
        return steps;
    }
    /**
     * Checks out a repository and applies a git patch that was created using
     * `createUploadGitPatch`.
     *
     * @param options Options
     * @returns Job steps
     */
    static checkoutWithPatch(options = {}) {
        return [
            {
                name: "Checkout",
                uses: "actions/checkout@v3",
                with: {
                    token: options.token,
                    ref: options.ref,
                    repository: options.repository,
                },
            },
            {
                name: "Download patch",
                uses: "actions/download-artifact@v3",
                with: { name: GIT_PATCH_FILE, path: RUNNER_TEMP },
            },
            {
                name: "Apply patch",
                run: `[ -s ${RUNNER_TEMP}/${GIT_PATCH_FILE} ] && git apply ${RUNNER_TEMP}/${GIT_PATCH_FILE} || echo "Empty patch. Skipping."`,
            },
        ];
    }
    /**
     * Configures the git identity (user name and email).
     * @param id The identity to use
     * @returns Job steps
     */
    static setGitIdentity(id) {
        return [
            {
                name: "Set git identity",
                run: [
                    `git config user.name "${id.name}"`,
                    `git config user.email "${id.email}"`,
                ].join("\n"),
            },
        ];
    }
}
exports.WorkflowActions = WorkflowActions;
//# sourceMappingURL=data:application/json;base64,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