"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.JsiiProject = exports.Stability = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const javascript_1 = require("../javascript");
const typescript_1 = require("../typescript");
const util_1 = require("../util");
const consts_1 = require("./consts");
const jsii_docgen_1 = require("./jsii-docgen");
const EMAIL_REGEX = /(?:[a-z0-9!#$%&'*+/=?^_`{|}~-]+(?:\.[a-z0-9!#$%&'*+/=?^_`{|}~-]+)*|"(?:[\x01-\x08\x0b\x0c\x0e-\x1f\x21\x23-\x5b\x5d-\x7f]|\\[\x01-\x09\x0b\x0c\x0e-\x7f])*")@(?:(?:[a-z0-9](?:[a-z0-9-]*[a-z0-9])?\.)+[a-z0-9](?:[a-z0-9-]*[a-z0-9])?|\[(?:(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.){3}(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?|[a-z0-9-]*[a-z0-9]:(?:[\x01-\x08\x0b\x0c\x0e-\x1f\x21-\x5a\x53-\x7f]|\\[\x01-\x09\x0b\x0c\x0e-\x7f])+)\])/;
const URL_REGEX = /((([A-Za-z]{3,9}:(?:\/\/)?)(?:[\-;:&=\+\$,\w]+@)?[A-Za-z0-9\.\-]+|(?:www\.|[\-;:&=\+\$,\w]+@)[A-Za-z0-9\.\-]+)((?:\/[\+~%\/\.\w\-_]*)?\??(?:[\-\+=&;%@\.\w_]*)#?(?:[\.\!\/\\\w]*))?)/;
const REPO_TEMP_DIRECTORY = ".repo";
var Stability;
(function (Stability) {
    Stability["EXPERIMENTAL"] = "experimental";
    Stability["STABLE"] = "stable";
    Stability["DEPRECATED"] = "deprecated";
})(Stability = exports.Stability || (exports.Stability = {}));
/**
 * Multi-language jsii library project
 *
 * @pjid jsii
 */
class JsiiProject extends typescript_1.TypeScriptProject {
    constructor(options) {
        const { authorEmail, authorUrl } = parseAuthorAddress(options);
        const defaultOptions = {
            repository: options.repositoryUrl,
            authorName: options.author,
            authorEmail,
            authorUrl,
            jestOptions: {
                jestVersion: "^27",
            },
        };
        const forcedOptions = {
            releaseToNpm: false,
            disableTsconfig: true,
            docgen: false,
        };
        const mergedOptions = util_1.deepMerge([
            defaultOptions,
            options,
            forcedOptions,
        ]);
        super(mergedOptions);
        const srcdir = this.srcdir;
        const libdir = this.libdir;
        this.addFields({ types: `${libdir}/index.d.ts` });
        const compressAssembly = options.compressAssembly ?? false;
        // this is an unhelpful warning
        const jsiiFlags = ["--silence-warnings=reserved-word"];
        if (compressAssembly) {
            jsiiFlags.push("--compress-assembly");
        }
        const compatIgnore = options.compatIgnore ?? ".compatignore";
        this.addFields({ stability: options.stability ?? Stability.STABLE });
        if (options.stability === Stability.DEPRECATED) {
            this.addFields({ deprecated: true });
        }
        const compatTask = this.addTask("compat", {
            description: "Perform API compatibility check against latest version",
            exec: `jsii-diff npm:$(node -p "require(\'./package.json\').name") -k --ignore-file ${compatIgnore} || (echo "\nUNEXPECTED BREAKING CHANGES: add keys such as \'removed:constructs.Node.of\' to ${compatIgnore} to skip.\n" && exit 1)`,
        });
        const compat = options.compat ?? false;
        if (compat) {
            this.compileTask.spawn(compatTask);
        }
        this.compileTask.reset(["jsii", ...jsiiFlags].join(" "));
        this.watchTask.reset(["jsii", "-w", ...jsiiFlags].join(" "));
        this.packageAllTask = this.addTask("package-all", {
            description: "Packages artifacts for all target languages",
        });
        // in jsii we consider the entire repo (post build) as the build artifact
        // which is then used to create the language bindings in separate jobs.
        const prepareRepoForCI = [
            `rsync -a . .repo --exclude .git --exclude node_modules`,
            `rm -rf ${this.artifactsDirectory}`,
            `mv .repo ${this.artifactsDirectory}`,
        ].join(" && ");
        // when running inside CI we just prepare the repo for packaging, which
        // takes place in separate tasks.
        // outside of CI (i.e locally) we simply package all targets.
        this.packageTask.reset(`if [ ! -z \${CI} ]; then ${prepareRepoForCI}; else ${this.runTaskCommand(this.packageAllTask)}; fi`);
        const targets = {};
        const jsii = {
            outdir: this.artifactsDirectory,
            targets,
            tsc: {
                outDir: libdir,
                rootDir: srcdir,
            },
        };
        if (options.excludeTypescript) {
            jsii.excludeTypescript = options.excludeTypescript;
        }
        this.addFields({ jsii });
        this.release?.publisher.addGitHubPrePublishingSteps({
            name: "Prepare Repository",
            run: `mv ${this.artifactsDirectory} ${REPO_TEMP_DIRECTORY}`,
        }, {
            name: "Collect GitHub Metadata",
            run: `mv ${REPO_TEMP_DIRECTORY}/${this.artifactsDirectory} ${this.artifactsDirectory}`,
        });
        const extraJobOptions = options.workflowRunsOn
            ? { runsOn: options.workflowRunsOn }
            : {};
        if (options.releaseToNpm != false) {
            const task = this.addPackagingTask("js");
            this.release?.publisher.publishToNpm({
                ...this.pacmakForLanguage("js", task),
                registry: this.package.npmRegistry,
                npmTokenSecret: this.package.npmTokenSecret,
                codeArtifactOptions: options.codeArtifactOptions,
            });
            this.addPackagingTarget("js", task, extraJobOptions);
        }
        // we cannot call an option `java` because the java code generated by jsii
        // does not compile due to a conflict between this option name and the `java`
        // package (e.g. when `java.util.Objects` is referenced).
        if ("java" in options) {
            throw new Error('the "java" option is now called "publishToMaven"');
        }
        if (options.publishToMaven) {
            targets.java = {
                package: options.publishToMaven.javaPackage,
                maven: {
                    groupId: options.publishToMaven.mavenGroupId,
                    artifactId: options.publishToMaven.mavenArtifactId,
                },
            };
            const task = this.addPackagingTask("java");
            this.release?.publisher.publishToMaven({
                ...this.pacmakForLanguage("java", task),
                ...options.publishToMaven,
            });
            this.addPackagingTarget("java", task, extraJobOptions);
        }
        const pypi = options.publishToPypi ?? options.python;
        if (pypi) {
            targets.python = {
                distName: pypi.distName,
                module: pypi.module,
            };
            const task = this.addPackagingTask("python");
            this.release?.publisher.publishToPyPi({
                ...this.pacmakForLanguage("python", task),
                ...pypi,
            });
            this.addPackagingTarget("python", task, extraJobOptions);
        }
        const nuget = options.publishToNuget ?? options.dotnet;
        if (nuget) {
            targets.dotnet = {
                namespace: nuget.dotNetNamespace,
                packageId: nuget.packageId,
                iconUrl: nuget.iconUrl,
            };
            const task = this.addPackagingTask("dotnet");
            this.release?.publisher.publishToNuget({
                ...this.pacmakForLanguage("dotnet", task),
                ...nuget,
            });
            this.addPackagingTarget("dotnet", task, extraJobOptions);
        }
        const golang = options.publishToGo;
        if (golang) {
            targets.go = {
                moduleName: golang.moduleName,
                packageName: golang.packageName,
            };
            const task = this.addPackagingTask("go");
            this.release?.publisher.publishToGo({
                ...this.pacmakForLanguage("go", task),
                ...golang,
            });
            this.addPackagingTarget("go", task, extraJobOptions);
        }
        this.addDevDeps("jsii", "jsii-diff", "jsii-pacmak");
        this.gitignore.exclude(".jsii", "tsconfig.json");
        this.npmignore?.include(".jsii");
        if (options.docgen ?? true) {
            new jsii_docgen_1.JsiiDocgen(this, { filePath: options.docgenFilePath });
        }
        // jsii updates .npmignore, so we make it writable
        if (this.npmignore) {
            this.npmignore.readonly = false;
        }
        // jsii relies on typescript < 4.0, which causes build errors
        // since @types/prettier@2.6.1 only supports typescript >= 4.2.
        // add a package resolution override to fix this.
        // this should have no effect if @types/prettier is not a transitive dependency
        this.package.addPackageResolutions("@types/prettier@2.6.0");
    }
    /**
     * Adds a target language to the build workflow and creates a package task.
     * @param language
     * @returns
     */
    addPackagingTarget(language, packTask, extraJobOptions) {
        if (!this.buildWorkflow) {
            return;
        }
        const pacmak = this.pacmakForLanguage(language, packTask);
        this.buildWorkflow.addPostBuildJob(`package-${language}`, {
            runsOn: ["ubuntu-latest"],
            permissions: {},
            tools: {
                node: { version: this.nodeVersion ?? "14.x" },
                ...pacmak.publishTools,
            },
            steps: pacmak.prePublishSteps ?? [],
            ...extraJobOptions,
        });
    }
    addPackagingTask(language) {
        const packageTask = this.tasks.addTask(`package:${language}`, {
            description: `Create ${language} language bindings`,
        });
        packageTask.exec(`jsii-pacmak -v --target ${language}`);
        this.packageAllTask.spawn(packageTask);
        return packageTask;
    }
    pacmakForLanguage(target, packTask) {
        // at this stage, `artifactsDirectory` contains the prebuilt repository.
        // for the publishing to work seamlessely, that directory needs to contain the actual artifact.
        // so we move the repo, create the artifact, and put it in the expected place.
        const prePublishSteps = [];
        if (this.package.packageManager === javascript_1.NodePackageManager.PNPM) {
            prePublishSteps.push({
                name: "Setup pnpm",
                uses: "pnpm/action-setup@v2.2.2",
                with: { version: "7" },
            });
        }
        prePublishSteps.push({
            name: "Prepare Repository",
            run: `mv ${this.artifactsDirectory} ${REPO_TEMP_DIRECTORY}`,
        }, {
            name: "Install Dependencies",
            run: `cd ${REPO_TEMP_DIRECTORY} && ${this.package.installCommand}`,
        }, {
            name: `Create ${target} artifact`,
            run: `cd ${REPO_TEMP_DIRECTORY} && npx projen ${packTask.name}`,
        }, {
            name: `Collect ${target} Artifact`,
            run: `mv ${REPO_TEMP_DIRECTORY}/${this.artifactsDirectory} ${this.artifactsDirectory}`,
        });
        return {
            publishTools: consts_1.JSII_TOOLCHAIN[target],
            prePublishSteps,
        };
    }
}
exports.JsiiProject = JsiiProject;
_a = JSII_RTTI_SYMBOL_1;
JsiiProject[_a] = { fqn: "projen.cdk.JsiiProject", version: "0.63.29" };
function parseAuthorAddress(options) {
    let authorEmail = options.authorEmail;
    let authorUrl = options.authorUrl;
    if (options.authorAddress) {
        if (options.authorEmail && options.authorEmail !== options.authorAddress) {
            throw new Error("authorEmail is deprecated and cannot be used in conjunction with authorAddress");
        }
        if (options.authorUrl && options.authorUrl !== options.authorAddress) {
            throw new Error("authorUrl is deprecated and cannot be used in conjunction with authorAddress.");
        }
        if (EMAIL_REGEX.test(options.authorAddress)) {
            authorEmail = options.authorAddress;
        }
        else if (URL_REGEX.test(options.authorAddress)) {
            authorUrl = options.authorAddress;
        }
        else {
            throw new Error(`authorAddress must be either an email address or a URL: ${options.authorAddress}`);
        }
    }
    return { authorEmail, authorUrl };
}
//# sourceMappingURL=data:application/json;base64,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