"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaRuntime = exports.LambdaFunction = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path_1 = require("path");
const case_1 = require("case");
const component_1 = require("../component");
const javascript_1 = require("../javascript");
const source_code_1 = require("../source-code");
const internal_1 = require("./internal");
/**
 * Generates a pre-bundled AWS Lambda function construct from handler code.
 *
 * To use this, create an AWS Lambda handler file under your source tree with
 * the `.lambda.ts` extension and add a `LambdaFunction` component to your
 * typescript project pointing to this entrypoint.
 *
 * This will add a task to your "compile" step which will use `esbuild` to
 * bundle the handler code into the build directory. It will also generate a
 * file `src/foo-function.ts` with a custom AWS construct called `FooFunction`
 * which extends `@aws-cdk/aws-lambda.Function` which is bound to the bundled
 * handle through an asset.
 *
 * @example
 *
 * new LambdaFunction(myProject, {
 *   srcdir: myProject.srcdir,
 *   entrypoint: 'src/foo.lambda.ts',
 * });
 */
class LambdaFunction extends component_1.Component {
    /**
     * Defines a pre-bundled AWS Lambda function construct from handler code.
     *
     * @param project The project to use
     * @param options Options
     */
    constructor(project, options) {
        super(project);
        const cdkDeps = options.cdkDeps;
        const bundler = javascript_1.Bundler.of(project);
        if (!bundler) {
            throw new Error("No bundler found. Please add a Bundler component to your project.");
        }
        const runtime = options.runtime ?? LambdaRuntime.NODEJS_14_X;
        // allow Lambda handler code to import dev-deps since they are only needed
        // during bundling
        const eslint = javascript_1.Eslint.of(project);
        eslint?.allowDevDeps(options.entrypoint);
        const entrypoint = options.entrypoint;
        if (!entrypoint.endsWith(internal_1.TYPESCRIPT_LAMBDA_EXT) &&
            !entrypoint.endsWith(internal_1.TYPESCRIPT_EDGE_LAMBDA_EXT)) {
            throw new Error(`${entrypoint} must have a ${internal_1.TYPESCRIPT_LAMBDA_EXT} or ${internal_1.TYPESCRIPT_EDGE_LAMBDA_EXT} extension`);
        }
        const basePath = path_1.join(path_1.dirname(entrypoint), path_1.basename(entrypoint, options.edgeLambda ? internal_1.TYPESCRIPT_EDGE_LAMBDA_EXT : internal_1.TYPESCRIPT_LAMBDA_EXT));
        const constructFile = options.constructFile ?? `${basePath}-function.ts`;
        if (path_1.extname(constructFile) !== ".ts") {
            throw new Error(`Construct file name "${constructFile}" must have a .ts extension`);
        }
        // type names
        const constructName = options.constructName ?? case_1.pascal(path_1.basename(basePath)) + "Function";
        const propsType = `${constructName}Props`;
        const bundle = bundler.addBundle(entrypoint, {
            target: runtime.esbuildTarget,
            platform: runtime.esbuildPlatform,
            externals: ["aws-sdk"],
            ...options.bundlingOptions,
            tsconfigPath: project?.tsconfigDev?.fileName,
        });
        // calculate the relative path between the directory containing the
        // generated construct source file to the directory containing the bundle
        // index.js by resolving them as absolute paths first.
        // e.g:
        //  - outfileAbs => `/project-outdir/assets/foo/bar/baz/foo-function/index.js`
        //  - constructAbs => `/project-outdir/src/foo/bar/baz/foo-function.ts`
        const outfileAbs = path_1.join(project.outdir, bundle.outfile);
        const constructAbs = path_1.join(project.outdir, constructFile);
        const relativeOutfile = path_1.relative(path_1.dirname(constructAbs), path_1.dirname(outfileAbs));
        const src = new source_code_1.SourceCode(project, constructFile);
        if (src.marker) {
            src.line(`// ${src.marker}`);
        }
        src.line("import * as path from 'path';");
        if (cdkDeps.cdkMajorVersion === 1) {
            if (options.edgeLambda) {
                src.line("import * as cloudfront from '@aws-cdk/aws-cloudfront';");
                cdkDeps.addV1Dependencies("@aws-cdk/aws-cloudfront");
            }
            src.line("import * as lambda from '@aws-cdk/aws-lambda';");
            src.line("import { Construct } from '@aws-cdk/core';");
            cdkDeps.addV1Dependencies("@aws-cdk/aws-lambda");
            cdkDeps.addV1Dependencies("@aws-cdk/core");
        }
        else {
            if (options.edgeLambda) {
                src.line("import * as cloudfront from 'aws-cdk-lib/aws-cloudfront';");
            }
            src.line("import * as lambda from 'aws-cdk-lib/aws-lambda';");
            src.line("import { Construct } from 'constructs';");
        }
        src.line();
        src.line("/**");
        src.line(` * Props for ${constructName}`);
        src.line(" */");
        if (options.edgeLambda) {
            src.open(`export interface ${propsType} extends cloudfront.experimental.EdgeFunctionProps {`);
        }
        else {
            src.open(`export interface ${propsType} extends lambda.FunctionOptions {`);
        }
        src.close("}");
        src.line();
        src.line("/**");
        src.line(` * An AWS Lambda function which executes ${internal_1.convertToPosixPath(basePath)}.`);
        src.line(" */");
        if (options.edgeLambda) {
            src.open(`export class ${constructName} extends cloudfront.experimental.EdgeFunction {`);
        }
        else {
            src.open(`export class ${constructName} extends lambda.Function {`);
        }
        src.open(`constructor(scope: Construct, id: string, props?: ${propsType}) {`);
        src.open("super(scope, id, {");
        src.line(`description: '${internal_1.convertToPosixPath(entrypoint)}',`);
        src.line("...props,");
        src.line(`runtime: new lambda.Runtime('${runtime.functionRuntime}', lambda.RuntimeFamily.NODEJS),`);
        src.line("handler: 'index.handler',");
        src.line(`code: lambda.Code.fromAsset(path.join(__dirname, '${internal_1.convertToPosixPath(relativeOutfile)}')),`);
        src.close("});");
        if ((options.awsSdkConnectionReuse ?? true) && !options.edgeLambda) {
            src.line("this.addEnvironment('AWS_NODEJS_CONNECTION_REUSE_ENABLED', '1', { removeInEdge: true });");
        }
        src.close("}");
        src.close("}");
        this.project.logger.verbose(`${basePath}: construct "${constructName}" generated under "${constructFile}"`);
        this.project.logger.verbose(`${basePath}: bundle task "${bundle.bundleTask.name}"`);
        if (bundle.watchTask) {
            this.project.logger.verbose(`${basePath}: bundle watch task "${bundle.watchTask.name}"`);
        }
    }
}
exports.LambdaFunction = LambdaFunction;
_a = JSII_RTTI_SYMBOL_1;
LambdaFunction[_a] = { fqn: "projen.awscdk.LambdaFunction", version: "0.63.29" };
/**
 * The runtime for the AWS Lambda function.
 */
class LambdaRuntime {
    constructor(
    /**
     * The Node.js runtime to use
     */
    functionRuntime, 
    /**
     * The esbuild setting to use.
     */
    esbuildTarget) {
        this.functionRuntime = functionRuntime;
        this.esbuildTarget = esbuildTarget;
        this.esbuildPlatform = "node";
    }
}
exports.LambdaRuntime = LambdaRuntime;
_b = JSII_RTTI_SYMBOL_1;
LambdaRuntime[_b] = { fqn: "projen.awscdk.LambdaRuntime", version: "0.63.29" };
/**
 * Node.js 10.x
 */
LambdaRuntime.NODEJS_10_X = new LambdaRuntime("nodejs10.x", "node10");
/**
 * Node.js 12.x
 */
LambdaRuntime.NODEJS_12_X = new LambdaRuntime("nodejs12.x", "node12");
/**
 * Node.js 14.x
 */
LambdaRuntime.NODEJS_14_X = new LambdaRuntime("nodejs14.x", "node14");
/**
 * Node.js 16.x
 */
LambdaRuntime.NODEJS_16_X = new LambdaRuntime("nodejs16.x", "node16");
//# sourceMappingURL=data:application/json;base64,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