"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AwsCdkDeps = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const semver = require("semver");
const component_1 = require("../component");
const dependencies_1 = require("../dependencies");
/**
 * Manages dependencies on the AWS CDK.
 */
class AwsCdkDeps extends component_1.Component {
    constructor(project, options) {
        super(project);
        this.cdkDependenciesAsDeps = options.cdkDependenciesAsDeps ?? true;
        this.dependencyType = options.dependencyType;
        this._packageNames = this.packageNames();
        const framework = determineFrameworkVersion(options);
        this.cdkVersion = framework.range;
        this.cdkMajorVersion = framework.major;
        this.cdkMinimumVersion = framework.minimum;
        this.addFrameworkDependency(options);
        // assert/assertions library
        this.addV1AssertionLibraryDependency(options);
        // constructs library
        this.addConstructsDependency(options.constructsVersion);
        // user-defined v1 dependencies (will only fail in CDK v2 if these have values)
        this.addV1Dependencies(...(options.cdkDependencies ?? []));
        this.addV1DevDependencies(...(options.cdkTestDependencies ?? []));
    }
    preSynthesize() {
        // Log a warning if any AWS CDK v1-only deps are found in the dependencies.
        const depNames = Array.from(new Set(this.project.deps.all.map((dep) => dep.name)));
        const v1Deps = depNames
            .filter((dep) => [PACKAGE_AWS_CDK_VERSION.V1].includes(cdkVersionOfPackage(dep)))
            .sort();
        if (this.cdkMajorVersion === 2 && v1Deps.length > 0) {
            this.project.logger.warn(`WARNING: Found CDK v1 deps in your project, even though your "cdkVersion" is 2.x: [${v1Deps.join(", ")}]. Check out https://docs.aws.amazon.com/cdk/v2/guide/migrating-v2.html for more information about using CDK v2 dependencies.`);
        }
    }
    /**
     * Adds dependencies to AWS CDK modules.
     *
     * The type of dependency is determined by the `dependencyType` option.
     *
     * This method is not supported in CDK v2. Use `project.addPeerDeps()` or
     * `project.addDeps()` as appropriate.
     *
     * @param deps names of cdk modules (e.g. `@aws-cdk/aws-lambda`).
     */
    addV1Dependencies(...deps) {
        if (deps.length > 0 && this.cdkMajorVersion !== 1) {
            throw new Error("addV1Dependencies() is not supported for CDK 2.x and above, use addDeps() or addPeerDeps() instead");
        }
        // this will add dependencies based on the type requested by the user
        // for libraries, this will be "peer" and for apps it will be "runtime"
        this.addV1DependenciesByType(this.dependencyType, ...deps);
        // add deps as runtime deps if `cdkDepsAsDeps` is true
        if (this.cdkDependenciesAsDeps) {
            this.addV1DependenciesByType(dependencies_1.DependencyType.RUNTIME, ...deps);
        }
    }
    /**
     * Adds AWS CDK modules as dev dependencies.
     *
     * This method is not supported in CDK v2. Use `project.addPeerDeps()` or
     * `project.addDeps()` as appropriate.
     *
     * @param deps fully qualified names of cdk modules (e.g. `@aws-cdk/aws-lambda`).
     */
    addV1DevDependencies(...deps) {
        if (deps.length > 0 && this.cdkMajorVersion !== 1) {
            throw new Error("addV1DevDependencies() is not supported for CDK 2.x and above, use addDevDeps()/addTestDeps() instead");
        }
        this.addV1DependenciesByType(dependencies_1.DependencyType.BUILD, ...deps);
    }
    addConstructsDependency(requestedVersion) {
        if (requestedVersion && !semver.parse(requestedVersion)) {
            throw new Error(`"constructsVersion" cannot be parsed as a semver version: ${requestedVersion}`);
        }
        const defaultVersion = this.cdkMajorVersion === 1 ? "3.2.27" : "10.0.5";
        const versionRequirement = `^${requestedVersion ?? defaultVersion}`;
        const constructsMajorVersion = semver.minVersion(versionRequirement)?.major;
        if (!constructsMajorVersion) {
            throw new Error(`Cannot determine major version of constructs version '${versionRequirement}'`);
        }
        switch (this.cdkMajorVersion) {
            case 1:
                if (constructsMajorVersion !== 3) {
                    throw new Error("AWS CDK 1.x requires constructs 3.x");
                }
                break;
            case 2:
                if (constructsMajorVersion !== 10) {
                    throw new Error("AWS CDK 2.x requires constructs 10.x");
                }
                break;
        }
        this.project.deps.addDependency(`${this._packageNames.constructs}@${versionRequirement}`, this.dependencyType);
        return versionRequirement;
    }
    /**
     * Adds a dependency on the AWS CDK framework (e.g. `@aws-cdk/core` for V1 or `aws-cdk-lib` for V1).
     */
    addFrameworkDependency(options) {
        switch (this.cdkMajorVersion) {
            case 1:
                this.addV1Dependencies(this._packageNames.coreV1);
                break;
            case 2:
                if (options.cdkDependencies !== undefined) {
                    throw new Error('cdkDependencies is not used for CDK 2.x. Use "peerDeps" or "deps" instead');
                }
                if (options.cdkDependenciesAsDeps !== undefined) {
                    throw new Error("cdkDependenciesAsDeps is not used for CDK 2.x");
                }
                if (options.cdkTestDependencies !== undefined) {
                    throw new Error('cdkTestDependencies is not used for CDK 2.x. Use "devDeps" or "testDeps" instead');
                }
                this.project.deps.addDependency(`${this._packageNames.coreV2}@${this.cdkVersion}`, this.dependencyType);
                break;
            default:
                throw new Error(`Unsupported AWS CDK major version ${this.cdkMajorVersion}.x`);
        }
    }
    addV1AssertionLibraryDependency(options) {
        if (this.cdkMajorVersion !== 1) {
            if (options.cdkAssert !== undefined) {
                throw new Error("cdkAssert is not used for CDK 2.x. Use the assertions library that is provided in aws-cdk-lib");
            }
            if (options.cdkAssertions !== undefined) {
                throw new Error("cdkAssertion is not used for CDK 2.x. Use the assertions library that is provided in aws-cdk-lib");
            }
            return;
        }
        const testDeps = new Array();
        if ((options.cdkAssert ?? true) && this._packageNames.assert) {
            testDeps.push(this._packageNames.assert);
        }
        // @aws-cdk/assertions is only available starting v1.111.0
        if (semver.gte(this.cdkMinimumVersion, "1.111.0") &&
            (options.cdkAssertions ?? true)) {
            testDeps.push(this._packageNames.assertions);
        }
        this.addV1DependenciesByType(dependencies_1.DependencyType.TEST, ...testDeps);
    }
    /**
     * Adds a set of dependencies with the user-specified dependency type.
     * @param deps The set of dependency specifications
     */
    addV1DependenciesByType(type, ...modules) {
        for (const module of modules) {
            this.project.deps.addDependency(`${module}@${this.cdkVersion}`, type);
        }
    }
}
exports.AwsCdkDeps = AwsCdkDeps;
_a = JSII_RTTI_SYMBOL_1;
AwsCdkDeps[_a] = { fqn: "projen.awscdk.AwsCdkDeps", version: "0.63.29" };
/**
 * Which AWS CDK version a construct library package belongs to.
 */
var PACKAGE_AWS_CDK_VERSION;
(function (PACKAGE_AWS_CDK_VERSION) {
    PACKAGE_AWS_CDK_VERSION["V1"] = "v1";
    PACKAGE_AWS_CDK_VERSION["V2"] = "v2";
    PACKAGE_AWS_CDK_VERSION["EITHER"] = "either";
    PACKAGE_AWS_CDK_VERSION["UNKNOWN"] = "unknown";
})(PACKAGE_AWS_CDK_VERSION || (PACKAGE_AWS_CDK_VERSION = {}));
function cdkVersionOfPackage(packageName) {
    if (packageName === "aws-cdk-lib") {
        return PACKAGE_AWS_CDK_VERSION.V2;
    }
    else if (packageName.startsWith("@aws-cdk/")) {
        if (packageName.endsWith("-alpha")) {
            return PACKAGE_AWS_CDK_VERSION.V2;
        }
        else if (AWS_CDK_V1_V2_SCOPED_PACKAGES.includes(packageName)) {
            return PACKAGE_AWS_CDK_VERSION.EITHER;
        }
        else {
            return PACKAGE_AWS_CDK_VERSION.V1;
        }
    }
    else {
        return PACKAGE_AWS_CDK_VERSION.UNKNOWN;
    }
}
/**
 * A list of all known packages in the "@aws-cdk/" scope that are published
 * both for v1 and v2.
 */
const AWS_CDK_V1_V2_SCOPED_PACKAGES = [
    "@aws-cdk/cfnspec",
    "@aws-cdk/cx-api",
    "@aws-cdk/region-info",
    "@aws-cdk/cloud-assembly-schema",
    "@aws-cdk/assert",
    "@aws-cdk/cloudformation-diff",
];
function determineFrameworkVersion(options) {
    const ver = semver.parse(options.cdkVersion);
    if (!ver) {
        throw new Error(`"cdkVersion" cannot be parsed as a semver version: ${options.cdkVersion}`);
    }
    return {
        minimum: ver.format(),
        range: options.cdkVersionPinning
            ? options.cdkVersion
            : `^${options.cdkVersion}`,
        major: ver.major,
    };
}
//# sourceMappingURL=data:application/json;base64,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