"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AwsCdkTypeScriptApp = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const fs = require("fs-extra");
const component_1 = require("../component");
const dependencies_1 = require("../dependencies");
const typescript_1 = require("../typescript");
const auto_discover_1 = require("./auto-discover");
const awscdk_deps_js_1 = require("./awscdk-deps-js");
const cdk_config_1 = require("./cdk-config");
const cdk_tasks_1 = require("./cdk-tasks");
/**
 * AWS CDK app in TypeScript
 *
 * @pjid awscdk-app-ts
 */
class AwsCdkTypeScriptApp extends typescript_1.TypeScriptAppProject {
    constructor(options) {
        super({
            ...options,
            sampleCode: false,
            bundlerOptions: {
                ...options.bundlerOptions,
                // we invoke the "bundle" task as part of the build step in cdk.json so
                // we don't want it to be added to the pre-compile phase.
                addToPreCompile: false,
            },
        });
        this.cdkDeps = new awscdk_deps_js_1.AwsCdkDepsJs(this, {
            dependencyType: dependencies_1.DependencyType.RUNTIME,
            ...options,
        });
        this.appEntrypoint = options.appEntrypoint ?? "main.ts";
        // CLI
        this.addDevDeps(`aws-cdk@${this.cdkDeps.cdkVersion}`);
        // no compile step because we do all of it in typescript directly
        this.compileTask.reset();
        this.cdkTasks = new cdk_tasks_1.CdkTasks(this);
        // add synth to the build
        this.postCompileTask.spawn(this.cdkTasks.synthSilent);
        const tsConfigFile = this.tsconfig?.fileName;
        if (!tsConfigFile) {
            throw new Error("Expecting tsconfig.json");
        }
        this.cdkConfig = new cdk_config_1.CdkConfig(this, {
            app: `npx ts-node -P ${tsConfigFile} --prefer-ts-exts ${path.posix.join(this.srcdir, this.appEntrypoint)}`,
            featureFlags: this.cdkDeps.cdkMajorVersion < 2,
            buildCommand: this.runTaskCommand(this.bundler.bundleTask),
            watchIncludes: [`${this.srcdir}/**/*.ts`, `${this.testdir}/**/*.ts`],
            watchExcludes: [
                "README.md",
                "cdk*.json",
                "**/*.d.ts",
                "**/*.js",
                "tsconfig.json",
                "package*.json",
                "yarn.lock",
                "node_modules",
            ],
            ...options,
        });
        this.gitignore.exclude(".parcel-cache/");
        this.npmignore?.exclude(`${this.cdkConfig.cdkout}/`);
        this.npmignore?.exclude(".cdk.staging/");
        if (this.tsconfig) {
            this.tsconfig.exclude.push(this.cdkConfig.cdkout);
        }
        this.addDevDeps("ts-node");
        if (options.sampleCode ?? true) {
            new SampleCode(this, this.cdkDeps.cdkMajorVersion);
        }
        new auto_discover_1.AutoDiscover(this, {
            srcdir: this.srcdir,
            testdir: this.testdir,
            lambdaOptions: options.lambdaOptions,
            tsconfigPath: this.tsconfigDev.fileName,
            cdkDeps: this.cdkDeps,
            lambdaAutoDiscover: options.lambdaAutoDiscover ?? true,
            edgeLambdaAutoDiscover: options.edgeLambdaAutoDiscover ?? true,
            lambdaExtensionAutoDiscover: options.lambdaExtensionAutoDiscover ?? true,
            integrationTestAutoDiscover: options.integrationTestAutoDiscover ?? true,
        });
    }
    /**
     * The CDK version this app is using.
     */
    get cdkVersion() {
        return this.cdkDeps.cdkVersion;
    }
    /**
     * Adds an AWS CDK module dependencies
     * @param modules The list of modules to depend on
     */
    addCdkDependency(...modules) {
        return this.cdkDeps.addV1Dependencies(...modules);
    }
}
exports.AwsCdkTypeScriptApp = AwsCdkTypeScriptApp;
_a = JSII_RTTI_SYMBOL_1;
AwsCdkTypeScriptApp[_a] = { fqn: "projen.awscdk.AwsCdkTypeScriptApp", version: "0.63.29" };
class SampleCode extends component_1.Component {
    constructor(project, cdkMajorVersion) {
        super(project);
        this.cdkMajorVersion = cdkMajorVersion;
        this.appProject = project;
    }
    synthesize() {
        const outdir = this.project.outdir;
        const srcdir = path.join(outdir, this.appProject.srcdir);
        if (fs.pathExistsSync(srcdir) &&
            fs.readdirSync(srcdir).filter((x) => x.endsWith(".ts"))) {
            return;
        }
        const srcImports = new Array();
        if (this.cdkMajorVersion < 2) {
            srcImports.push("import { App, Construct, Stack, StackProps } from '@aws-cdk/core';");
        }
        else {
            srcImports.push("import { App, Stack, StackProps } from 'aws-cdk-lib';");
            srcImports.push("import { Construct } from 'constructs';");
        }
        const srcCode = `${srcImports.join("\n")}

export class MyStack extends Stack {
  constructor(scope: Construct, id: string, props: StackProps = {}) {
    super(scope, id, props);

    // define resources here...
  }
}

// for development, use account/region from cdk cli
const devEnv = {
  account: process.env.CDK_DEFAULT_ACCOUNT,
  region: process.env.CDK_DEFAULT_REGION,
};

const app = new App();

new MyStack(app, '${this.project.name}-dev', { env: devEnv });
// new MyStack(app, '${this.project.name}-prod', { env: prodEnv });

app.synth();`;
        fs.mkdirpSync(srcdir);
        fs.writeFileSync(path.join(srcdir, this.appProject.appEntrypoint), srcCode);
        const testdir = path.join(outdir, this.appProject.testdir);
        if (fs.pathExistsSync(testdir) &&
            fs.readdirSync(testdir).filter((x) => x.endsWith(".ts"))) {
            return;
        }
        const testImports = new Array();
        if (this.cdkMajorVersion < 2) {
            testImports.push("import { App } from '@aws-cdk/core';");
            testImports.push("import { Template } from '@aws-cdk/assertions';");
        }
        else {
            testImports.push("import { App } from 'aws-cdk-lib';");
            testImports.push("import { Template } from 'aws-cdk-lib/assertions';");
        }
        const appEntrypointName = path.basename(this.appProject.appEntrypoint, ".ts");
        const testCode = `${testImports.join("\n")}
import { MyStack } from '../src/${appEntrypointName}';

test('Snapshot', () => {
  const app = new App();
  const stack = new MyStack(app, 'test');

  const template = Template.fromStack(stack);
  expect(template.toJSON()).toMatchSnapshot();
});`;
        fs.mkdirpSync(testdir);
        fs.writeFileSync(path.join(testdir, `${appEntrypointName}.test.ts`), testCode);
    }
}
//# sourceMappingURL=data:application/json;base64,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