import sys
import platform
import os
import re
import subprocess
from setuptools import find_packages
from setuptools.command.build_ext import build_ext
from distutils.core import setup, Extension
from distutils.version import LooseVersion


MINIMAL_DESCRIPTION = '''Sarina is an ASCII art generator written in Python3 and C++. It transforms an input image and a text file containing words and their weights into a unique ASCII art representation. The algorithm behind Sarina is randomized, ensuring that every output is distinct, even for identical inputs.'''

def get_requires():
    """Read requirements.txt."""
    requirements = open('requirements.txt', "r").read()
    return list(filter(lambda x: x != "", requirements.split()))

def read_description():
    """Read README.md and CHANGELOG.md."""
    try:
        with open("README.md") as r:
            description = "\n"
            description += r.read()
        return description
    except Exception:
        return MINIMAL_DESCRIPTION
    
class CMakeExtension(Extension):
    def __init__(self, name, sourcedir=''):
        Extension.__init__(self, name, sources=[])
        self.sourcedir = os.path.abspath(sourcedir)    

class CMakeBuild(build_ext):
    def run(self):
        try:
            out = subprocess.check_output(['cmake', '--version'])
        except OSError:
            raise RuntimeError("CMake must be installed to build the following extensions: " +
                               ", ".join(e.name for e in self.extensions))

        if platform.system() == "Windows":
            cmake_version = LooseVersion(re.search(r'version\s*([\d.]+)', out.decode()).group(1))
            if cmake_version < '3.1.0':
                raise RuntimeError("CMake >= 3.1.0 is required on Windows")

        for ext in self.extensions:
            self.build_extension(ext)

    def build_extension(self, ext):
        extdir = os.path.abspath(os.path.dirname(self.get_ext_fullpath(ext.name)))
        cmake_args = ['-DCMAKE_LIBRARY_OUTPUT_DIRECTORY=' + extdir,
                      '-DPYTHON_EXECUTABLE=' + sys.executable]

        cfg = 'Debug' if self.debug else 'Release'
        build_args = ['--config', cfg]

        if platform.system() == "Windows":
            cmake_args += ['-DCMAKE_LIBRARY_OUTPUT_DIRECTORY_{}={}'.format(cfg.upper(), extdir)]
            if sys.maxsize > 2**32:
                cmake_args += ['-A', 'x64']
            build_args += ['--', '/m']
        else:
            cmake_args += ['-DCMAKE_BUILD_TYPE=' + cfg]
            build_args += ['--', '-j2']

        env = os.environ.copy()
        env['CXXFLAGS'] = '{} -DVERSION_INFO=\\"{}\\"'.format(env.get('CXXFLAGS', ''),
                                                              self.distribution.get_version())
        if not os.path.exists(self.build_temp):
            os.makedirs(self.build_temp)
        subprocess.check_call(['cmake', ext.sourcedir] + cmake_args, cwd=self.build_temp, env=env)
        subprocess.check_call(['cmake', '--build', '.'] + build_args, cwd=self.build_temp)

    

setup(
    name="sarina-cli",
    version="0.0.8",
    author='Amin Alam',
    description='ASCII Art Generator',
    long_description=read_description(),
    long_description_content_type='text/markdown',
    # add requirements.txt to the package data
    install_requires=get_requires(),
    python_requires='>=3.5',
    license='MIT',
    include_package_data=True,
    url='https://github.com/AminAlam/Sarina',
    keywords="ASCII-Art Word-Cloud-Generator",
    entry_points={
        'console_scripts': [
            'sarina=sarina.__main__:main',
        ]
    },
    ext_modules=[CMakeExtension('sarina')],
    cmdclass=dict(build_ext=CMakeBuild),
    packages=find_packages()
    )

