#!/usr/bin/env python
"""
xchatbot - the Xtensible xmpp Chat Bot

Build an XMPP bot extending the base class `XChatBot`

Example:
    A simple bot with one public command "echo" and
    one private to admin command "hello"
    "help" command is autogenerated.

.. code-block:: python

    from xchatbot import XChatBot

    class MyBot(XChatBot):
        def cmd_echo(self, peer, *args):
            "Echo back what you typed"
            msg = "You said: " + " ".join(args)
            peer.send(msg)

        @private
        def cmd_hello(self, peer, name):
            "<name> - Private command for admin user"
            peer.send("Welcome " + name)

    if __name__ == "__main__":
        MyBot.start()


"""

import sys
import os
import warnings
import logging
from datetime import datetime
import collections
import pickle
import shlex
from functools import wraps
import signal

import nbxmpp
try:
    from gi.repository import GLib
except ImportError:
    import glib as GLib


__version__ = "0.3.1"
__all__ = ['Peer', 'XChatBot', 'store', 'load', 'private']

# logger
FORMATTER = logging.Formatter('%(levelname)-8s %(name)-10s %(message)s')
HANDLER = logging.StreamHandler()
HANDLER.setFormatter(FORMATTER)
logger = logging.getLogger(__name__)  # pylint: disable=invalid-name
logger.setLevel('DEBUG')
logger.addHandler(HANDLER)


# utility because the simplexml is.. too simple
# pylint: disable=invalid-name
def getTagAll(tag, tagname, **kwargs):
    """Get child nodes by name

    Search for a tag by name which is child of a root element,
    optionally with matching attributes

    Args:
        tag (:class:`nbxmpp.simplexml.Node`):  Root node
        tagname (:obj:`str`): Tag name to search
        **kwargs: Node attributes (key=value)


    Returns:
        :class:`nbxmpp.simplexml.Node`: None if tagname is not found
    """
    checkattrs = len(kwargs.keys()) > 0
    for child in tag.getChildren():
        if child.getName() == tagname:
            if not checkattrs or \
               all([child.getAttr(k) == v for k, v in kwargs.items()]):
                return child
        granchild = getTagAll(child, tagname, **kwargs)
        if granchild is not None:
            return granchild
    return None


def store(name, data):
    """store data as pickle value to file

    Args:
        name (:obj:`str`): Filename, without extension
        data (:obj:`str`):  Python data to store
    """
    with open(name + ".data", "wb") as fp:
        pickle.dump(data, fp)


def load(name, default):
    """Load pickled data from file.

    Args:
        name (:obj:`str`): Filename to load, without extension
        default (:obj:`any`): Python data returned if file is not found

    Returns:
        :obj:`str`: Loaded data

        If file is not found, the value of ``default`` parameter is returned
    """
    if not os.path.isfile(name + ".data"):
        return default
    with open(name + ".data", "rb") as fp:
        return pickle.load(fp)


def private(func):
    """Decorator to mark command private for admin user
    """
    @wraps(func)
    def command_wrapper(self, peer, *args):
        if peer.is_admin:
            func(self, peer, *args)
    command_wrapper.is_private = True
    return command_wrapper


class Peer:
    """The peer that sent the message

    Attributes:
        jid (:obj:`str`): the peer jid
        nick (:obj:`str`): the nickname
        is_admin (:obj:`bool`): ``True`` if the peer is a bot admin
        is_groupchat (:obj:`bool`): ``True`` if the peer wrote to the bot
                                    is in a MUC
    """

    def __init__(self, bot, jid, nick, is_groupchat=False):
        self.bot = bot
        self.jid = jid
        self.nick = nick
        self.is_admin = jid == self.bot.admin_jid
        self.is_groupchat = is_groupchat

    def send(self, message):
        """Send a message to the peer

        Note:
            If peer is in a groupchat, the nickname is prepended
            to the message:

                {nick}: {message}


        Args:
            message (:obj:`str`): The message
        """
        if self.is_groupchat:
            message = self.nick + ": " + message
            self.bot.send_groupchat_to(self.jid, message)
        else:
            self.bot.send_message_to(self.jid, message)

    def __str__(self):
        return "{} <{} > {}".format(
            self.nick, self.jid,
            " in a groupchat" if self.is_groupchat else ""
        )


# main class
class XChatBot:  # pylint: disable=too-many-instance-attributes
    """The Xtensible xmpp Chat Bot

    Attributes:
        options (:obj:`dict`): options loaded from config file
        jid (:obj:`nbxmpp.protocol.JID`): bot JID
        muc_nick (:obj:`str`): bot nick in MUC rooms
        admin_jid (:obj:`str`): admin user JID
        accept_presence (:obj:`bool`):
            ``True`` if bot accept to be included in rooster by any users
        accept_muc_invite(:obj:`bool`):
            ``True`` if bot accept invites to MUC rooms by any users
        logger (:obj:`logging.Logger`):
            configured logger to be used by the bot subclass

    """

    def __init__(self, jidparams):
        self.options = jidparams
        self.jid = nbxmpp.protocol.JID(jidparams['jid'])
        self.password = jidparams['password']
        self.muc_nick = jidparams.get("muc_nick", "xbot")
        self.sm = nbxmpp.Smacks(self)  # Stream Management
        self.client_cert = None
        self.client = None

        self.admin_jid = nbxmpp.protocol.JID(jidparams['admin']).getStripped()

        self.accept_presence = jidparams.get("accept_presence_request", False)
        self.accept_muc_invite = jidparams.get("muc_accept_invite", False)

        self.conferences = set()
        mucs = jidparams.get("mucs", "").strip()
        if mucs != "":
            self.conferences.update(mucs.split(";"))
        self.conferences.update(load("mucs", []))
        self.conferences_lastmessage = collections.defaultdict(str)
        logger.debug("conferences %r", self.conferences)

        self.seen_ids = load("ids", collections.deque(maxlen=1000))

        self.ignore_before = datetime.now()

        self.commands = {}
        for name in dir(self):
            func = getattr(self, name)
            if name.startswith("cmd_") and callable(func):
                cmd = name.replace("cmd_", "")
                self.commands[cmd] = func

        self.logger = logging.getLogger(self.__class__.__name__)
        self.logger.setLevel(self.options.get('logger', 'DEBUG'))
        self.logger.addHandler(HANDLER)

    @classmethod
    def start(cls):
        """Start bot

        Loads config file, connects the bot to the server, setups error and
        quit handlers, start :class:`GLib.MainLoop` loop.
        """
        conf = loadconf(cls.__name__.lower())

        ml = GLib.MainLoop()

        bot = cls(conf)
        bot.connect()

        def signal_handler(_signal, _frame):
            logger.info("disconnecting...")
            bot.disconnect()

        def on_disconnect():
            logger.info("quitting...")
            GLib.timeout_add(2000, ml.quit)

        def _excepthook(etype, value, traceback):
            print(etype, value, traceback)
            ml.quit()
            sys.exit(-1)

        sys.excepthook = _excepthook
        signal.signal(signal.SIGINT, signal_handler)
        bot.register_on_disconnect(on_disconnect)

        try:
            ml.run()
        except KeyboardInterrupt:
            logger.info("keyboard interrupt")
            bot.disconnect()

    # # manage xmpp connection
    def connect(self):
        """Connect to the server
        """
        logger.info("connecting to %s", self.jid.getDomain())
        idle_queue = nbxmpp.idlequeue.get_idlequeue()
        self.client = nbxmpp.NonBlockingClient(
            domain=self.jid.getDomain(),
            idlequeue=idle_queue,
            caller=self
        )
        self.client.connect(
            on_connect=self._on_connected,
            on_connect_failure=self._on_connection_failed,
            on_stream_error_cb=self._on_stream_error,
            secure_tuple=('tls', '', '', None, None)
        )

    def send_message(self, message):
        """Send a message

        Args:
            message (:obj:`nbxmpp.protocol.Message`): the message to send
        """
        id_ = self.client.send(message)  # pylint: disable=no-member
        self._add_seeenid(id_)
        logger.debug('sent message with id %s', id_)

    def send_message_to(self, to_jid, text):
        """Send a chat message to a JID

        Args:
            to_jid (:obj:`str`): Recipient JID
            text (:obj:`str`): Message text
        """
        id_ = self.client.send(  # pylint: disable=no-member
            nbxmpp.protocol.Message(to_jid, text, typ='chat')
        )
        self._add_seeenid(id_)
        logger.debug('sent message to %s with id %s', to_jid, id_)

    def send_groupchat_to(self, to_jid, text):
        """Send a message to a MUC by JID

        Args:
            to_jid (:obj:`str`): MUC JID
            text (:obj:`str`): Message text
        """
        id_ = self.client.send(  # pylint: disable=no-member
            nbxmpp.protocol.Message(to_jid, text, typ='groupchat')
        )
        self._add_seeenid(id_)
        logger.debug('sent groupchat message to %s with id %s', to_jid, id_)

    def send_received(self, to_jid, message_id):
        """Send a message delivery receipt

        Will mark the message as received by the bot in user's client.
        It's sent automatically when incoming messages are parsed.

        Args:
            to_jid (:obj:`str`): Recipient JID
            message_id (:obj:`str`): Message id
        """
        message = nbxmpp.protocol.Message(to_jid)
        message.setTag('received',
                       namespace=nbxmpp.protocol.NS_RECEIPTS,
                       attrs={'id': message_id})
        self.client.send(message)  # pylint: disable=no-member
        logger.debug('sent message delivery receipt for id %s', message_id)

    def quit(self):
        """Disconnecty the bot and quit

        Note:
            Deprecated. Use :func:`~xchatbot.XChatBot.disconnect`
        """
        warnings.warn("deprecated", DeprecationWarning)
        self.disconnect()
        # ml.quit()

    def disconnect(self):
        """Disconnect from the server
        """
        self.client.send(  # pylint: disable=no-member
            nbxmpp.protocol.Presence(typ='unavailable')
        )
        GLib.timeout_add(2000, lambda: self.client.disconnect(message="Hop!"))

    def register_on_disconnect(self, handler):
        """Register handler that will be called on disconnect
        """
        self.client.RegisterDisconnectHandler(handler)

    def _on_auth(self, _con, auth):
        if not auth:
            logger.error('could not authenticate!')
            sys.exit()
        logger.debug('authenticated using %r', auth)

        self.client.Dispatcher.RegisterHandler(  # pylint: disable=no-member
            "message", self._on_message)

        self.client.Dispatcher.RegisterHandler(  # pylint: disable=no-member
            "presence", self._on_presence_request, typ="subscribe")

        self.client.sendPresence()
        self.client.send(nbxmpp.protocol.Presence(  # pylint: disable=no-member
            show=True, status="Ricordati! Che devi morire!"))

        for muc_jid in self.conferences:
            self.enter_muc(muc_jid)
        logger.debug("Ready to go.")

    def _on_connected(self, _con, con_type):
        logger.info('connected with %r', con_type)
        self.client.auth(
            self.jid.getNode(), self.password,
            resource=self.jid.getResource(), sasl=1, on_auth=self._on_auth)

    def get_password(self, cb, _mech):
        """Get password

        By default, calls ``cb()`` with password from config file.

        Args:
            cb (:obj:`func(str)`): callback to call with the password
        """
        cb(self.password)

    def _on_connection_failed(self):  # pylint: disable=no-self-use
        logger.error('could not connect!')

    def _on_stream_error(self, *args, **kwargs):  # pylint: disable=no-self-use
        logger.error("!on_stream_error %r %r", args, kwargs)

    # # / manage xmpp connection

    # # manage xmpp events

    def _event_dispatcher(self, realm, event, data):
        # print("%r, %r, %r" % (realm, event, data))
        pass

    def _on_presence_request(self, _dispatcher, message):
        logger.debug("presence request %r", message)
        from_jid = message.getAttr("from")
        if from_jid != self.jid.getStripped():
            # https://xmpp.org/rfcs/rfc3921.html#rfc.section.2.2.1
            if self.accept_presence or from_jid == self.admin_jid:
                self.send_message(
                    nbxmpp.protocol.Presence(to=from_jid, typ="subscribed"))
            else:
                self.send_message(
                    nbxmpp.protocol.Presence(to=from_jid, typ="unsubscribed"))

    def _on_message(self, _dispatcher, message):
        logger.debug("on_message %s", message)

        if message.getAttr("type") == "error":
            logger.error("error: %r", message)
            return

        if message.getTag("received") is not None:
            message = getTagAll(message, "message")

        if message is None:
            return

        message_id = message.getAttr("id")
        if message_id in self.seen_ids:
            return

        delay = getTagAll(message, "delay") \
            or getTagAll(message, "x", xmlns="jabber:x:delay")

        if delay is not None:
            stamp = delay.getAttr("stamp")
            fmt = "%Y-%m-%dT%H:%M:%SZ"
            if "-" not in stamp:
                fmt = "%Y%m%dT%H:%M:%S"
            stampdatetime = datetime.strptime(stamp, fmt)
            if stampdatetime < self.ignore_before:
                return

        self._add_seeenid(message_id)

        myself = [self.jid.getStripped()]
        myself += [muc + "/" + self.muc_nick for muc in self.conferences]
        frm = str(message.getAttr("from"))
        if any([frm.startswith(me) or me.startswith(frm) for me in myself]):
            logger.debug("drop message event from self")
            return

        # TODO: skip history messages

        # manage invite to MUC
        if self.accept_muc_invite and getTagAll(message, "invite") is not None:
            conference = getTagAll(
                message, "x", xmlns=nbxmpp.protocol.NS_CONFERENCE)
            logger.debug("invited to %r", conference.getAttr("jid"))
            self.enter_muc(conference.getAttr("jid"))
            return

        if message.getAttr("type") == "chat":
            self._parse_message(message)

        if message.getAttr("type") == "groupchat":
            self._parse_message(message, True)

    def enter_muc(self, muc_jid):
        """Join a multiuser conference

        Args:
            muc_jid (:obj:`str`): MUC JID
        """
        muc_jid_nick = muc_jid + "/" + self.muc_nick
        logger.debug("send presence to %s", muc_jid_nick)
        x = nbxmpp.simplexml.Node(
            "x",
            attrs={"xmlns": "http://jabber.org/protocol/muc"})
        presence = nbxmpp.protocol.Presence(
            to=muc_jid_nick,
            payload=[x],
            attrs={"from": self.jid})
        logger.debug(presence)
        self.send_message(presence)
        self._add_conference(muc_jid)

    def _add_conference(self, muc_jid):
        if muc_jid not in self.conferences:
            self.conferences.add(muc_jid)
            store("mucs", self.conferences)

    def _add_seeenid(self, id_):
        self.seen_ids.append(id_)
        store("ids", self.seen_ids)

    # # / manage xmpp events

    # # bot logic

    def _parse_message(self, message, groupchat=False):
        bodytag = getTagAll(message, "body")
        if bodytag is None:
            logger.debug("can't find body tag!")
            return

        body = bodytag.getData().strip()
        logger.debug("parse_message: %r", body)

        from_jid_full = message.getAttr("from")
        from_jid = nbxmpp.protocol.JID(from_jid_full).getStripped()

        nick = ""
        if groupchat:
            nick = nbxmpp.protocol.JID(from_jid_full).getResource()
        else:
            nick = from_jid.split("@", 1)[0]

        if getTagAll(message, "request", xmlns=nbxmpp.protocol.NS_RECEIPTS):
            self.send_received(from_jid, message.getAttr("id"))

        peer = Peer(self, from_jid, nick, groupchat)

        logger.debug("from %s", peer)

        # check groupchat command
        if groupchat:
            tag = "!"+self.muc_nick
            if body.startswith(tag):
                body = body.replace(tag, "").strip()
            else:
                logger.debug("last message in %s : %s", from_jid, body)
                self.conferences_lastmessage[from_jid] = body
                return

        # parse commands

        body = body.strip()

        # # help
        if body.lower() in ["help", "aiuto", "guida", "?"]:
            self.do_help(peer)
            return

        cmdline = shlex.split(body)
        if len(cmdline) > 0:
            cmd = cmdline[0]
            args = cmdline[1:]
            if cmd in self.commands:
                try:
                    logger.debug("call %r(%s, *%r)",
                                 self.commands[cmd], peer, args)
                    self.commands[cmd](peer, *args)
                except TypeError as e:
                    logger.debug(e)
                    peer.send("Invalid arguments!")
                return

        # no commands givend, fallback to default
        self.default(peer, *cmdline)

    def do_help(self, peer):
        """Send help message to the peer

        Args:
            peer (:class:`Peer`): The remote peer
        """
        logger.debug("do_help")
        msg = "This bot reply to:\n"
        msg += "*help* - this message\n"
        for name, func in self.commands.items():
            if not getattr(func, "is_private", False) \
               or peer.jid == self.admin_jid:
                sep = "" if " - " in func.__doc__ else "- "
                msg += "*{}* {}{}\n".format(name, sep, func.__doc__.strip())
        peer.send(msg)

    # Bot callbacks

    def default(self, peer, *args):
        """Default command

        This is called when the peer sends a command not defined in the bot.

        Args:
            peer (:class:`Peer`): The remote peer
            *args: arguments of the command
        """


# # ENTRY POINT

def loadconf(botname='xchatbot'):
    """load and parse config file

    looks in ``$PWD/botname.rc``, ``~/.botname.rc`` and ``/etc/botname.rc``

    Args:
        botname (:obj:`str`): bot name.
            Usually it's the bot class name in lowercase.
    """
    jidparams = {}
    confiles = [
        '{}.rc'.format(botname),
        '{}/.{}.rc'.format(os.environ['HOME'], botname),
        '/etc/{}.rc'.format(botname)
    ]
    confile = None
    for cf in confiles:
        if os.access(cf, os.R_OK):
            confile = cf
            break
    if confile is None:
        print("Can't find config file in " + ", ".join(confiles))
        sys.exit(1)

    for ln in open(confile).readlines():
        ln = ln.strip()
        if ln != "" and not ln[0] in ('#', ';'):
            key, val = ln.strip().split('=', 1)
            if val.lower() == "false":
                val = False
            elif val.lower() == "true":
                val = True
            jidparams[key.lower()] = val

    for mandatory in ['jid', 'password', 'admin']:
        if mandatory not in jidparams.keys():
            print('Please set in config file {} a value for {}.'.format(
                confile, mandatory.upper()))
            sys.exit(2)

    if "logger" in jidparams:
        logger.setLevel(jidparams['logger'])

    return jidparams


if __name__ == "__main__":
    class EchoBot(XChatBot):
        """Demo echo bot"""
        def cmd_echo(self, peer, *args):  # pylint: disable=no-self-use
            """echo back the text"""
            peer.send(" ".join(args))

        @private
        def cmd_hello(self, peer, name):  # pylint: disable=no-self-use
            """<name> - Private command for admin user"""
            peer.send("Welcome " + name)

    EchoBot.start()
