"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Transliterator = void 0;
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const core_1 = require("@aws-cdk/core");
const s3 = require("../../s3");
const constants = require("../shared/constants");
const transliterator_1 = require("./transliterator");
/**
 * Transliterates jsii assemblies to various other languages.
 */
class Transliterator extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        const repository = props.vpcEndpoints
            ? (_a = props.codeArtifact) === null || _a === void 0 ? void 0 : _a.throughVpcEndpoint(props.vpcEndpoints.codeArtifactApi, props.vpcEndpoints.codeArtifact) : props.codeArtifact;
        const bucket = props.vpcEndpoints
            ? s3.throughVpcEndpoint(props.bucket, props.vpcEndpoints.s3)
            : props.bucket;
        const environment = {
            // temporaty hack to generate construct-hub compliant markdown.
            // see https://github.com/cdklabs/jsii-docgen/blob/master/src/docgen/render/markdown.ts#L172
            HEADER_SPAN: 'true',
            TARGET_LANGUAGE: props.language.toString(),
        };
        if (props.vpcEndpoints) {
            // Those are returned as an array of HOSTED_ZONE_ID:DNS_NAME... We care
            // only about the DNS_NAME of the first entry in that array (which is
            // the AZ-agnostic DNS name).
            environment.CODE_ARTIFACT_API_ENDPOINT = core_1.Fn.select(1, core_1.Fn.split(':', core_1.Fn.select(0, props.vpcEndpoints.codeArtifactApi.vpcEndpointDnsEntries)));
        }
        if (props.codeArtifact) {
            environment.CODE_ARTIFACT_DOMAIN_NAME = props.codeArtifact.repositoryDomainName;
            environment.CODE_ARTIFACT_DOMAIN_OWNER = props.codeArtifact.repositoryDomainOwner;
            environment.CODE_ARTIFACT_REPOSITORY_ENDPOINT = props.codeArtifact.repositoryNpmEndpoint;
        }
        const lambda = new transliterator_1.Transliterator(this, 'Resource', {
            description: `Creates ${props.language} documentation from jsii-enabled npm packages`,
            environment,
            logRetention: (_b = props.logRetention) !== null && _b !== void 0 ? _b : aws_logs_1.RetentionDays.TEN_YEARS,
            memorySize: 10240,
            timeout: core_1.Duration.minutes(15),
            tracing: aws_lambda_1.Tracing.PASS_THROUGH,
            vpc: props.vpc,
            vpcSubnets: props.vpcSubnets,
        });
        this.function = lambda;
        repository === null || repository === void 0 ? void 0 : repository.grantReadFromRepository(this.function);
        // The handler reads & writes to this bucket.
        bucket.grantRead(this.function, `${constants.STORAGE_KEY_PREFIX}*${constants.ASSEMBLY_KEY_SUFFIX}`);
        bucket.grantWrite(this.function, `${constants.STORAGE_KEY_PREFIX}*${constants.docsKeySuffix(props.language)}`);
        bucket.grantWrite(this.function, `${constants.STORAGE_KEY_PREFIX}*${constants.docsKeySuffix(props.language, '*')}`);
        bucket.grantWrite(this.function, `${constants.STORAGE_KEY_PREFIX}*${constants.docsKeySuffix(props.language)}${constants.NOT_SUPPORTED_SUFFIX}`);
        bucket.grantWrite(this.function, `${constants.STORAGE_KEY_PREFIX}*${constants.docsKeySuffix(props.language, '*')}${constants.NOT_SUPPORTED_SUFFIX}`);
    }
}
exports.Transliterator = Transliterator;
//# sourceMappingURL=data:application/json;base64,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