# AUTOGENERATED! DO NOT EDIT! File to edit: nbs/dataframe_pipeline.ipynb (unless otherwise specified).

__all__ = ['Node', 'chunkNode', 'frameEdge', 'colEdge', 'donothing', 'nothing', 'fillNaEdge', 'engTokEdge', 'CNTok',
           'capMinMaxEdge', 'trackVocabEdge', 'saveCSV', 'saveSQL', 'eng_twt_tk']

# Cell
import numpy as np
import pandas as pd
from itertools import chain
from collections import Counter

# Cell
class Node(object):
    def __init__(self, df, verbose=1):
        super(Node,self).__init__()
        self.df = df
        self.verbose = verbose
        self.pipenames = list()
        self.pipe = list()

    def __repr__(self):
        return "<forge pipeline node>\n\t|"+"\n\t|".join(self.pipenames)

    def __or__(self, edge):
        """
        use it as:
        node|edge|edge
        :param process_step:
        :return:
        """
        self.pipe.append(edge)
        self.pipenames.append(edge.edge_name)
        return self

    def run(self):
        for pro_i in range(len(self.pipe)):
            if self.verbose > 0: print("[df edge]:%s" % self.pipenames[pro_i])
            pro = self.pipe[pro_i]
            self.df = pro.pro(self.df)
        return self.df

class chunkNode(Node):
    def __init__(self, df_chunk, verbose=1):
        """
        Use this class instead of Node class, for huge data sourse like big csv or huge SQL table
        chunkNode(pd.read_csv("xxx.csv",chunksize = 1000), verbose =1)
        :param df_chunk: pandas dataframe with chunksize parameter
        :param verbose:
        """
        super(chunkNode, self).__init__(df = df_chunk, verbose = verbose)
        self.df_chunk = self.df

    def run(self):
        """
        Running iterations on the entire dataset
        :return: None
        """
        for df in self.df:
            for pro_i in range(len(self.pipe)):
                if self.verbose > 0: print("[df edge]:%s" % self.pipenames[pro_i])
                pro = self.pipe[pro_i]
                df = pro.pro(df)

    def testrun(self):
        """
        testing for 1 iteration
        :return: the result dataframe
        """
        testdf = next(self.df)
        print("Please restart the generator after running test",flush=True)
        for pro_i in range(len(self.pipe)):
            if self.verbose > 0: print("[df edge]:%s" % self.pipenames[pro_i])
            pro = self.pipe[pro_i]
            testdf = pro.pro(testdf)

        return testdf

# Cell
class frameEdge(object):
    def __init__(self, edge_name=None):
        super(frameEdge, self).__init__()
        if edge_name == None:
            edge_name = "DataFrame_Processing_Edge"
        self.edge_name = edge_name
        self.i = None

    def __mul__(self, cols):
        assert 0, "Only colEdge support * columns operation"

    def define(self, f):
        def wraper(df):
            return f(df)

        self.pro = wraper
        return wraper


class colEdge(object):
    def __init__(self, edge_name=None):
        super().__init__()
        if edge_name == None:
            edge_name = "DataSeries_Processing_Edge"
        self.edge_name = edge_name
        self.cols = []

    def __mul__(self, cols):
        self.cols = cols
        return self

    def __mod__(self,col):
        self.cols = [col]
        return self

    def define(self, f):
        def wraper(col):
            col = f(col)
            return col

        self.colpro = wraper
        return wraper

    def pro(self, df):
        for c in self.cols:
            df[c] = self.colpro(df[c])
        return df

nothing = frameEdge("empety_process")

@nothing.define
def donothing(df):
    return df

# Cell
class fillNaEdge(colEdge):
    def __init__(self, fill=0.):
        super().__init__("fillna_%s"%(fill))
        self.fill = fill

    def colpro(self, col):
        return col.fillna(self.fill)


class engTokEdge(colEdge):
    def __init__(self, tokenizer=None, max_len=None):
        """
        Default using tweet tokenizer
        """
        super().__init__("En Tokenization")
        if tokenizer ==None:
            from nltk.tokenize import TweetTokenizer
            self.tokenizer = TweetTokenizer()
        else:
            self.tokenizer = tokenizer
        self.max_len = max_len

    def colpro(self, c):
        return c.apply(lambda x: self.tokenizer.tokenize(x)[:self.max_len])

eng_twt_tk = engTokEdge()

class CNTok(colEdge):
    def __init__(self):
        """
        cntok = CNTok()
        datanode = start_node|cntok*["col1","col2"]
        datanode.run()
        """
        super().__init__("chinese_tokenize")
        from jieba import cut
        self.cut = cut

    def colpro(self, col):
        col = col.apply(lambda x:list(self.cut(str(x))))
        return col

class capMinMaxEdge(colEdge):
    def __init__(self, min_ = None, max_ = None):
        super().__init__("cap min:%s max:%s"%(min_,max_))
        self.min_ = min_
        self.max_ = max_

    def colpro(self,col):
        col = np.clip(col.values,a_min = self.min_, a_max = self.max_)
        return col

class trackVocabEdge(colEdge):
    """
    a colEdge
    input column should contain python list
    This edge will keep track a vocabulary pandas DataFrame
    tck_vcb = TrackVocab()
    tck_vcb.vocab is the accumulated vocabulary
    """
    def __init__(self):
        super().__init__("track vocab")
        self.vocab = pd.DataFrame({"token": [], "cnt": []})

    def colpro(self, col):
        lists = list(col)
        biglist = list(chain.from_iterable(lists))
        self.vocab = self.combine_vocab(self.build_vocab(biglist))
        return col

    def get_token_count_dict(self, full_tok):
        """count the token to a list"""
        return Counter(full_tok)

    def build_vocab(self, full_tok):
        ct_dict = self.get_token_count_dict(full_tok)
        tk, ct = list(ct_dict.keys()), list(ct_dict.values())

        return pd.DataFrame({"token": tk, "cnt": ct})

    def combine_vocab(self, new_vocab):
        combinedf = pd.concat([self.vocab, new_vocab]).groupby("token").sum().reset_index()
        return combinedf.sort_values(by="cnt", ascending=False).reset_index().rename(columns = {"index":"idx"})

    def save_vocab(self, json_url):
        self.vocab.to_json(json_url)


class saveCSV(frameEdge):
    """
    DataFrame Edge
    SaveCsv("/path/to/file.csv")
    """
    def __init__(self, csvpath, tocsv_conf={"sep": "\t", "index":False}):
        super().__init__("save to csv")
        self.csvpath = csvpath
        self.tocsv_conf = tocsv_conf
        self.header = True

    def pro(self, df):
        df.to_csv(self.csvpath,header=self.header, mode="a", **self.tocsv_conf)
        self.header = False
        return df

class saveSQL(frameEdge):
    """
    DataFrame Edge
    SaveSQL("table_name", con)
    """
    def __init__(self, table_name, con, tosql_conf={"index": False, "if_exists":"append"}):
        super().__init__("save to sql_table")
        self.table_name = table_name
        self.con = con
        self.tosql_conf = tosql_conf

    def pro(self, df):
        df.to_sql(self.table_name, con=self.con, **self.tosql_conf)
        return df