import { InterfaceVpcEndpoint } from '@aws-cdk/aws-ec2';
import { Grant, IGrantable } from '@aws-cdk/aws-iam';
import { Construct, IConstruct } from '@aws-cdk/core';
import * as api from './api';
export interface RepositoryProps {
    /**
     * The description of the Repository resource.
     */
    readonly description?: string;
    /**
     * The name of the Domain.
     *
     * @default - a name is generated by CDK.
     */
    readonly domainName?: string;
    /**
     * Specify `domainName` and `domainExists: true` in order to use an existing
     * CodeArtifact domain instead of creating a new one.
     *
     * @default false
     */
    readonly domainExists?: boolean;
    /**
     * The name of the Repository.
     *
     * @default - a name is generated by CDK.
     */
    readonly repositoryName?: string;
    /**
     * The name of upstream repositories to configure on this repository. Those
     * repositories must be in the same domain, hence this property can only be
     * used if `domainExists` is `true`.
     *
     * @default - none
     */
    readonly upstreams?: string[];
}
export interface IRepository extends IConstruct, api.IRepository {
    /** The ARN of the CodeArtifact Domain that contains the repository. */
    readonly repositoryDomainArn: string;
    /** The effective name of the CodeArtifact Domain. */
    readonly repositoryDomainName: string;
    /** The owner account of the CodeArtifact Domain. */
    readonly repositoryDomainOwner: string;
    /** The ARN of the CodeArtifact Repository. */
    readonly repositoryArn: string;
    /** The effective name of the CodeArtifact Repository. */
    readonly repositoryName: string;
    /** The URL to the endpoint of the CodeArtifact Repository for use with NPM. */
    readonly repositoryNpmEndpoint: string;
    /**
     * Grants read-only access to the repository, for use with NPM.
     *
     * @param grantee the entity to be granted read access.
     *
     * @returns the resulting `Grant`.
     */
    grantReadFromRepository(grantee: IGrantable): Grant;
}
/**
 * A CodeArtifact repository.
 */
export declare class Repository extends Construct implements IRepository {
    #private;
    /**
     * The ARN of the CodeArtifact Domain that contains the repository.
     */
    readonly repositoryDomainArn: string;
    /**
     * The name of the CodeArtifact Domain that contains the repository.
     */
    readonly repositoryDomainName: string;
    /**
     * The account ID that owns the CodeArtifact Domain that contains the repository.
     */
    readonly repositoryDomainOwner: string;
    /**
     * The ARN of the CodeArtifact Repository.
     */
    readonly repositoryArn: string;
    /**
     * The name of the CodeArtifact Repository.
     */
    readonly repositoryName: string;
    constructor(scope: Construct, id: string, props?: RepositoryProps);
    /**
     * Adds an external connection to this repository.
     *
     * @param id the id of the external connection (i.e: `public:npmjs`).
     */
    addExternalConnection(id: string): void;
    /**
     * The npm repository endpoint to use for interacting with this repository.
     */
    get repositoryNpmEndpoint(): string;
    /**
     * The S3 bucket in which CodeArtifact stores the package data. When using
     * VPC Endpoints for CodeArtifact, an S3 Gateway Endpoint must also be
     * available, which allows reading from this bucket.
     */
    get s3BucketArn(): string;
    grantReadFromRepository(grantee: IGrantable): Grant;
    /**
     * Obtains a view of this repository that is intended to be accessed though
     * VPC endpoints.
     *
     * @param apiEndpoint  an `InterfaceVpcEndpoint` to the `codeartifact.api`
     *                     service.
     * @param repoEndpoint an `InterfaceVpcEndpoint` to the
     *                     `codeartifact.repositories` service.
     *
     * @returns a view of this repository that appropriately grants permissions on
     *          the VPC endpoint policies, too.
     */
    throughVpcEndpoint(apiEndpoint: InterfaceVpcEndpoint, repoEndpoint: InterfaceVpcEndpoint): IRepository;
}
