"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const os = require("os");
const path = require("path");
const zlib_1 = require("zlib");
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
const fs = require("fs-extra");
const docgen = require("jsii-docgen");
const aws = require("../shared/aws.lambda-shared");
const code_artifact_lambda_shared_1 = require("../shared/code-artifact.lambda-shared");
const constants = require("../shared/constants");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
const language_1 = require("../shared/language");
const shell_out_lambda_shared_1 = require("../shared/shell-out.lambda-shared");
const constants_1 = require("./constants");
const ASSEMBLY_KEY_REGEX = new RegExp(`^${constants.STORAGE_KEY_PREFIX}((?:@[^/]+/)?[^/]+)/v([^/]+)${constants.ASSEMBLY_KEY_SUFFIX}$`);
// Capture groups:                                                    ┗━━━━━━━━━1━━━━━━━┛  ┗━━2━━┛
/**
 * This function receives an S3 event, and for each record, proceeds to download
 * the `.jsii` assembly the event refers to, transliterates it to the language,
 * configured in `TARGET_LANGUAGE`, and uploads the resulting `.jsii.<lang>`
 * object to S3.
 *
 * @param event   an S3 event payload
 * @param context a Lambda execution context
 *
 * @returns nothing
 */
function handler(event) {
    console.log(`Event: ${JSON.stringify(event, null, 2)}`);
    // We'll need a writable $HOME directory, or this won't work well, because
    // npm will try to write stuff like the `.npmrc` or package caches in there
    // and that'll bail out on EROFS if that fails.
    return ensureWritableHome(async () => {
        var _a, _b;
        const endpoint = process.env.CODE_ARTIFACT_REPOSITORY_ENDPOINT;
        if (!endpoint) {
            console.log('No CodeArtifact endpoint configured - using npm\'s default registry');
        }
        else {
            console.log(`Using CodeArtifact registry: ${endpoint}`);
            const domain = env_lambda_shared_1.requireEnv('CODE_ARTIFACT_DOMAIN_NAME');
            const domainOwner = process.env.CODE_ARTIFACT_DOMAIN_OWNER;
            const apiEndpoint = process.env.CODE_ARTIFACT_API_ENDPOINT;
            await code_artifact_lambda_shared_1.logInWithCodeArtifact({ endpoint, domain, domainOwner, apiEndpoint });
        }
        // Set up NPM shared cache directory (https://docs.npmjs.com/cli/v7/using-npm/config#cache)
        const npmCacheDir = process.env.NPM_CACHE;
        if (npmCacheDir) {
            // Create it if it does not exist yet...
            await fs.mkdirp(npmCacheDir);
            console.log(`Using shared NPM cache at: ${npmCacheDir}`);
            await shell_out_lambda_shared_1.shellOut('npm', 'config', 'set', `cache=${npmCacheDir}`);
        }
        const language = env_lambda_shared_1.requireEnv('TARGET_LANGUAGE');
        console.log(`TARGET_LANGUAGE=${language}`);
        const created = new Array();
        const [, packageName, packageVersion] = (_a = event.assembly.key.match(ASSEMBLY_KEY_REGEX)) !== null && _a !== void 0 ? _a : [];
        if (packageName == null) {
            throw new Error(`Invalid object key: "${event.assembly.key}". It was expected to match ${ASSEMBLY_KEY_REGEX}!`);
        }
        const packageFqn = `${packageName}@${packageVersion}`;
        console.log(`Source Bucket:  ${event.bucket}`);
        console.log(`Source Key:     ${event.assembly.key}`);
        console.log(`Source Version: ${event.assembly.versionId}`);
        console.log(`Fetching assembly: ${event.assembly.key}`);
        const assemblyResponse = await aws.s3().getObject({ Bucket: event.bucket, Key: event.assembly.key }).promise();
        if (!assemblyResponse.Body) {
            throw new Error(`Response body for assembly at key ${event.assembly.key} is empty`);
        }
        const assembly = JSON.parse(assemblyResponse.Body.toString('utf-8'));
        const submodules = Object.keys((_b = assembly.submodules) !== null && _b !== void 0 ? _b : {}).map(s => s.split('.')[1]);
        const isCSharpAndSupported = language === 'csharp' && assembly.targets.dotnet;
        const isOtherwiseSupported = language === 'typescript' || assembly.targets[language];
        if (!isCSharpAndSupported
            && !isOtherwiseSupported) {
            console.error(`Package ${assembly.name}@${assembly.version} does not support ${language}, skipping!`);
            console.log(`Assembly targets: ${JSON.stringify(assembly.targets, null, 2)}`);
            for (const submodule of [undefined, ...submodules]) {
                const key = event.assembly.key.replace(/\/[^/]+$/, constants.docsKeySuffix(language_1.DocumentationLanguage.fromString(language), submodule)) + constants.NOT_SUPPORTED_SUFFIX;
                const response = await uploadFile(event.bucket, key, event.assembly.versionId);
                created.push({ bucket: event.bucket, key, versionId: response.VersionId });
            }
            return created;
        }
        const generateDocs = aws_embedded_metrics_1.metricScope((metrics) => async (lang) => {
            metrics.setDimensions();
            metrics.setNamespace(constants_1.METRICS_NAMESPACE);
            const uploads = new Map();
            const docs = await docgen.Documentation.forPackage(packageFqn, { language: docgen.Language.fromString(lang) });
            function renderAndDispatch(submodule) {
                console.log(`Rendering documentation in ${lang} for ${packageFqn} (submodule: ${submodule})`);
                const page = Buffer.from(docs.render({ submodule, linkFormatter: linkFormatter(docs) }).render());
                metrics.putMetric("DocumentSizeBytes" /* DOCUMENT_SIZE */, page.length, aws_embedded_metrics_1.Unit.Bytes);
                const { buffer: body, contentEncoding } = compressContent(page);
                metrics.putMetric("CompressedDocumentSizeBytes" /* COMPRESSED_DOCUMENT_SIZE */, body.length, aws_embedded_metrics_1.Unit.Bytes);
                const key = event.assembly.key.replace(/\/[^/]+$/, constants.docsKeySuffix(language_1.DocumentationLanguage.fromString(lang), submodule));
                console.log(`Uploading ${key}`);
                const upload = uploadFile(event.bucket, key, event.assembly.versionId, body, contentEncoding);
                uploads.set(key, upload);
            }
            renderAndDispatch();
            for (const submodule of submodules) {
                renderAndDispatch(submodule);
            }
            for (const [key, upload] of uploads.entries()) {
                const response = await upload;
                created.push({ bucket: event.bucket, key, versionId: response.VersionId });
                console.log(`Finished uploading ${key} (Version ID: ${response.VersionId})`);
            }
        });
        await generateDocs(language);
        return created;
    });
}
exports.handler = handler;
function compressContent(buffer) {
    if (buffer.length < 1024) {
        return { buffer };
    }
    const gz = zlib_1.gzipSync(buffer, { level: 9 });
    // If it did not compress well, we'll keep the un-compressed original...
    if (gz.length >= buffer.length) {
        return { buffer };
    }
    return { buffer: gz, contentEncoding: 'gzip' };
}
async function ensureWritableHome(cb) {
    // Since $HOME is not set, or is not writable, we'll just go make our own...
    const fakeHome = await fs.mkdtemp(path.join(os.tmpdir(), 'fake-home'));
    console.log(`Made temporary $HOME directory: ${fakeHome}`);
    const oldHome = process.env.HOME;
    try {
        process.env.HOME = fakeHome;
        return await cb();
    }
    finally {
        process.env.HOME = oldHome;
        await fs.remove(fakeHome);
        console.log(`Cleaned-up temporary $HOME directory: ${fakeHome}`);
    }
}
function uploadFile(bucket, key, sourceVersionId, body, contentEncoding) {
    return aws.s3().putObject({
        Bucket: bucket,
        Key: key,
        Body: body,
        CacheControl: 'public, max-age=300, must-revalidate, proxy-revalidate',
        ContentEncoding: contentEncoding,
        ContentType: 'text/markdown; charset=UTF-8',
        Metadata: {
            'Origin-Version-Id': sourceVersionId !== null && sourceVersionId !== void 0 ? sourceVersionId : 'N/A',
        },
    }).promise();
}
/**
 * A link formatter to make sure type links redirect to the appropriate package
 * page in the webapp.
 */
function linkFormatter(docs) {
    function _formatter(type) {
        const packageName = type.source.assembly.name;
        const packageVersion = type.source.assembly.version;
        // the webapp sanitizes anchors - so we need to as well when
        // linking to them.
        const hash = sanitize(type.fqn);
        if (docs.assembly.name === packageName) {
            // link to the same package - just add the hash
            return `#${hash}`;
        }
        // cross link to another package
        return `/packages/${packageName}/v/${packageVersion}?lang=${type.language.toString()}${type.submodule ? `&submodule=${type.submodule}` : ''}#${hash}`;
    }
    return _formatter;
}
function sanitize(input) {
    return input
        .toLowerCase()
        .replace(/[^a-zA-Z0-9 ]/g, '')
        .replace(/ /g, '-');
}
;
//# sourceMappingURL=data:application/json;base64,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