"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Orchestration = void 0;
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_sqs_1 = require("@aws-cdk/aws-sqs");
const aws_stepfunctions_1 = require("@aws-cdk/aws-stepfunctions");
const tasks = require("@aws-cdk/aws-stepfunctions-tasks");
const core_1 = require("@aws-cdk/core");
const deep_link_1 = require("../../deep-link");
const runbook_url_1 = require("../../runbook-url");
const catalog_builder_1 = require("../catalog-builder");
const constants_1 = require("../shared/constants");
const language_1 = require("../shared/language");
const transliterator_1 = require("../transliterator");
const redrive_state_machine_1 = require("./redrive-state-machine");
const SUPPORTED_LANGUAGES = [
    language_1.DocumentationLanguage.PYTHON,
    language_1.DocumentationLanguage.TYPESCRIPT,
    language_1.DocumentationLanguage.JAVA,
    language_1.DocumentationLanguage.CSHARP,
];
/**
 * This retry policy is used for all items in the state machine and allows ample
 * retry attempts in order to avoid having to implement a custom backpressure
 * handling mehanism.
 *
 * This is meant as a stop-gap until we can implement a more resilient system,
 * which likely will involve more SQS queues, but will probably need to be
 * throughoutly vetted before it is rolled out everywhere.
 *
 * After 30 attempts, given the parameters, the last attempt will wait just
 * under 16 minutes, which should be enough for currently running Lambda
 * functions to complete (or time out after 15 minutes). The total time spent
 * waiting between retries by this time is just over 3 hours. This is a lot of
 * time, but in extreme burst situations (i.e: reprocessing everything), this
 * is actually a good thing.
 */
const THROTTLE_RETRY_POLICY = { backoffRate: 1.1, interval: core_1.Duration.minutes(1), maxAttempts: 30 };
/**
 * Orchestrates the backend processing tasks using a StepFunctions State Machine.
 */
class Orchestration extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.deadLetterQueue = new aws_sqs_1.Queue(this, 'DLQ', {
            encryption: aws_sqs_1.QueueEncryption.KMS_MANAGED,
            retentionPeriod: core_1.Duration.days(14),
            visibilityTimeout: core_1.Duration.minutes(15),
        });
        props.monitoring.addHighSeverityAlarm('Backend Orchestration Dead-Letter Queue is not empty', new aws_cloudwatch_1.MathExpression({
            expression: 'm1 + m2',
            label: 'Dead-Letter Queue not empty',
            usingMetrics: {
                m1: this.deadLetterQueue.metricApproximateNumberOfMessagesVisible({ period: core_1.Duration.minutes(1) }),
                m2: this.deadLetterQueue.metricApproximateNumberOfMessagesNotVisible({ period: core_1.Duration.minutes(1) }),
            },
        }).createAlarm(this, 'DLQAlarm', {
            alarmName: `${this.deadLetterQueue.node.path}/NotEmpty`,
            alarmDescription: [
                'Backend orchestration dead-letter queue is not empty.',
                '',
                `RunBook: ${runbook_url_1.RUNBOOK_URL}`,
                '',
                `Direct link to queue: ${deep_link_1.sqsQueueUrl(this.deadLetterQueue)}`,
                'Warning: State Machines executions that sent messages to the DLQ will not show as "failed".',
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 1,
        }));
        const sendToDeadLetterQueue = new tasks.SqsSendMessage(this, 'Send to Dead Letter Queue', {
            messageBody: aws_stepfunctions_1.TaskInput.fromJsonPathAt('$'),
            queue: this.deadLetterQueue,
            resultPath: aws_stepfunctions_1.JsonPath.DISCARD,
        });
        this.catalogBuilder = new catalog_builder_1.CatalogBuilder(this, 'CatalogBuilder', props);
        const addToCatalog = new tasks.LambdaInvoke(this, 'Add to catalog.json', {
            lambdaFunction: this.catalogBuilder.function,
            resultPath: '$.catalogBuilderOutput',
            resultSelector: {
                'ETag.$': '$.Payload.ETag',
                'VersionId.$': '$.Payload.VersionId',
            },
        })
            // This has a concurrency of 1, so we want to aggressively retry being throttled here.
            .addRetry({ errors: ['Lambda.TooManyRequestsException'], ...THROTTLE_RETRY_POLICY })
            .addCatch(new aws_stepfunctions_1.Pass(this, '"Add to catalog.json" throttled', {
            parameters: { 'error.$': '$.Cause' },
            resultPath: '$.error',
        }).next(sendToDeadLetterQueue), { errors: ['Lambda.TooManyRequestsException'] })
            .addCatch(new aws_stepfunctions_1.Pass(this, '"Add to catalog.json" failure', {
            parameters: { 'error.$': 'States.StringToJson($.Cause)' },
            resultPath: '$.error',
        }).next(sendToDeadLetterQueue), { errors: ['States.TaskFailed'] })
            .addCatch(new aws_stepfunctions_1.Pass(this, '"Add to catalog.json" fault', {
            parameters: { 'error.$': '$.Cause' },
            resultPath: '$.error',
        }).next(sendToDeadLetterQueue), { errors: ['States.ALL'] });
        const docGenResultsKey = 'DocGen';
        const sendToDlqIfNeeded = new aws_stepfunctions_1.Choice(this, 'Any Failure?')
            .when(aws_stepfunctions_1.Condition.or(...SUPPORTED_LANGUAGES.map((_, i) => aws_stepfunctions_1.Condition.isPresent(`$.${docGenResultsKey}[${i}].error`))), sendToDeadLetterQueue)
            .otherwise(new aws_stepfunctions_1.Succeed(this, 'Success'));
        this.ecsCluster = new aws_ecs_1.Cluster(this, 'Cluster', {
            containerInsights: true,
            enableFargateCapacityProviders: true,
            vpc: props.vpc,
        });
        this.transliterator = new transliterator_1.Transliterator(this, 'Transliterator', props);
        const definition = new aws_stepfunctions_1.Pass(this, 'Track Execution Infos', {
            inputPath: '$$.Execution',
            parameters: {
                'Id.$': '$.Id',
                'Name.$': '$.Name',
                'RoleArn.$': '$.RoleArn',
                'StartTime.$': '$.StartTime',
            },
            resultPath: '$.$TaskExecution',
        }).next(new aws_stepfunctions_1.Parallel(this, 'DocGen', { resultPath: `$.${docGenResultsKey}` })
            .branch(...SUPPORTED_LANGUAGES.map((language) => {
            return new aws_stepfunctions_1.Choice(this, `Is ${language} needed?`)
                .when(aws_stepfunctions_1.Condition.or(aws_stepfunctions_1.Condition.isNotPresent('$.languages'), aws_stepfunctions_1.Condition.and(aws_stepfunctions_1.Condition.isPresent(`$.languages.${language}`), aws_stepfunctions_1.Condition.booleanEquals(`$.languages.${language}`, true))), 
            // We have to prepare the input to be a JSON string, within an array, because the ECS task integration expects
            // an array of strings (the model if that of a CLI invocation).
            // Unfortunately, we have to split this in two Pass states, because I don't know how to make it work otherwise.
            new aws_stepfunctions_1.Pass(this, `Prepare ${language}`, {
                parameters: { command: { 'bucket.$': '$.bucket', 'assembly.$': '$.assembly', '$TaskExecution.$': '$.$TaskExecution' } },
                resultPath: '$',
            })
                .next(new aws_stepfunctions_1.Pass(this, `Stringify ${language} input`, {
                parameters: { 'commands.$': 'States.Array(States.JsonToString($.command))' },
                resultPath: '$',
            })
                .next(this.transliterator.createEcsRunTask(this, `Generate ${language} docs`, {
                cluster: this.ecsCluster,
                inputPath: '$.commands',
                language,
                resultSelector: { result: { 'language': language.toString(), 'success.$': '$' } },
                // Expect this to complete within one hour
                timeout: core_1.Duration.hours(1),
                vpcSubnets: props.vpcSubnets,
            })
                // Do not retry NoSpaceLeftOnDevice errors, these are typically not transient.
                .addRetry({ errors: ['jsii-docgen.NoSpaceLeftOnDevice'], maxAttempts: 0 })
                .addRetry({
                errors: [
                    'ECS.AmazonECSException',
                    'ECS.InvalidParameterException',
                    'jsii-docgen.NpmError.E429',
                    'jsii-codgen.NpmError.EPROTO',
                ],
                ...THROTTLE_RETRY_POLICY,
            })
                .addRetry({
                errors: ['jsii-docgen.NpmError.ETARGET'],
                // We'll wait longer between retries. This is to account for CodeArtifact's lag behind npm
                backoffRate: 2,
                interval: core_1.Duration.minutes(5),
                maxAttempts: 3,
            })
                .addRetry({ maxAttempts: 3 })
                .addCatch(new aws_stepfunctions_1.Pass(this, `"Generate ${language} docs" timed out`, { parameters: { error: 'Timed out!', language: language.toString() } }), { errors: ['States.Timeout'] })
                .addCatch(new aws_stepfunctions_1.Pass(this, `"Generate ${language} docs" service error`, { parameters: { 'error.$': '$.Cause', 'language': language.toString() } }), { errors: ['ECS.AmazonECSException', 'ECS.InvalidParameterException'] })
                .addCatch(new aws_stepfunctions_1.Pass(this, `"Generate ${language} docs" failure`, { parameters: { 'error.$': 'States.StringToJson($.Cause)', 'language': language.toString() } }), { errors: ['States.TaskFailed'] })
                .addCatch(new aws_stepfunctions_1.Pass(this, `"Generate ${language} docs" fault`, { parameters: { 'error.$': '$.Cause', 'language': language.toString() } }), { errors: ['States.ALL'] }))))
                .otherwise(new aws_stepfunctions_1.Pass(this, `Skip ${language}`, { result: aws_stepfunctions_1.Result.fromObject({ language: language.toString(), success: 'NOOP (Skipped)' }), resultPath: '$' }));
        }))
            .next(new aws_stepfunctions_1.Choice(this, 'Any Success?')
            .when(aws_stepfunctions_1.Condition.or(...SUPPORTED_LANGUAGES.map((_, i) => aws_stepfunctions_1.Condition.isNotPresent(`$.${docGenResultsKey}[${i}].error`))), addToCatalog.next(sendToDlqIfNeeded))
            .otherwise(sendToDlqIfNeeded)));
        this.stateMachine = new aws_stepfunctions_1.StateMachine(this, 'Resource', {
            definition,
            stateMachineName: stateMachineNameFrom(this.node.path),
            timeout: core_1.Duration.days(1),
            tracingEnabled: true,
        });
        if (props.vpc) {
            // Ensure the State Machine does not get to run before the VPC can be used.
            this.stateMachine.node.addDependency(props.vpc.internetConnectivityEstablished);
        }
        props.monitoring.addHighSeverityAlarm('Backend Orchestration Failed', this.stateMachine.metricFailed()
            .createAlarm(this, 'OrchestrationFailed', {
            alarmName: `${this.stateMachine.node.path}/${this.stateMachine.metricFailed().metricName}`,
            alarmDescription: [
                'Backend orchestration failed!',
                '',
                `RunBook: ${runbook_url_1.RUNBOOK_URL}`,
                '',
                `Direct link to state machine: ${deep_link_1.stateMachineUrl(this.stateMachine)}`,
                'Warning: messages that resulted in a failed exectuion will NOT be in the DLQ!',
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 1,
        }));
        // This function is intended to be manually triggered by an operrator to
        // attempt redriving messages from the DLQ.
        this.redriveFunction = new redrive_state_machine_1.RedriveStateMachine(this, 'Redrive', {
            description: '[ConstructHub/Redrive] Manually redrives all messages from the backend dead letter queue',
            environment: {
                STATE_MACHINE_ARN: this.stateMachine.stateMachineArn,
                QUEUE_URL: this.deadLetterQueue.queueUrl,
            },
            memorySize: 1024,
            timeout: core_1.Duration.minutes(15),
            tracing: aws_lambda_1.Tracing.ACTIVE,
        });
        this.stateMachine.grantStartExecution(this.redriveFunction);
        this.deadLetterQueue.grantConsumeMessages(this.redriveFunction);
        // The workflow is intended to be manually triggered by an operator to
        // reprocess all package versions currently in store through the orchestrator.
        this.regenerateAllDocumentation = new RegenerateAllDocumentation(this, 'RegenerateAllDocumentation', {
            bucket: props.bucket,
            stateMachine: this.stateMachine,
        }).stateMachine;
    }
    metricEcsTaskCount(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'TaskCount',
            namespace: 'ECS/ContainerInsights',
        });
    }
    metricEcsCpuReserved(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'CpuReserved',
            namespace: 'ECS/ContainerInsights',
        });
    }
    metricEcsCpuUtilized(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'CpuUtilized',
            namespace: 'ECS/ContainerInsights',
        });
    }
    metricEcsCpuUtilization(opts) {
        return new aws_cloudwatch_1.MathExpression({
            ...opts,
            // Calculates the % CPU utilization from the CPU units utilization &
            // reservation. FILL is used to make a non-sparse time-series (the metrics
            // are not emitted if no task runs)
            expression: '100 * FILL(mCpuUtilized, 0) / FILL(mCpuReserved, REPEAT)',
            usingMetrics: {
                mCpuReserved: this.metricEcsCpuReserved(),
                mCpuUtilized: this.metricEcsCpuUtilized(),
            },
        });
    }
    metricEcsMemoryReserved(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'MemoryReserved',
            namespace: 'ECS/ContainerInsights',
        });
    }
    metricEcsMemoryUtilized(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'MemoryUtilized',
            namespace: 'ECS/ContainerInsights',
        });
    }
    metricEcsMemoryUtilization(opts) {
        return new aws_cloudwatch_1.MathExpression({
            ...opts,
            // Calculates the % memory utilization from the RAM utilization &
            // reservation. FILL is used to make a non-sparse time-series (the metrics
            // are not emitted if no task runs)
            expression: '100 * FILL(mMemoryUtilized, 0) / FILL(mMemoryReserved, REPEAT)',
            usingMetrics: {
                mMemoryReserved: this.metricEcsMemoryReserved(),
                mMemoryUtilized: this.metricEcsMemoryUtilized(),
            },
        });
    }
    metricEcsNetworkRxBytes(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'NetworkRxBytes',
            namespace: 'ECS/ContainerInsights',
        });
    }
    metricEcsNetworkTxBytes(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'NetworkTxBytes',
            namespace: 'ECS/ContainerInsights',
        });
    }
}
exports.Orchestration = Orchestration;
class RegenerateAllDocumentation extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const processVersions = new aws_stepfunctions_1.Choice(this, 'Get package versions page')
            .when(aws_stepfunctions_1.Condition.isPresent('$.response.NextContinuationToken'), new tasks.CallAwsService(this, 'Next versions page', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                ContinuationToken: aws_stepfunctions_1.JsonPath.stringAt('$.response.NextContinuationToken'),
                Delimiter: '/',
                Prefix: aws_stepfunctions_1.JsonPath.stringAt('$.Prefix'),
            },
            resultPath: '$.response',
        }))
            .otherwise(new tasks.CallAwsService(this, 'First versions page', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                Delimiter: '/',
                Prefix: aws_stepfunctions_1.JsonPath.stringAt('$.Prefix'),
            },
            resultPath: '$.response',
        }))
            .afterwards()
            .next(new aws_stepfunctions_1.Map(this, 'For each key prefix', { itemsPath: '$.response.CommonPrefixes', resultPath: aws_stepfunctions_1.JsonPath.DISCARD })
            .iterator(new tasks.StepFunctionsStartExecution(this, 'Start Orchestration Workflow', {
            stateMachine: props.stateMachine,
            input: aws_stepfunctions_1.TaskInput.fromObject({
                // Associate the child workflow with the execution that started it.
                AWS_STEP_FUNCTIONS_STARTED_BY_EXECUTION_ID: aws_stepfunctions_1.JsonPath.stringAt('$$.Execution.Id'),
                bucket: props.bucket.bucketName,
                assembly: { key: aws_stepfunctions_1.JsonPath.stringAt(`States.Format('{}${constants_1.ASSEMBLY_KEY_SUFFIX.substr(1)}', $.Prefix)`) },
                metadata: { key: aws_stepfunctions_1.JsonPath.stringAt(`States.Format('{}${constants_1.METADATA_KEY_SUFFIX.substr(1)}', $.Prefix)`) },
                package: { key: aws_stepfunctions_1.JsonPath.stringAt(`States.Format('{}${constants_1.PACKAGE_KEY_SUFFIX.substr(1)}', $.Prefix)`) },
            }),
            integrationPattern: aws_stepfunctions_1.IntegrationPattern.REQUEST_RESPONSE,
        })));
        processVersions.next(new aws_stepfunctions_1.Choice(this, 'Has more versions?')
            .when(aws_stepfunctions_1.Condition.isPresent('$.response.NextContinuationToken'), processVersions)
            .otherwise(new aws_stepfunctions_1.Succeed(this, 'Success')));
        const processPackageVersions = new aws_stepfunctions_1.StateMachine(this, 'PerPackage', {
            definition: processVersions,
            timeout: core_1.Duration.hours(1),
            tracingEnabled: true,
        });
        // This workflow is broken into two sub-workflows because otherwise it hits the 25K events limit
        // of StepFunction executions relatively quickly.
        const processNamespace = new aws_stepfunctions_1.Choice(this, 'Get @scope page')
            .when(aws_stepfunctions_1.Condition.isPresent('$.response.NextContinuationToken'), new tasks.CallAwsService(this, 'Next @scope page', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                ContinuationToken: aws_stepfunctions_1.JsonPath.stringAt('$.response.NextContinuationToken'),
                Delimiter: '/',
                Prefix: aws_stepfunctions_1.JsonPath.stringAt('$.Prefix'),
            },
            resultPath: '$.response',
        }))
            .otherwise(new tasks.CallAwsService(this, 'First @scope page', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                Delimiter: '/',
                Prefix: aws_stepfunctions_1.JsonPath.stringAt('$.Prefix'),
            },
            resultPath: '$.response',
        }))
            .afterwards()
            .next(new aws_stepfunctions_1.Map(this, 'For each @scope/pkg', { itemsPath: '$.response.CommonPrefixes', resultPath: aws_stepfunctions_1.JsonPath.DISCARD })
            .iterator(new tasks.StepFunctionsStartExecution(this, 'Process scoped package', {
            stateMachine: processPackageVersions,
            input: aws_stepfunctions_1.TaskInput.fromObject({
                // Associate the child workflow with the execution that started it,
                AWS_STEP_FUNCTIONS_STARTED_BY_EXECUTION_ID: aws_stepfunctions_1.JsonPath.stringAt('$$.Execution.Id'),
                Prefix: aws_stepfunctions_1.JsonPath.stringAt('$.Prefix'),
            }),
            integrationPattern: aws_stepfunctions_1.IntegrationPattern.RUN_JOB,
        })));
        processNamespace.next(new aws_stepfunctions_1.Choice(this, 'Has more packages?')
            .when(aws_stepfunctions_1.Condition.isPresent('$.response.NextContinuationToken'), processNamespace)
            .otherwise(new aws_stepfunctions_1.Succeed(this, 'All Done')));
        const start = new aws_stepfunctions_1.Choice(this, 'Get prefix page')
            .when(aws_stepfunctions_1.Condition.isPresent('$.response.NextContinuationToken'), new tasks.CallAwsService(this, 'Next prefixes page', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                ContinuationToken: aws_stepfunctions_1.JsonPath.stringAt('$.response.NextContinuationToken'),
                Delimiter: '/',
                Prefix: constants_1.STORAGE_KEY_PREFIX,
            },
            resultPath: '$.response',
        }))
            .otherwise(new tasks.CallAwsService(this, 'First prefix page', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                Delimiter: '/',
                Prefix: constants_1.STORAGE_KEY_PREFIX,
            },
            resultPath: '$.response',
        })).afterwards()
            .next(new aws_stepfunctions_1.Map(this, 'For each prefix', { itemsPath: '$.response.CommonPrefixes', resultPath: aws_stepfunctions_1.JsonPath.DISCARD })
            .iterator(new aws_stepfunctions_1.Choice(this, 'Is this a @scope/ prefix?')
            .when(aws_stepfunctions_1.Condition.stringMatches('$.Prefix', `${constants_1.STORAGE_KEY_PREFIX}@*`), processNamespace)
            .otherwise(new tasks.StepFunctionsStartExecution(this, 'Process unscoped package', {
            stateMachine: processPackageVersions,
            input: aws_stepfunctions_1.TaskInput.fromObject({
                // Associate the child workflow with the execution that started it,
                AWS_STEP_FUNCTIONS_STARTED_BY_EXECUTION_ID: aws_stepfunctions_1.JsonPath.stringAt('$$.Execution.Id'),
                Prefix: aws_stepfunctions_1.JsonPath.stringAt('$.Prefix'),
            }),
            integrationPattern: aws_stepfunctions_1.IntegrationPattern.RUN_JOB,
        }))
            .afterwards()));
        start.next(new aws_stepfunctions_1.Choice(this, 'Has more prefixes?')
            .when(aws_stepfunctions_1.Condition.isPresent('$.response.NextContinuationToken'), start)
            .otherwise(new aws_stepfunctions_1.Succeed(this, 'Done')));
        this.stateMachine = new aws_stepfunctions_1.StateMachine(this, 'Resource', {
            definition: start,
            stateMachineName: stateMachineNameFrom(this.node.path),
            timeout: core_1.Duration.hours(4),
            tracingEnabled: true,
        });
        props.bucket.grantRead(processPackageVersions);
        props.bucket.grantRead(this.stateMachine);
    }
}
/**
 * This turns a node path into a valid state machine name, to try and improve
 * the StepFunction's AWS console experience while minimizing the risk for
 * collisons.
 */
function stateMachineNameFrom(nodePath) {
    // Poor man's replace all...
    return nodePath.split(/[^a-z0-9+!@.()=_'-]+/i).join('.');
}
//# sourceMappingURL=data:application/json;base64,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