"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Ingestion = void 0;
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_lambda_event_sources_1 = require("@aws-cdk/aws-lambda-event-sources");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const aws_sqs_1 = require("@aws-cdk/aws-sqs");
const aws_stepfunctions_1 = require("@aws-cdk/aws-stepfunctions");
const aws_stepfunctions_tasks_1 = require("@aws-cdk/aws-stepfunctions-tasks");
const core_1 = require("@aws-cdk/core");
const deep_link_1 = require("../../deep-link");
const runbook_url_1 = require("../../runbook-url");
const constants_1 = require("../shared/constants");
const constants_2 = require("./constants");
const ingestion_1 = require("./ingestion");
const re_ingest_1 = require("./re-ingest");
/**
 * The ingestion function receives messages from discovery integrations and
 * processes their payloads into the provided S3 Bucket.
 *
 * This function is also an `IGrantable`, so that it can be granted permissions
 * to read from the source S3 buckets.
 */
class Ingestion extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b, _c;
        super(scope, id);
        this.queueRetentionPeriod = core_1.Duration.days(14);
        this.deadLetterQueue = new aws_sqs_1.Queue(this, 'DLQ', {
            encryption: aws_sqs_1.QueueEncryption.KMS_MANAGED,
            retentionPeriod: this.queueRetentionPeriod,
            visibilityTimeout: core_1.Duration.minutes(15),
        });
        this.queue = new aws_sqs_1.Queue(this, 'Queue', {
            deadLetterQueue: {
                maxReceiveCount: 5,
                queue: this.deadLetterQueue,
            },
            encryption: aws_sqs_1.QueueEncryption.KMS_MANAGED,
            retentionPeriod: this.queueRetentionPeriod,
            visibilityTimeout: core_1.Duration.minutes(15),
        });
        const handler = new ingestion_1.Ingestion(this, 'Default', {
            description: '[ConstructHub/Ingestion] Ingests new package versions into the Construct Hub',
            environment: {
                AWS_EMF_ENVIRONMENT: 'Local',
                BUCKET_NAME: props.bucket.bucketName,
                PACKAGE_LINKS: JSON.stringify((_a = props.packageLinks) !== null && _a !== void 0 ? _a : []),
                PACKAGE_TAGS: JSON.stringify((_b = props.packageTags) !== null && _b !== void 0 ? _b : []),
                STATE_MACHINE_ARN: props.orchestration.stateMachine.stateMachineArn,
            },
            logRetention: (_c = props.logRetention) !== null && _c !== void 0 ? _c : aws_logs_1.RetentionDays.TEN_YEARS,
            memorySize: 10240,
            timeout: core_1.Duration.minutes(15),
            tracing: aws_lambda_1.Tracing.ACTIVE,
        });
        this.function = handler;
        props.bucket.grantWrite(this.function);
        props.orchestration.stateMachine.grantStartExecution(this.function);
        this.function.addEventSource(new aws_lambda_event_sources_1.SqsEventSource(this.queue, { batchSize: 1 }));
        // This event source is disabled, and can be used to re-process dead-letter-queue messages
        this.function.addEventSource(new aws_lambda_event_sources_1.SqsEventSource(this.deadLetterQueue, { batchSize: 1, enabled: false }));
        // Reprocess workflow
        const reprocessQueue = new aws_sqs_1.Queue(this, 'ReprocessQueue', {
            deadLetterQueue: {
                maxReceiveCount: 5,
                queue: this.deadLetterQueue,
            },
            encryption: aws_sqs_1.QueueEncryption.KMS_MANAGED,
            retentionPeriod: this.queueRetentionPeriod,
            // Visibility timeout of 15 minutes matches the Lambda maximum execution time.
            visibilityTimeout: core_1.Duration.minutes(15),
        });
        props.bucket.grantRead(this.function, `${constants_1.STORAGE_KEY_PREFIX}*${constants_1.PACKAGE_KEY_SUFFIX}`);
        this.function.addEventSource(new aws_lambda_event_sources_1.SqsEventSource(reprocessQueue, { batchSize: 1 }));
        new ReprocessIngestionWorkflow(this, 'ReprocessWorkflow', { bucket: props.bucket, queue: reprocessQueue });
        this.grantPrincipal = this.function.grantPrincipal;
        props.monitoring.addHighSeverityAlarm('Ingestion Dead-Letter Queue not empty', new aws_cloudwatch_1.MathExpression({
            expression: 'm1 + m2',
            usingMetrics: {
                m1: this.deadLetterQueue.metricApproximateNumberOfMessagesVisible({ period: core_1.Duration.minutes(1) }),
                m2: this.deadLetterQueue.metricApproximateNumberOfMessagesNotVisible({ period: core_1.Duration.minutes(1) }),
            },
        }).createAlarm(this, 'DLQAlarm', {
            alarmName: `${this.node.path}/DLQNotEmpty`,
            alarmDescription: [
                'The dead-letter queue for the Ingestion function is not empty!',
                '',
                `RunBook: ${runbook_url_1.RUNBOOK_URL}`,
                '',
                `Direct link to the queue: ${deep_link_1.sqsQueueUrl(this.deadLetterQueue)}`,
                `Direct link to the function: ${deep_link_1.lambdaFunctionUrl(this.function)}`,
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 1,
            // SQS does not emit metrics if the queue has been empty for a while, which is GOOD.
            treatMissingData: aws_cloudwatch_1.TreatMissingData.NOT_BREACHING,
        }));
        props.monitoring.addHighSeverityAlarm('Ingestion failures', this.function.metricErrors().createAlarm(this, 'FailureAlarm', {
            alarmName: `${this.node.path}/Failure`,
            alarmDescription: [
                'The Ingestion function is failing!',
                '',
                `RunBook: ${runbook_url_1.RUNBOOK_URL}`,
                '',
                `Direct link to the function: ${deep_link_1.lambdaFunctionUrl(this.function)}`,
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 2,
            threshold: 1,
            // Lambda only emits metrics when the function is invoked. No invokation => no errors.
            treatMissingData: aws_cloudwatch_1.TreatMissingData.NOT_BREACHING,
        }));
    }
    metricFoundLicenseFile(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "FoundLicenseFile" /* FOUND_LICENSE_FILE */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    metricIneligibleLicense(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "IneligibleLicense" /* INELIGIBLE_LICENSE */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    metricInvalidAssembly(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "InvalidAssembly" /* INVALID_ASSEMBLY */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    metricInvalidTarball(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "InvalidTarball" /* INVALID_TARBALL */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    /**
     * This metrics is the total count of packages that were rejected due to
     * mismatched identity (name, version, license) between the `package.json`
     * file and te `.jsii` attribute.
     */
    metricMismatchedIdentityRejections(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "MismatchedIdentityRejections" /* MISMATCHED_IDENTITY_REJECTIONS */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
}
exports.Ingestion = Ingestion;
/**
 * A StepFunctions State Machine to reprocess every currently indexed package
 * through the ingestion function. This should not be frequently required, but
 * can come in handy at times.
 *
 * For more information, refer to the runbook at
 * https://github.com/cdklabs/construct-hub/blob/main/docs/operator-runbook.md
 */
class ReprocessIngestionWorkflow extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const lambdaFunction = new re_ingest_1.ReIngest(this, 'Function', {
            description: '[ConstructHub/Ingestion/ReIngest] The function used to reprocess packages through ingestion',
            environment: { BUCKET_NAME: props.bucket.bucketName, QUEUE_URL: props.queue.queueUrl },
            tracing: aws_lambda_1.Tracing.ACTIVE,
        });
        props.queue.grantSendMessages(lambdaFunction);
        props.bucket.grantRead(lambdaFunction, `${constants_1.STORAGE_KEY_PREFIX}*${constants_1.METADATA_KEY_SUFFIX}`);
        props.bucket.grantRead(lambdaFunction, `${constants_1.STORAGE_KEY_PREFIX}*${constants_1.PACKAGE_KEY_SUFFIX}`);
        const listBucket = new aws_stepfunctions_1.Choice(this, 'Has a NextContinuationToken?')
            .when(aws_stepfunctions_1.Condition.isPresent('$.response.NextContinuationToken'), new aws_stepfunctions_tasks_1.CallAwsService(this, 'S3.ListObjectsV2(NextPage)', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                ContinuationToken: aws_stepfunctions_1.JsonPath.stringAt('$.response.NextContinuationToken'),
                MaxKeys: 250,
                Prefix: constants_1.STORAGE_KEY_PREFIX,
            },
            resultPath: '$.response',
        }))
            .otherwise(new aws_stepfunctions_tasks_1.CallAwsService(this, 'S3.ListObjectsV2(FirstPage)', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                Prefix: constants_1.STORAGE_KEY_PREFIX,
            },
            resultPath: '$.response',
        })).afterwards();
        const process = new aws_stepfunctions_1.Map(this, 'Process Result', {
            itemsPath: '$.response.Contents',
            resultPath: aws_stepfunctions_1.JsonPath.DISCARD,
        }).iterator(new aws_stepfunctions_1.Choice(this, 'Is metadata object?')
            .when(aws_stepfunctions_1.Condition.stringMatches('$.Key', `*${constants_1.METADATA_KEY_SUFFIX}`), new aws_stepfunctions_tasks_1.LambdaInvoke(this, 'Send for reprocessing', { lambdaFunction })
            // Ample retries here... We should never fail because of throttling....
            .addRetry({ errors: ['Lambda.TooManyRequestsException'], backoffRate: 1.1, interval: core_1.Duration.minutes(1), maxAttempts: 30 }))
            .otherwise(new aws_stepfunctions_1.Succeed(this, 'Nothing to do')));
        listBucket.next(process.next(new aws_stepfunctions_1.Choice(this, 'Is there more?')
            .when(aws_stepfunctions_1.Condition.isPresent('$.response.NextContinuationToken'), listBucket)
            .otherwise(new aws_stepfunctions_1.Succeed(this, 'All Done'))));
        const stateMachine = new aws_stepfunctions_1.StateMachine(this, 'StateMachine', {
            definition: listBucket,
            timeout: core_1.Duration.hours(1),
        });
        props.bucket.grantRead(stateMachine);
        props.queue.grantSendMessages(stateMachine);
    }
}
//# sourceMappingURL=data:application/json;base64,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