"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const zlib_1 = require("zlib");
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
const semver_1 = require("semver");
const tar_stream_1 = require("tar-stream");
const client_lambda_shared_1 = require("../deny-list/client.lambda-shared");
const aws = require("../shared/aws.lambda-shared");
const constants = require("../shared/constants");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
const constants_1 = require("./constants");
aws_embedded_metrics_1.Configuration.namespace = constants_1.METRICS_NAMESPACE;
/**
 * Regenerates the `catalog.json` object in the configured S3 bucket.
 *
 * @param event configuration for the rebuild job. In particular, the `rebuild`
 *              property can be set to `true` in order to trigger a full (i.e:
 *              non-incremental) rebuild of the object.
 * @param context the lambda context in which this execution runs.
 *
 * @returns the information about the updated S3 object.
 */
async function handler(event, context) {
    console.log(JSON.stringify(event, null, 2));
    const BUCKET_NAME = env_lambda_shared_1.requireEnv('BUCKET_NAME');
    const packages = new Map();
    const denyList = await client_lambda_shared_1.DenyListClient.newClient();
    console.log('Loading existing catalog (if present)...');
    const data = await aws.s3().getObject({ Bucket: BUCKET_NAME, Key: constants.CATALOG_KEY }).promise()
        .catch((err) => err.code !== 'NoSuchKey'
        ? Promise.reject(err)
        : Promise.resolve({ /* no data */}));
    if (data.Body) {
        console.log('Catalog found. Loading...');
        const catalog = JSON.parse(data.Body.toString('utf-8'));
        for (const info of catalog.packages) {
            if (!packages.has(info.name)) {
                packages.set(info.name, new Map());
            }
            packages.get(info.name).set(info.major, info);
        }
    }
    // If defined, the function will invoke itself again to resume the work from that key (this
    // happens only in "from scratch" or "rebuild" cases).
    let nextStartAfter;
    if (event.package) {
        if (!event.package.key.endsWith(constants.PACKAGE_KEY_SUFFIX)) {
            throw new Error(`The provided package key is invalid: ${event.package.key} does not end in ${constants.PACKAGE_KEY_SUFFIX}`);
        }
        console.log('Registering new packages...');
        // note that we intentionally don't catch errors here to let these
        // event go to the DLQ for manual inspection.
        await appendPackage(packages, event.package.key, BUCKET_NAME, denyList);
    }
    // If we don't have a package in event, then we're refreshing the catalog. This is also true if we
    // don't have a catalog body (from scratch) or if "startAfter" is set (continuation of from
    // scratch).
    if (!(event === null || event === void 0 ? void 0 : event.package) || !data.Body || event.startAfter) {
        console.log('Recreating or refreshing catalog...');
        const failures = {};
        for await (const { Key: pkgKey } of relevantObjects(BUCKET_NAME, event.startAfter)) {
            try {
                await appendPackage(packages, pkgKey, BUCKET_NAME, denyList);
            }
            catch (e) {
                failures[pkgKey] = e;
            }
            // If we're getting short on time (1 minute out of 15 left), we'll be continuing in a new
            // invocation after writing what we've done so far to S3...
            if (context.getRemainingTimeInMillis() <= 60000) {
                nextStartAfter = pkgKey;
                break;
            }
        }
        for (const [key, error] of Object.entries(failures)) {
            console.log(`Failed processing ${key}: ${error}`);
        }
        await aws_embedded_metrics_1.metricScope((metrics) => async () => {
            metrics.setDimensions();
            const failedCount = Object.keys(failures).length;
            console.log(`Marking ${failedCount} failed packages`);
            metrics.putMetric("FailedPackagesOnRecreation" /* FAILED_PACKAGES_ON_RECREATION */, failedCount, aws_embedded_metrics_1.Unit.Count);
        })();
    }
    // Build the final data package...
    console.log('Consolidating catalog...');
    const catalog = { packages: new Array(), updated: new Date().toISOString() };
    for (const majors of packages.values()) {
        for (const pkg of majors.values()) {
            catalog.packages.push(pkg);
        }
    }
    console.log(`There are now ${catalog.packages.length} registered package major versions`);
    await aws_embedded_metrics_1.metricScope((metrics) => async () => {
        metrics.setDimensions();
        metrics.putMetric("RegisteredPackagesMajorVersion" /* REGISTERED_PACKAGES_MAJOR_VERSION */, catalog.packages.length, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("MissingConstructFrameworkCount" /* MISSING_CONSTRUCT_FRAMEWORK_COUNT */, catalog.packages.filter((pkg) => pkg.constructFramework == null).length, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("MissingConstructFrameworkVersionCount" /* MISSING_CONSTRUCT_FRAMEWORK_VERSION_COUNT */, catalog.packages.filter((pkg) => pkg.constructFramework && pkg.constructFramework.majorVersion == null).length, aws_embedded_metrics_1.Unit.Count);
    })();
    // Clean up existing entries if necessary. In particular, remove the license texts as they make
    // the catalog unnecessarily large, and may hinder some search queries' result quality.
    for (const entry of catalog.packages) {
        if (entry.metadata) {
            delete entry.metadata.licenseText;
        }
    }
    // Upload the result to S3 and exit.
    const result = await aws.s3().putObject({
        Bucket: BUCKET_NAME,
        Key: constants.CATALOG_KEY,
        Body: JSON.stringify(catalog, null, 2),
        ContentType: 'application/json',
        CacheControl: 'public, max-age=300, must-revalidate, proxy-revalidate',
        Metadata: {
            'Lambda-Log-Group': context.logGroupName,
            'Lambda-Log-Stream': context.logStreamName,
            'Lambda-Run-Id': context.awsRequestId,
            'Package-Count': `${catalog.packages.length}`,
        },
    }).promise();
    if (nextStartAfter != null) {
        console.log(`Will continue from ${nextStartAfter} in new invocation...`);
        const nextEvent = { ...event, startAfter: nextStartAfter };
        // We start it asynchronously, as this function has a provisionned
        // concurrency of 1 (so a synchronous attempt would always be throttled).
        await aws.lambda().invokeAsync({
            FunctionName: context.functionName,
            InvokeArgs: JSON.stringify(nextEvent, null, 2),
        }).promise();
    }
    return result;
}
exports.handler = handler;
/**
 * A generator that asynchronously traverses the set of "interesting" objects
 * found by listing the configured S3 bucket. Those objects correspond to all
 * npm package tarballs present under the `packages/` prefix in the bucket.
 *
 * @param bucket the bucket in which to list objects
 * @param startAfter the key to start reading from, if provided.
 */
async function* relevantObjects(bucket, startAfter) {
    var _a, _b;
    const request = {
        Bucket: bucket,
        Prefix: constants.STORAGE_KEY_PREFIX,
        StartAfter: startAfter,
    };
    do {
        const result = await aws.s3().listObjectsV2(request).promise();
        for (const object of (_a = result.Contents) !== null && _a !== void 0 ? _a : []) {
            if (!((_b = object.Key) === null || _b === void 0 ? void 0 : _b.endsWith(constants.PACKAGE_KEY_SUFFIX))) {
                continue;
            }
            // We only register packages if they have AT LEAST docs in one language.
            const tsDocs = `${object.Key.substring(0, object.Key.length - constants.PACKAGE_KEY_SUFFIX.length)}${constants.DOCS_KEY_SUFFIX_TYPESCRIPT}`;
            const pyDocs = `${object.Key.substring(0, object.Key.length - constants.PACKAGE_KEY_SUFFIX.length)}${constants.DOCS_KEY_SUFFIX_PYTHON}`;
            const javaDocs = `${object.Key.substring(0, object.Key.length - constants.PACKAGE_KEY_SUFFIX.length)}${constants.DOCS_KEY_SUFFIX_JAVA}`;
            const csharpDocs = `${object.Key.substring(0, object.Key.length - constants.PACKAGE_KEY_SUFFIX.length)}${constants.DOCS_KEY_SUFFIX_CSHARP}`;
            if (!(await aws.s3ObjectExists(bucket, tsDocs)) &&
                !(await aws.s3ObjectExists(bucket, pyDocs)) &&
                !(await aws.s3ObjectExists(bucket, javaDocs)) &&
                !(await aws.s3ObjectExists(bucket, csharpDocs))) {
                continue;
            }
            yield object;
        }
        request.ContinuationToken = result.NextContinuationToken;
    } while (request.ContinuationToken != null);
}
async function appendPackage(packages, pkgKey, bucketName, denyList) {
    var _a, _b, _c;
    console.log(`Processing key: ${pkgKey}`);
    const [, packageName, versionStr] = constants.STORAGE_KEY_FORMAT_REGEX.exec(pkgKey);
    const version = new semver_1.SemVer(versionStr);
    const found = (_a = packages.get(packageName)) === null || _a === void 0 ? void 0 : _a.get(version.major);
    // If the version is === to the current latest, we'll be replacing that (so re-generated metadata are taken into account)
    if (found != null && version.compare(found.version) < 0) {
        console.log(`Skipping ${packageName}@${version} because it is not newer than the existing ${found.version}`);
        return;
    }
    console.log(`Checking if ${packageName}@${version.version} matches a deny list rule`);
    const blocked = denyList.lookup(packageName, version.version);
    if (blocked) {
        console.log(`Skipping ${packageName}@${version.version} because it is blocked by the deny list rule: ${JSON.stringify(blocked)}`);
        return;
    }
    console.log(`Registering ${packageName}@${version}`);
    // Donwload the tarball to inspect the `package.json` data therein.
    const pkg = await aws.s3().getObject({ Bucket: bucketName, Key: pkgKey }).promise();
    const metadataKey = pkgKey.replace(constants.PACKAGE_KEY_SUFFIX, constants.METADATA_KEY_SUFFIX);
    const metadataResponse = await aws.s3().getObject({ Bucket: bucketName, Key: metadataKey }).promise();
    const manifest = await new Promise((ok, ko) => {
        zlib_1.gunzip(Buffer.from(pkg.Body), (err, tar) => {
            if (err) {
                return ko(err);
            }
            tar_stream_1.extract()
                .on('entry', (header, stream, next) => {
                if (header.name !== 'package/package.json') {
                    // Not the file we are looking for, skip ahead (next run-loop tick).
                    return setImmediate(next);
                }
                const chunks = new Array();
                return stream
                    .on('data', (chunk) => chunks.push(Buffer.from(chunk)))
                    .once('end', () => {
                    ok(Buffer.concat(chunks));
                    next();
                })
                    .resume();
            })
                .once('finish', () => {
                ko(new Error('Could not find package/package.json in tarball!'));
            })
                .write(tar, (writeErr) => {
                if (writeErr) {
                    ko(writeErr);
                }
            });
        });
    });
    // Add the PackageInfo into the working set
    const pkgMetadata = JSON.parse(manifest.toString('utf-8'));
    const npmMetadata = JSON.parse((_c = (_b = metadataResponse === null || metadataResponse === void 0 ? void 0 : metadataResponse.Body) === null || _b === void 0 ? void 0 : _b.toString('utf-8')) !== null && _c !== void 0 ? _c : '{}');
    const major = new semver_1.SemVer(pkgMetadata.version).major;
    if (!packages.has(pkgMetadata.name)) {
        packages.set(pkgMetadata.name, new Map());
    }
    packages.get(pkgMetadata.name).set(major, {
        author: pkgMetadata.author,
        description: pkgMetadata.description,
        keywords: pkgMetadata.keywords,
        languages: pkgMetadata.jsii.targets,
        license: pkgMetadata.license,
        major,
        metadata: npmMetadata,
        name: pkgMetadata.name,
        version: pkgMetadata.version,
    });
}
//# sourceMappingURL=data:application/json;base64,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