# Nautical ![build workflow](https://github.com/barbacbd/nautical/actions/workflows/python-app.yml/badge.svg) [![made-with-python](https://img.shields.io/badge/Made%20with-Python-1f425f.svg)](https://www.python.org/) [![PyPI version fury.io](https://badge.fury.io/py/nautical.svg)](https://pypi.python.org/pypi/nautical/)

[![Maintenance](https://img.shields.io/badge/Maintained%3F-yes-green.svg)](https://github.com/barbacbd/nautical/pulse/commit-activity)
[![GitHub latest commit](https://badgen.net/github/last-commit/barbacbd/nautical)](https://github.com/barbacbd/nautical/commit/)
[![PyPi license](https://badgen.net/pypi/license/pip/)](https://pypi.com/project/pip/)


[![Linux](https://svgshare.com/i/Zhy.svg)](https://svgshare.com/i/Zhy.svg)
[![macOS](https://svgshare.com/i/ZjP.svg)](https://svgshare.com/i/ZjP.svg)
[![Windows](https://svgshare.com/i/ZhY.svg)](https://svgshare.com/i/ZhY.svg)


## Description

A python based web scraper to grab the buoy data from [NOAA](https://www.ndbc.noaa.gov/). The scraper utilizes kml parsing
and BeautifulSoup to parse through data found online. NOAA is very kind in the fact that they allow the lookup of Buoy
data very easily using the same url with the id of the buoy at the end of the url. We can grab all of the buoy ids, append
the id to the url, and get several tables of output from the url. All of the data stored in the tables is updated in 30 minute
increments.

For more information about the package please see the [documentation](https://barbacbd.github.io/nautical/html/index.html).

## Structure

### [error](./error) 

The module contains the custom exception used in this package.

```python
class NauticalError(Exception):
```

In the event of an error generated by this package, the user should except `NauticalError`.

### [io](./io) 

The module contains the majority of the code where the user can read all buoy data, parse kml, parse beautiful soup 
html output, and grab some of the specific data from the tables that we are looking for.

### [location](./location)
The module contains a 3D Point class that can be used to store locations as well as determine distance
to and from other points.

The module also provides the user with a simple distance function to provide distance between two points and
a function to determine if the point is located within a specified area. 

### [noaa](./noaa/buoy) 
The module contains classes to store/utilize/manipulate swell and wave data read in from NOAA's website. 
There are also some extra functions such as getting the sea state based on the current wave height.

### [sea_state](./sea_state)
Module to allow the user to estimate the current sea state based on wave heights.

|Sea State|Wave Height (meters)|
|---------|----------------|
|0|0 - 0|
|1|0 - 0.1|
|2|0.1 - 0.5|
|3|0.5 - 1.25|
|4|1.25 - 2.5|
|5|2.5 - 4.0|
|6|4.0 - 6.0|
|7|6.0 - 9.0|
|8|9.0 - 14.0|
|9|14.0+|


### [tests](./tests)
The module contains the unit tests for the Nautical package.

The module contains the link to the executable to run all tests for the package with 
 the executable named `NauticalTests`.

### [time](./time) 
Time module to parse and store the time as it is represented from NOAA's webpages.

### [units](./units) 
Utility package to provide the user with the easy ability to alter the units for the data.

The following display the supported unit types for each category:

|Time   |Temperature|Speed|Distance|
|-------|----------|-------------------|--------------|
|Seconds|Fahrenheit|knots              |Centimeters   |
|Minutes|Celsius   |Meters per second  |Feet          |
|Hours  |          |Miles per hour     |Yards         |
|Days   |          |Kilometers per hour|Meters        |
|       |          |Feet per second    |Kilometers    |
|       |          |                   |Miles         |
|       |          |                   |Nautical Miles|

The user can convert values to a different unit if and only if the units are in the same class. 

```python
from nautical.units.conversion import convert

def convert(value, init_units, final_units):
    """
    Convert the value given the current units to the new units. If the
    units are not in the same set of units then the value cannot be converted,
    and None will be returned.
    """
```

## Examples


### Sources

NOAA categorizes all buoys by a source. We can obtain all sources and the buoys the are cateogrized with each
source with one function.

```python
from nautical.io.sources import get_buoy_sources

sources = get_buoy_sources()
```

If this action was successful, sources will be a dictionary mapping the name of the source to the 
[source object](./nautical/noaa/buoy/source.py).  

We can also obtain the information about each buoy contained in the source.

```python
for _, source in sources.items():
    print(source)

    for buoy in source:
        print("\t{}".format(str(buoy)))
```

### Buoy Location

In the [previous example](###Sources) we were able to find all sources and their respective buoys. If we want to
search through this list to find a buoy near or at a location we can.

```python
for _, source in sources.items():
    for buoy in source
        location = buoy.location

        if location:
            # determine if the location meets criteria
```

### Buoy Information

In the [previous examples](###Sources) we were able to find all of the buoys that NOAA provides updated information about. If the
user finds a buoy of interest. If we want to retrieve all of the information for a particular buoy including present and 
past recordings, we can utilize the tools below. 

```python
from nautical.io import create_buoy

buoy_id = "example_buoy_id"

buoy = create_buoy(buoy_id)
```

This will return a `nautical.noaa.buoy.buoy.Buoy` object. 

_The `past` variable of Buoy objects are considered deprecated in versions >2.3.0._

# NOAA National Centers for Environmental Information

The module was added to version 2.3.0 (nautical.noaa.ncei). The module adds an interface to the API that NOAA created for environmental information. The API information can be found [here](https://www.ncdc.noaa.gov/cdo-web/webservices/v2#gettingStarted).

The module provides a default token that was generated by the email associated with this account, but a new token can be generated [here](https://www.ncdc.noaa.gov/cdo-web/token). The token is used for all authentication when querying the API. Each token will be limitted to 5 requests per second and 10,000 requests per day.

## Endpoints

- [Datasets](https://www.ncdc.noaa.gov/cdo-web/webservices/v2#datasets)
- [Data Categories](https://www.ncdc.noaa.gov/cdo-web/webservices/v2#dataCategories)
- [Data Types](https://www.ncdc.noaa.gov/cdo-web/webservices/v2#dataTypes)
- [Location Categories](https://www.ncdc.noaa.gov/cdo-web/webservices/v2#locationCategories)
- [Locations](https://www.ncdc.noaa.gov/cdo-web/webservices/v2#locations)
- [Stations](https://www.ncdc.noaa.gov/cdo-web/webservices/v2#stations)
- [Data](https://www.ncdc.noaa.gov/cdo-web/webservices/v2#data)

## Parameters

The module contains a class called `Parameter` which is essnetially a tuple. The `Parameters` are used to create more complex queries when used with the `query_all` function.

## Extension

As endpoints change, the user can alter the extension via the classes that extend `NCEIBase`. These classes define the allowed parameters (when querying the endpoint), the endpoint, and the variables that are returned from the queries.

As endpoints are added, the user can add new classes that extend the `NCEIBase` class.
