# AUTOGENERATED! DO NOT EDIT! File to edit: nbs/01_tabular.interpretation.ipynb (unless otherwise specified).

__all__ = ['base_error']

# Cell
from fastai2.tabular.all import *
from scipy.cluster import hierarchy as hc
from sklearn import manifold

# Cell
def base_error(err, val): return (err/val)/err

# Cell
@patch
def feature_importance(x:TabularLearner, df=None, dl=None, perm_func=base_error, metric=accuracy, bs=None, reverse=True, plot=True):
    "Calculate and plot the Feature Importance based on `df`"
    x.df = df
    bs = bs if bs is not None else x.dls.bs
    if df is not None:
        dl = x.dls.test_dl(df, bs=bs)
    else:
        dl = x.dls[1]
    x_names = x.dls.x_names.filter(lambda x: '_na' not in x)
    na = x.dls.x_names.filter(lambda x: '_na' in x)
    y = x.dls.y_names
    orig_metrics = x.metrics[1:]
    x.metrics = [metric]
    results = _calc_feat_importance(x, dl, x_names, na, perm_func, reverse)
    if plot:
        _plot_importance(_ord_dic_to_df(results))
    x.metrics = orig_metrics
    return results

# Cell
def _measure_col(learn:TabularLearner, dl:TabDataLoader, name:str, na:list):
    "Measures change after column permutation"
    col = [name]
    if f'{name}_na' in na: col.append(name)
    orig = dl.items[col].values
    perm = np.random.permutation(len(orig))
    dl.items[col] = dl.items[col].values[perm]
    with learn.no_bar(), learn.no_logging():
        metric = learn.validate(dl=dl)[1]
    dl.items[col] = orig
    return metric

# Cell
def _calc_feat_importance(learn:TabularLearner, dl:TabDataLoader, x_names:list, na:list, perm_func=base_error, reverse=True):
    "Calculates permutation importance by shuffling a column by `perm_func`"
    with learn.no_bar(), learn.no_logging():
        base_error = learn.validate(dl=dl)[1]
    importance = {}
    pbar = progress_bar(x_names)
    print("Calculating Permutation Importance")
    for col in pbar:
        importance[col] = _measure_col(learn, dl, col, na)
    for key, value in importance.items():
        importance[key] = perm_func(base_error, value)
    return OrderedDict(sorted(importance.items(), key=lambda kv: kv[1], reverse=True))

# Cell
def _ord_dic_to_df(dict:OrderedDict): return pd.DataFrame([[k,v] for k,v in dict.items()], columns=['feature','importance'])

# Cell
def _plot_importance(df:pd.DataFrame, limit=20, asc=False, **kwargs):
    "Plot importance with an optional limit to how many variables shown"
    df_copy = df.copy()
    df_copy['feature'] = df_copy['feature'].str.slice(0,25)
    df_copy = df_copy.sort_values(by='importance', ascending=asc)[:limit].sort_values(by='importance', ascending=not(asc))
    ax = df_copy.plot.barh(x='feature', y='importance', sort_columns=True, **kwargs)
    for p in ax.patches:
        ax.annotate(f'{p.get_width():.4f}', ((p.get_width() * 1.005), p.get_y()  * 1.005))

# Cell
def _get_top_corr(df, matrix, thresh:float=0.8):
    corr = np.where(abs(matrix) < thresh,0,matrix)
    idxs = []
    for i in range(corr.shape[0]):
        if (corr[i,:].sum() + corr[:, i].sum() > 2):
            idxs.append(i)
    cols = df.columns[idxs]
    return pd.DataFrame(corr[np.ix_(idxs,idxs)], columns=cols, index=cols)

# Cell
def _cramers_corrected_stat(cm):
    "Calculates Cramers V Statistic for categorical-categorical"
    try: chi2 = scipy.stats.chi2_contingency(cm)[0]
    except: return 0.0

    if chi2 == 0: return 0.0
    n = cm.sum().sum()
    phi2 = chi2 / n
    r,k = cm.shape
    phi2corr = max(0, phi2 - ((k-1)*(r-1))/(n-1))
    rcorr = r - ((r-1)**2)/(n-1)
    kcorr = k - ((k-1)**2)/(n-1)
    return np.sqrt(phi2corr/min((kcorr-1), (rcorr-1)))

# Cell
def _get_cramer_v_matr(dl:TabDataLoader):
    "Calculate Cramers V statistic on every pair in `df`'s columns'"
    df = dl.xs
    cols = list(df.columns)
    corrM = np.zeros((len(cols), len(cols)))
    for col1, col2 in progress_bar(list(itertools.combinations(cols, 2))):
        idx1, idx2 = cols.index(col1), cols.index(col2)
        corrM[idx1,idx2] = _cramers_corrected_stat(pd.crosstab(df[col1], df[col2]))
        corrM[idx2, idx1] = corrM[idx1, idx2]
    np.fill_diagonal(corrM, 1.0)
    return corrM

# Cell
def _get_top_corr_dict_corrs(top_corrs):
    cols = top_corrs.columns
    top_corrs_np = top_corrs.to_numpy()
    corr_dict = {}
    for i in range(top_corrs_np.shape[0]):
        for j in range(i+1, top_corrs_np.shape[0]):
            if top_corrs_np[i,j] > 0:
                corr_dict[cols[i] + ' vs ' + cols[j]] = np.round(top_corrs_np[i,j],3)
    return OrderedDict(sorted(corr_dict.items(), key=lambda kv: abs(kv[1]), reverse=True))

# Cell
@patch
def get_top_corr_dict(x:TabularLearner, df:pd.DataFrame, thresh:float=0.8):
    "Grabs top pairs of correlation with a given correlation matrix on `df` filtered by `thresh`"
    dl = x.dls.test_dl(df)
    matrix = _get_cramer_v_matr(dl)
    top_corrs = _get_top_corr(df, matrix, thresh=thresh)
    return _get_top_corr_dict_corrs(top_corrs)

# Cell
@patch
def plot_dendrogram(x:TabularLearner, df:pd.DataFrame, figsize=None, leaf_font_size=16):
    "Plots dendrogram for a calculated correlation matrix"
    dl = x.dls.test_dl(df)
    matrix = _get_cramer_v_matr(dl)
    if figsize is None:
        figsize = (15, 0.02*leaf_font_size*len(dl.x_names))
    corr_condensed = hc.distance.squareform(1-matrix)
    z = hc.linkage(corr_condensed, method='average')
    fig = plt.figure(figsize=figsize)
    dendrogram = hc.dendrogram(z, labels=dl.x_names, orientation='left', leaf_font_size = leaf_font_size)
    plt.show()