# -*- coding: utf-8 -*-
from setuptools import setup

modules = \
['trycast']
setup_kwargs = {
    'name': 'trycast',
    'version': '0.2.0',
    'description': '',
    'long_description': "# trycast\n\nThis module provides a single function `trycast` which can be used to parse a\nJSON-like value.\n\nHere is an example of parsing a `Point2D` object defined as a `TypedDict`:\n\n```python\nfrom bottle import HTTPResponse, request, route\nfrom trycast import trycast\nfrom typing import TypedDict\n\nclass Point2D(TypedDict):\n    x: float\n    y: float\n    name: str\n\n@route('/draw_point')\ndef draw_point_endpoint() -> None:\n    request_json = request.json  # type: object\n    if (point := trycast(Point2D, request_json)) is not None:\n        draw_point(point)  # type is narrowed to Point2D\n    else:\n        return HTTPResponse(status=400)  # Bad Request\n\ndef draw_point(point: Point2D) -> None:\n    # ...\n```\n\nIn this example the `trycast` function is asked to parse a `request_json`\ninto a `Point2D` object, returning the original object (with its type narrowed\nappropriately) if parsing was successful.\n\nMore complex types can be parsed as well, such as the `Shape` in the following\nexample, which is a tagged union that can be either a `Circle` or `Rect` value:\n\n```python\nfrom bottle import HTTPResponse, request, route\nfrom trycast import trycast\nfrom typing import Literal, TypedDict, Union\n\nclass Point2D(TypedDict):\n    x: float\n    y: float\n\nclass Circle(TypedDict):\n    type: Literal['circle']\n    center: Point2D  # a nested TypedDict!\n    radius: float\n\nclass Rect(TypedDict):\n    type: Literal['rect']\n    x: float\n    y: float\n    width: float\n    height: float\n\nShape = Union[Circle, Rect]  # a Tagged Union!\n\n@route('/draw_shape')\ndef draw_shape_endpoint() -> None:\n    request_json = request.json  # type: object\n    if (shape := trycast(Shape, request_json)) is not None:\n        draw_shape(shape)  # type is narrowed to Shape\n    else:\n        return HTTPResponse(status=400)  # Bad Request\n```\n\n> **Important:** Current limitations in the mypy typechecker require that you\n> add an extra `cast(Optional[Shape], ...)` around the call to `trycast`\n> in the example so that it is accepted by the typechecker without complaining.\n> These limitations are in the process of being resolved by\n> [introducing TypeForm support to mypy](https://github.com/python/mypy/issues/9773).\n\n\n## Recommendations when using trycast\n\n- So that `trycast()` can recognize TypedDicts with mixed required and\n  optional keys correctly:\n    * Use Python 3.9+ if possible.\n    * Prefer using `typing.TypedDict`, unless you must use Python 3.8.\n      In Python 3.8 prefer `typing_extensions.TypedDict` instead.\n    * Avoid using `mypy_extensions.TypedDict` in general.\n\n\n## Release Notes\n\n### Future\n\n* See the [Roadmap](https://github.com/davidfstr/trycast/wiki/Roadmap).\n\n### v0.2.0\n\n* TypedDict improvements & fixes:\n    * Fix `trycast()` to recognize TypedDicts from `mypy_extensions`.\n    * Extend `trycast()` to recognize TypedDicts that contain forward-references\n      to other types.\n        - Unfortunately there appears to be no easy way to support arbitrary kinds\n          of types that contain forward-references.\n        - In particular {Union, Optional} types and collection types (List, Dict)\n          with forward-references remain unsupported by `trycast()`.\n    * Recognize TypedDicts that have mixed required and optional keys correctly.\n        - Exception: Does not work for mypy_extensions.TypedDict or\n          Python 3.8's typing.TypedDict due to insufficient runtime\n          type annotation information.\n    * Fix recognition of a total=False TypedDict so that extra keys are disallowed.\n* Alter `typing_extensions` to be an optional dependency of `trycast`.\n\n### v0.1.0\n\n* Add support for Python 3.6, 3.7, and 3.9, in addition to 3.8.\n\n### v0.0.2\n\n* Fix README to appear on PyPI.\n* Add other package metadata, such as the supported Python versions.\n\n### v0.0.1\n\n* Initial release.\n* Supports typechecking all types found in JSON.\n",
    'author': 'David Foster',
    'author_email': 'david@dafoster.net',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/davidfstr/trycast',
    'py_modules': modules,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
