# Forked version of github.com/protobuf/protobuf/cmake protobuf-generate.cmake
# See the protobuf LICENSE for more details:
#
# Copyright 2008 Google Inc.  All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are
# met:
#
#    * Redistributions of source code must retain the above copyright
# notice, this list of conditions and the following disclaimer.
#     * Redistributions in binary form must reproduce the above
# copyright notice, this list of conditions and the following disclaimer
# in the documentation and/or other materials provided with the
# distribution.
#     * Neither the name of Google Inc. nor the names of its
# contributors may be used to endorse or promote products derived from
# this software without specific prior written permission.
#


# Collects import directories from a list of IMPORT_TARGETS, IMPORT_DIRS,
# and IMPORT_VARS, storing the result in OUT_VAR. The output list will
# be unique.
#
# btc_transitive_import_dirs(
#   OUT_VAR <variable name>
#     List of import directories returned to caller.
#   IMPORT_DIRS <list>
#     List of directories to be added to the OUT_VARS import list.
#   IMPORT_TARGETS <list>
#     List of targets. For existing targets the INTERFACE_INCLUDE_DIRECTORIES
#     property will be added to the OUT_VAR import list.
#   IMPORT_VARS <list>
#     List of variable names. For defined variables, contents will be added to
#     the OUT_VAR import list.
# )
#
function(btc_transitive_import_dirs)
  include(CMakeParseArguments)

  set(_singleargs OUT_VAR)
  set(_multiargs IMPORT_TARGETS IMPORT_VARS IMPORT_DIRS)

  cmake_parse_arguments(btc_protobuf "" "${_singleargs}" "${_multiargs}" "${ARGN}")

  if(NOT btc_protobuf_OUT_VAR)
    message(SEND_ERROR "Error: btc_transitive_import_dirs called without an OUT_VAR")
    return()
  endif()

  foreach(_tgt IN LISTS btc_protobuf_IMPORT_TARGETS)
    if(TARGET ${_tgt})
    get_property(_inc TARGET ${_tgt} PROPERTY INTERFACE_INCLUDE_DIRECTORIES)
    list(APPEND btc_protobuf_IMPORT_DIRS ${_inc})
    endif()
  endforeach()

  foreach(_var IN LISTS btc_protobuf_IMPORT_VARS)
    if(DEFINED ${_var})
    list(APPEND btc_protobuf_IMPORT_DIRS ${_var})
    endif()
  endforeach()

  list(REMOVE_DUPLICATES btc_protobuf_IMPORT_DIRS)

  set(${btc_protobuf_OUT_VAR} ${btc_protobuf_IMPORT_DIRS} PARENT_SCOPE)
endfunction()

# Invokes the protoc compiler on the .proto sources in a TARGET, adding the
# generated outputs to the TARGET along with the computed import directories.
#
# btc_protobuf(
#   TARGET <source>
#     Build .proto files included in the target.
#
#   PROTOC_OPTIONS  <options list>
#     Addional options passed to protoc
#
#   LANGUAGE  <language>
#     Generated language, such as cpp, upb, etc. Should match the plugin.
#   GENERATE_EXTENSIONS  <extension list>
#     Extensions generated by the protoc compiler for each file.
#     For example, ".pb.h" ".pb.cc" are the cpp extensions.
#   PLUGIN  <plugin executable> 
#     Plugin executable name [empty for cpp]. Typically something like:
#     protoc-gen-${LANGUAGE}=$<TARGET_FILE:lanugage_plugin_target>
#   PLUGIN_OPTIONS
#    Options passed to the plugin.
#   PROTOC_OUT_DIR  <directory>
#     Output directory for generated .c/.h files
#
#   IMPORT_DIRS <list>
#     List of directories added to the import flags.
#   IMPORT_TARGETS <list>
#     List of targets used to resolve imports. For existing targets the
#     INTERFACE_INCLUDE_DIRECTORIES will be added to the import flags.
#   IMPORT_VARS <list>
#     List of variable used to reslove imports. For defined variables,
#     contents will be added to the import flags.
#   APPEND_PATH <option>
#     Adds an import directory for each .proto.
#
#   DEPENDENCIES  <dependencies>
#    CMake dependencies.
# )
function(btc_protobuf)
  include(CMakeParseArguments)

  set(_options APPEND_PATH)
  set(_singleargs TARGET LANGUAGE PROTOC_OUT_DIR PLUGIN PLUGIN_OPTIONS DEPENDENCIES)
  set(_multiargs IMPORT_TARGETS IMPORT_VARS IMPORT_DIRS GENERATE_EXTENSIONS PROTOC_OPTIONS)

  cmake_parse_arguments(btc_protobuf "${_options}" "${_singleargs}" "${_multiargs}" "${ARGN}")

  if(NOT btc_protobuf_TARGET)
    message(SEND_ERROR "Error: btc_protobuf called without a TARGET")
    return()
  endif()

  if(NOT btc_protobuf_LANGUAGE)
    set(btc_protobuf_LANGUAGE cpp)
  endif()
  string(TOLOWER ${btc_protobuf_LANGUAGE} btc_protobuf_LANGUAGE)

  if(NOT btc_protobuf_GENERATE_EXTENSIONS)
    message(SEND_ERROR "Error: btc_protobuf called without GENERATE_EXTENSIONS for LANGUAGE ${btc_protobuf_LANGUAGE}")
    return()
  endif()

  if(NOT btc_protobuf_PROTOC_OUT_DIR)
    set(btc_protobuf_PROTOC_OUT_DIR ${CMAKE_CURRENT_BINARY_DIR})
  endif()

  foreach(_option ${btc_protobuf_PLUGIN_OPTIONS})
    # append comma - not using CMake lists and string replacement as users
    # might have semicolons in options
    if(_plugin_options)
      set( _plugin_options "${_plugin_options},")
    endif()
    set(_plugin_options "${_plugin_options}${_option}")
  endforeach()

  if(btc_protobuf_PLUGIN)
    set(_plugin "--plugin=${btc_protobuf_PLUGIN}")
  endif()

  get_target_property(_source_list ${btc_protobuf_TARGET} SOURCES)
  foreach(_file ${_source_list})
    if(_file MATCHES "proto$")
      list(APPEND btc_protobuf_PROTOS ${_file})
    endif()
  endforeach()

  if(NOT btc_protobuf_PROTOS)
    message(SEND_ERROR "Error: protobuf_generate could not find any .proto files")
    return()
  endif()

  if(btc_protobuf_APPEND_PATH)
    # Create an include path for each file specified
    foreach(_file ${btc_protobuf_PROTOS})
      get_filename_component(_abs_file ${_file} ABSOLUTE)
      get_filename_component(_abs_dir "${_abs_file}" DIRECTORY)
      list(FIND _protobuf_include_path "${_abs_dir}" _contains_already)
      if(${_contains_already} EQUAL -1)
          list(APPEND _protobuf_include_path "-I" "${_abs_dir}")
      endif()
    endforeach()
  endif()

  btc_transitive_import_dirs(
    OUT_VAR _protobuf_imports
    IMPORT_TARGETS ${btc_protobuf_IMPORT_TARGETS}
    IMPORT_VARS ${btc_protobuf_IMPORT_VARS}
    IMPORT_DIRS ${btc_protobuf_IMPORT_DIRS}
  )

  foreach(_dir ${_protobuf_imports})
    get_filename_component(_abs_path "${_dir}" ABSOLUTE)
    list(FIND _protobuf_include_path "${_abs_path}" _contains_already)
    if(${_contains_already} EQUAL -1)
        list(APPEND _protobuf_include_path "-I" "${_abs_path}")
    endif()
  endforeach()

  list(FIND _protobuf_imports "${btc_protobuf_PROTOC_OUT_DIR}" _contains_already)
  if(${_contains_already} EQUAL -1)
    list(APPEND _protobuf_imports  "${btc_protobuf_PROTOC_OUT_DIR}")
  endif()

  list(REMOVE_ITEM btc_protobuf_DEPENDENCIES protobuf::protoc)

  if(NOT _protobuf_include_path)
    set(_protobuf_include_path -I ${PROJECT_SOURCE_DIR} -I ${PROJECT_BINARY_DIR})
  endif()

  set(_generated_srcs_all)
  foreach(_proto ${btc_protobuf_PROTOS})
    get_filename_component(_abs_file ${_proto} ABSOLUTE)
    get_filename_component(_abs_dir ${_abs_file} DIRECTORY)

    get_filename_component(_file_full_name ${_proto} NAME)
    string(FIND "${_file_full_name}" "." _file_last_ext_pos REVERSE)
    string(SUBSTRING "${_file_full_name}" 0 ${_file_last_ext_pos} _basename)

    set(_suitable_include_found FALSE)
    foreach(DIR ${_protobuf_include_path})
      if(NOT DIR STREQUAL "-I")
        file(RELATIVE_PATH _rel_dir ${DIR} ${_abs_dir})
        string(FIND "${_rel_dir}" "../" _is_in_parent_folder)
        if (NOT ${_is_in_parent_folder} EQUAL 0)
          set(_suitable_include_found TRUE)
          break()
        endif()
      endif()
    endforeach()

    if(NOT _suitable_include_found)
      message(SEND_ERROR "Error: btc_protobuf could not find any correct proto include directory.")
      return()
    endif()

    set(_generated_srcs)
    foreach(_ext ${btc_protobuf_GENERATE_EXTENSIONS})
      list(APPEND _generated_srcs "${btc_protobuf_PROTOC_OUT_DIR}/${_rel_dir}/${_basename}${_ext}")
    endforeach()
    list(APPEND _generated_srcs_all ${_generated_srcs})

    set(_comment "Running ${btc_protobuf_LANGUAGE} protocol buffer compiler on ${_proto}")
    if(btc_protobuf_PROTOC_OPTIONS)
      set(_comment "${_comment}, protoc-options: ${btc_protobuf_PROTOC_OPTIONS}")
    endif()
    if(_plugin_options)
      set(_comment "${_comment}, plugin-options: ${_plugin_options}")
    endif()

    add_custom_command(
      OUTPUT ${_generated_srcs}
      COMMAND protobuf::protoc
      ARGS ${btc_protobuf_PROTOC_OPTIONS} --${btc_protobuf_LANGUAGE}_out ${_plugin_options}:${btc_protobuf_PROTOC_OUT_DIR} ${_plugin} ${_protobuf_include_path} ${_abs_file}
      DEPENDS ${_abs_file} protobuf::protoc ${btc_protobuf_DEPENDENCIES}
      COMMENT ${_comment}
      VERBATIM )
  endforeach()

  set_source_files_properties(${_generated_srcs_all} PROPERTIES GENERATED TRUE)
  target_sources(${btc_protobuf_TARGET} PRIVATE ${_generated_srcs_all})
  set_property(TARGET ${btc_protobuf_TARGET} PROPERTY INTERFACE_INCLUDE_DIRECTORIES ${_protobuf_imports})

endfunction()


