"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.NpmDownloadsClient = exports.NpmDownloadsPeriod = void 0;
var NpmDownloadsPeriod;
(function (NpmDownloadsPeriod) {
    /**
     * Gets downloads for the last available day. In practice, this will usually
     * be "yesterday" (in GMT) but if stats for that day have not yet landed, it
     * will be the day before.
     */
    NpmDownloadsPeriod["LAST_DAY"] = "last-day";
    /**
     * Gets downloads for the last 7 available days.
     */
    NpmDownloadsPeriod["LAST_WEEK"] = "last-week";
    /**
     * Gets downloads for the last 30 available days.
     */
    NpmDownloadsPeriod["LAST_MONTH"] = "last-month";
})(NpmDownloadsPeriod = exports.NpmDownloadsPeriod || (exports.NpmDownloadsPeriod = {}));
class NpmDownloadsClient {
    constructor(gotService) {
        this.got = gotService;
    }
    async getDownloadsRaw(packages, period, throwErrors) {
        if (packages.length > NpmDownloadsClient.MAX_PACKAGES_PER_QUERY) {
            throw new Error(`Too many packages were provided (max: ${NpmDownloadsClient.MAX_PACKAGES_PER_QUERY})`);
        }
        if (packages.some((pkg) => this.isScopedPackage(pkg)) && packages.length > 1) {
            throw new Error('Scoped packages aren\'t supported by the bulk query API.');
        }
        if (packages.length === 0)
            return new Map();
        console.log(`Querying NPM for ${packages.length} package(s): [${packages.join(', ')}]`);
        const result = await this.got(`${NpmDownloadsClient.NPM_DOWNLOADS_API_URL}/${period}/${packages.join(',')}`, {
            timeout: 5000,
        }).catch((err) => {
            if (throwErrors) {
                throw err;
            }
            else {
                return { body: JSON.stringify({ error: JSON.stringify(err) }) };
            }
        });
        const data = JSON.parse(result.body);
        // single package query error
        // ex. { "error": "package foo not found" }
        if ('error' in data) {
            if (throwErrors) {
                throw new Error(`Could not retrieve download metrics: ${data.error}`);
            }
            else {
                console.error(`Could not retrieve download metrics: ${data.error}`);
                return new Map();
            }
        }
        // only a single package was returned
        if (isSingleDownloadsEntry(data)) {
            return new Map([[packages[0], data]]);
        }
        // bulk query result
        for (const key of Object.keys(data)) {
            if (data[key] === null) {
                if (throwErrors) {
                    throw new Error(`Could not retrieve download metrics for package ${key}`);
                }
                else {
                    console.error(`Could not retrieve download metrics for package ${key}`);
                    delete data[key];
                }
            }
        }
        // typescript can't figure out that we removed all null values
        return new Map(Object.entries(data));
    }
    /**
     * Retrieves the number of downloads each package has on npm in the latest period.
     * Output is not guaranteed to be returned in a specific order.
     * If throwErrors option is specified, an error is thrown when a package's
     * download count is unavailable - otherwise, it's just omitted from
     * the output.
     */
    async getDownloads(packages, options = {}) {
        var _a, _b;
        const period = (_a = options.period) !== null && _a !== void 0 ? _a : NpmDownloadsPeriod.LAST_WEEK;
        const throwErrors = (_b = options.throwErrors) !== null && _b !== void 0 ? _b : true;
        // separate scoped and unscoped packages since scoped packages are not
        // supported by the bulk query API
        const scopedPackages = [];
        const unscopedPackages = [];
        for (const pkg of packages) {
            if (this.isScopedPackage(pkg)) {
                scopedPackages.push(pkg);
            }
            else {
                unscopedPackages.push(pkg);
            }
        }
        // we could parallelize this, but then it's more likely we get throttled
        const output = new Map();
        for (const pkg of scopedPackages) {
            const partialResults = await this.getDownloadsRaw([pkg], period, throwErrors);
            for (const [key, value] of partialResults) {
                output.set(key, value);
            }
        }
        for (let i = 0; i < unscopedPackages.length; i += NpmDownloadsClient.MAX_PACKAGES_PER_QUERY) {
            const batch = unscopedPackages.slice(i, i + NpmDownloadsClient.MAX_PACKAGES_PER_QUERY);
            const partialResults = await this.getDownloadsRaw(batch, period, throwErrors);
            for (const [key, value] of partialResults) {
                output.set(key, value);
            }
        }
        return output;
    }
    isScopedPackage(packageName) {
        return packageName.startsWith('@');
    }
}
exports.NpmDownloadsClient = NpmDownloadsClient;
NpmDownloadsClient.NPM_DOWNLOADS_API_URL = 'https://api.npmjs.org/downloads/point';
NpmDownloadsClient.MAX_PACKAGES_PER_QUERY = 128; // this is a limitation from npm's API! do not change
function isSingleDownloadsEntry(data) {
    return 'downloads' in data && typeof data.downloads === 'number';
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibnBtLWRvd25sb2Fkcy5sYW1iZGEtc2hhcmVkLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vLi4vc3JjL2JhY2tlbmQvY2F0YWxvZy1idWlsZGVyL25wbS1kb3dubG9hZHMubGFtYmRhLXNoYXJlZC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFFQSxJQUFZLGtCQWlCWDtBQWpCRCxXQUFZLGtCQUFrQjtJQUM1Qjs7OztPQUlHO0lBQ0gsMkNBQXFCLENBQUE7SUFFckI7O09BRUc7SUFDSCw2Q0FBdUIsQ0FBQTtJQUV2Qjs7T0FFRztJQUNILCtDQUF5QixDQUFBO0FBQzNCLENBQUMsRUFqQlcsa0JBQWtCLEdBQWxCLDBCQUFrQixLQUFsQiwwQkFBa0IsUUFpQjdCO0FBeUJELE1BQWEsa0JBQWtCO0lBTTdCLFlBQVksVUFBZTtRQUN6QixJQUFJLENBQUMsR0FBRyxHQUFHLFVBQVUsQ0FBQztJQUN4QixDQUFDO0lBRU8sS0FBSyxDQUFDLGVBQWUsQ0FDM0IsUUFBMkIsRUFDM0IsTUFBMEIsRUFDMUIsV0FBb0I7UUFFcEIsSUFBSSxRQUFRLENBQUMsTUFBTSxHQUFHLGtCQUFrQixDQUFDLHNCQUFzQixFQUFFO1lBQy9ELE1BQU0sSUFBSSxLQUFLLENBQUMseUNBQXlDLGtCQUFrQixDQUFDLHNCQUFzQixHQUFHLENBQUMsQ0FBQztTQUN4RztRQUNELElBQUksUUFBUSxDQUFDLElBQUksQ0FBQyxDQUFDLEdBQUcsRUFBRSxFQUFFLENBQUMsSUFBSSxDQUFDLGVBQWUsQ0FBQyxHQUFHLENBQUMsQ0FBQyxJQUFJLFFBQVEsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxFQUFFO1lBQzVFLE1BQU0sSUFBSSxLQUFLLENBQUMsMERBQTBELENBQUMsQ0FBQztTQUM3RTtRQUNELElBQUksUUFBUSxDQUFDLE1BQU0sS0FBSyxDQUFDO1lBQUUsT0FBTyxJQUFJLEdBQUcsRUFBRSxDQUFDO1FBRTVDLE9BQU8sQ0FBQyxHQUFHLENBQUMsb0JBQW9CLFFBQVEsQ0FBQyxNQUFNLGlCQUFpQixRQUFRLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUN4RixNQUFNLE1BQU0sR0FBRyxNQUFNLElBQUksQ0FBQyxHQUFHLENBQUMsR0FBRyxrQkFBa0IsQ0FBQyxxQkFBcUIsSUFBSSxNQUFNLElBQUksUUFBUSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsRUFBRSxFQUFFO1lBQzNHLE9BQU8sRUFBRSxJQUFLO1NBQ2YsQ0FBQyxDQUFDLEtBQUssQ0FBQyxDQUFDLEdBQUcsRUFBRSxFQUFFO1lBQ2YsSUFBSSxXQUFXLEVBQUU7Z0JBQ2YsTUFBTSxHQUFHLENBQUM7YUFDWDtpQkFBTTtnQkFDTCxPQUFPLEVBQUUsSUFBSSxFQUFFLElBQUksQ0FBQyxTQUFTLENBQUMsRUFBRSxLQUFLLEVBQUUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLEVBQUUsQ0FBQzthQUNqRTtRQUNILENBQUMsQ0FBQyxDQUFDO1FBRUgsTUFBTSxJQUFJLEdBQWlCLElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBRW5ELDZCQUE2QjtRQUM3QiwyQ0FBMkM7UUFDM0MsSUFBSSxPQUFPLElBQUksSUFBSSxFQUFFO1lBQ25CLElBQUksV0FBVyxFQUFFO2dCQUNmLE1BQU0sSUFBSSxLQUFLLENBQUMsd0NBQXdDLElBQUksQ0FBQyxLQUFLLEVBQUUsQ0FBQyxDQUFDO2FBQ3ZFO2lCQUFNO2dCQUNMLE9BQU8sQ0FBQyxLQUFLLENBQUMsd0NBQXdDLElBQUksQ0FBQyxLQUFLLEVBQUUsQ0FBQyxDQUFDO2dCQUNwRSxPQUFPLElBQUksR0FBRyxFQUFFLENBQUM7YUFDbEI7U0FDRjtRQUVELHFDQUFxQztRQUNyQyxJQUFJLHNCQUFzQixDQUFDLElBQUksQ0FBQyxFQUFFO1lBQ2hDLE9BQU8sSUFBSSxHQUFHLENBQUMsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsRUFBRSxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUM7U0FDdkM7UUFFRCxvQkFBb0I7UUFDcEIsS0FBSyxNQUFNLEdBQUcsSUFBSSxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxFQUFFO1lBQ25DLElBQUksSUFBSSxDQUFDLEdBQUcsQ0FBQyxLQUFLLElBQUksRUFBRTtnQkFDdEIsSUFBSSxXQUFXLEVBQUU7b0JBQ2YsTUFBTSxJQUFJLEtBQUssQ0FBQyxtREFBbUQsR0FBRyxFQUFFLENBQUMsQ0FBQztpQkFDM0U7cUJBQU07b0JBQ0wsT0FBTyxDQUFDLEtBQUssQ0FBQyxtREFBbUQsR0FBRyxFQUFFLENBQUMsQ0FBQztvQkFDeEUsT0FBTyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUM7aUJBQ2xCO2FBQ0Y7U0FDRjtRQUVELDhEQUE4RDtRQUM5RCxPQUFPLElBQUksR0FBRyxDQUFDLE1BQU0sQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFRLENBQUMsQ0FBQztJQUM5QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksS0FBSyxDQUFDLFlBQVksQ0FDdkIsUUFBa0IsRUFDbEIsVUFBK0IsRUFBRTs7UUFFakMsTUFBTSxNQUFNLFNBQUcsT0FBTyxDQUFDLE1BQU0sbUNBQUksa0JBQWtCLENBQUMsU0FBUyxDQUFDO1FBQzlELE1BQU0sV0FBVyxTQUFHLE9BQU8sQ0FBQyxXQUFXLG1DQUFJLElBQUksQ0FBQztRQUVoRCxzRUFBc0U7UUFDdEUsa0NBQWtDO1FBQ2xDLE1BQU0sY0FBYyxHQUFHLEVBQUUsQ0FBQztRQUMxQixNQUFNLGdCQUFnQixHQUFHLEVBQUUsQ0FBQztRQUM1QixLQUFLLE1BQU0sR0FBRyxJQUFJLFFBQVEsRUFBRTtZQUMxQixJQUFJLElBQUksQ0FBQyxlQUFlLENBQUMsR0FBRyxDQUFDLEVBQUU7Z0JBQzdCLGNBQWMsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUM7YUFDMUI7aUJBQU07Z0JBQ0wsZ0JBQWdCLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDO2FBQzVCO1NBQ0Y7UUFFRCx3RUFBd0U7UUFDeEUsTUFBTSxNQUFNLEdBQXVCLElBQUksR0FBRyxFQUFFLENBQUM7UUFDN0MsS0FBSyxNQUFNLEdBQUcsSUFBSSxjQUFjLEVBQUU7WUFDaEMsTUFBTSxjQUFjLEdBQUcsTUFBTSxJQUFJLENBQUMsZUFBZSxDQUFDLENBQUMsR0FBRyxDQUFDLEVBQUUsTUFBTSxFQUFFLFdBQVcsQ0FBQyxDQUFDO1lBQzlFLEtBQUssTUFBTSxDQUFDLEdBQUcsRUFBRSxLQUFLLENBQUMsSUFBSSxjQUFjLEVBQUU7Z0JBQ3pDLE1BQU0sQ0FBQyxHQUFHLENBQUMsR0FBRyxFQUFFLEtBQUssQ0FBQyxDQUFDO2FBQ3hCO1NBQ0Y7UUFDRCxLQUFLLElBQUksQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLEdBQUcsZ0JBQWdCLENBQUMsTUFBTSxFQUFFLENBQUMsSUFBSSxrQkFBa0IsQ0FBQyxzQkFBc0IsRUFBRTtZQUMzRixNQUFNLEtBQUssR0FBRyxnQkFBZ0IsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMsR0FBRyxrQkFBa0IsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO1lBQ3ZGLE1BQU0sY0FBYyxHQUFHLE1BQU0sSUFBSSxDQUFDLGVBQWUsQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFLFdBQVcsQ0FBQyxDQUFDO1lBQzlFLEtBQUssTUFBTSxDQUFDLEdBQUcsRUFBRSxLQUFLLENBQUMsSUFBSSxjQUFjLEVBQUU7Z0JBQ3pDLE1BQU0sQ0FBQyxHQUFHLENBQUMsR0FBRyxFQUFFLEtBQUssQ0FBQyxDQUFDO2FBQ3hCO1NBQ0Y7UUFFRCxPQUFPLE1BQU0sQ0FBQztJQUNoQixDQUFDO0lBRU8sZUFBZSxDQUFDLFdBQW1CO1FBQ3pDLE9BQU8sV0FBVyxDQUFDLFVBQVUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUNyQyxDQUFDOztBQW5ISCxnREFvSEM7QUFuSHdCLHdDQUFxQixHQUFHLHVDQUF1QyxDQUFDO0FBQ2hFLHlDQUFzQixHQUFHLEdBQUcsQ0FBQyxDQUFDLHFEQUFxRDtBQTBINUcsU0FBUyxzQkFBc0IsQ0FBQyxJQUFTO0lBQ3ZDLE9BQU8sV0FBVyxJQUFJLElBQUksSUFBSSxPQUFPLElBQUksQ0FBQyxTQUFTLEtBQUssUUFBUSxDQUFDO0FBQ25FLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgdHlwZSB7IEdvdCB9IGZyb20gJ2dvdC9kaXN0L3NvdXJjZS90eXBlcyc7XG5cbmV4cG9ydCBlbnVtIE5wbURvd25sb2Fkc1BlcmlvZCB7XG4gIC8qKlxuICAgKiBHZXRzIGRvd25sb2FkcyBmb3IgdGhlIGxhc3QgYXZhaWxhYmxlIGRheS4gSW4gcHJhY3RpY2UsIHRoaXMgd2lsbCB1c3VhbGx5XG4gICAqIGJlIFwieWVzdGVyZGF5XCIgKGluIEdNVCkgYnV0IGlmIHN0YXRzIGZvciB0aGF0IGRheSBoYXZlIG5vdCB5ZXQgbGFuZGVkLCBpdFxuICAgKiB3aWxsIGJlIHRoZSBkYXkgYmVmb3JlLlxuICAgKi9cbiAgTEFTVF9EQVkgPSAnbGFzdC1kYXknLFxuXG4gIC8qKlxuICAgKiBHZXRzIGRvd25sb2FkcyBmb3IgdGhlIGxhc3QgNyBhdmFpbGFibGUgZGF5cy5cbiAgICovXG4gIExBU1RfV0VFSyA9ICdsYXN0LXdlZWsnLFxuXG4gIC8qKlxuICAgKiBHZXRzIGRvd25sb2FkcyBmb3IgdGhlIGxhc3QgMzAgYXZhaWxhYmxlIGRheXMuXG4gICAqL1xuICBMQVNUX01PTlRIID0gJ2xhc3QtbW9udGgnXG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgTnBtRG93bmxvYWRzRW50cnkge1xuICByZWFkb25seSBkb3dubG9hZHM6IG51bWJlcjtcbiAgcmVhZG9ubHkgc3RhcnQ6IHN0cmluZztcbiAgcmVhZG9ubHkgZW5kOiBzdHJpbmc7XG4gIHJlYWRvbmx5IHBhY2thZ2U6IHN0cmluZztcbn1cblxuZXhwb3J0IHR5cGUgTnBtRG93bmxvYWRzT3V0cHV0ID0gTWFwPHN0cmluZywgTnBtRG93bmxvYWRzRW50cnk+O1xuXG5leHBvcnQgaW50ZXJmYWNlIE5wbURvd25sb2Fkc09wdGlvbnMge1xuICAvKipcbiAgICogVGhlIHBlcmlvZCB0byBxdWVyeSBmb3IgcGFja2FnZSBkb3dubG9hZCBjb3VudHMuXG4gICAqIEBkZWZhdWx0IE5wbURvd25sb2Fkc1BlcmlvZC5MQVNUX1dFRUtcbiAgICovXG4gIHJlYWRvbmx5IHBlcmlvZD86IE5wbURvd25sb2Fkc1BlcmlvZDtcblxuICAvKipcbiAgICogVGhyb3cgYW4gZXJyb3Igd2hlbiBhbnkgcGFja2FnZSdzIGRvd25sb2FkIG1ldHJpY3MgYXJlIG5vdCBhdmFpbGFibGUuXG4gICAqIEBkZWZhdWx0IHRydWVcbiAgICovXG4gIHJlYWRvbmx5IHRocm93RXJyb3JzPzogYm9vbGVhbjtcbn1cblxuZXhwb3J0IGNsYXNzIE5wbURvd25sb2Fkc0NsaWVudCB7XG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgTlBNX0RPV05MT0FEU19BUElfVVJMID0gJ2h0dHBzOi8vYXBpLm5wbWpzLm9yZy9kb3dubG9hZHMvcG9pbnQnO1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IE1BWF9QQUNLQUdFU19QRVJfUVVFUlkgPSAxMjg7IC8vIHRoaXMgaXMgYSBsaW1pdGF0aW9uIGZyb20gbnBtJ3MgQVBJISBkbyBub3QgY2hhbmdlXG5cbiAgcHJpdmF0ZSByZWFkb25seSBnb3Q6IEdvdDtcblxuICBjb25zdHJ1Y3Rvcihnb3RTZXJ2aWNlOiBHb3QpIHtcbiAgICB0aGlzLmdvdCA9IGdvdFNlcnZpY2U7XG4gIH1cblxuICBwcml2YXRlIGFzeW5jIGdldERvd25sb2Fkc1JhdyhcbiAgICBwYWNrYWdlczogcmVhZG9ubHkgc3RyaW5nW10sXG4gICAgcGVyaW9kOiBOcG1Eb3dubG9hZHNQZXJpb2QsXG4gICAgdGhyb3dFcnJvcnM6IGJvb2xlYW4sXG4gICk6IFByb21pc2U8TnBtRG93bmxvYWRzT3V0cHV0PiB7XG4gICAgaWYgKHBhY2thZ2VzLmxlbmd0aCA+IE5wbURvd25sb2Fkc0NsaWVudC5NQVhfUEFDS0FHRVNfUEVSX1FVRVJZKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYFRvbyBtYW55IHBhY2thZ2VzIHdlcmUgcHJvdmlkZWQgKG1heDogJHtOcG1Eb3dubG9hZHNDbGllbnQuTUFYX1BBQ0tBR0VTX1BFUl9RVUVSWX0pYCk7XG4gICAgfVxuICAgIGlmIChwYWNrYWdlcy5zb21lKChwa2cpID0+IHRoaXMuaXNTY29wZWRQYWNrYWdlKHBrZykpICYmIHBhY2thZ2VzLmxlbmd0aCA+IDEpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignU2NvcGVkIHBhY2thZ2VzIGFyZW5cXCd0IHN1cHBvcnRlZCBieSB0aGUgYnVsayBxdWVyeSBBUEkuJyk7XG4gICAgfVxuICAgIGlmIChwYWNrYWdlcy5sZW5ndGggPT09IDApIHJldHVybiBuZXcgTWFwKCk7XG5cbiAgICBjb25zb2xlLmxvZyhgUXVlcnlpbmcgTlBNIGZvciAke3BhY2thZ2VzLmxlbmd0aH0gcGFja2FnZShzKTogWyR7cGFja2FnZXMuam9pbignLCAnKX1dYCk7XG4gICAgY29uc3QgcmVzdWx0ID0gYXdhaXQgdGhpcy5nb3QoYCR7TnBtRG93bmxvYWRzQ2xpZW50Lk5QTV9ET1dOTE9BRFNfQVBJX1VSTH0vJHtwZXJpb2R9LyR7cGFja2FnZXMuam9pbignLCcpfWAsIHtcbiAgICAgIHRpbWVvdXQ6IDVfMDAwLCAvLyA1IHNlY29uZHNcbiAgICB9KS5jYXRjaCgoZXJyKSA9PiB7XG4gICAgICBpZiAodGhyb3dFcnJvcnMpIHtcbiAgICAgICAgdGhyb3cgZXJyO1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgcmV0dXJuIHsgYm9keTogSlNPTi5zdHJpbmdpZnkoeyBlcnJvcjogSlNPTi5zdHJpbmdpZnkoZXJyKSB9KSB9O1xuICAgICAgfVxuICAgIH0pO1xuXG4gICAgY29uc3QgZGF0YTogTnBtQXBpUmVzdWx0ID0gSlNPTi5wYXJzZShyZXN1bHQuYm9keSk7XG5cbiAgICAvLyBzaW5nbGUgcGFja2FnZSBxdWVyeSBlcnJvclxuICAgIC8vIGV4LiB7IFwiZXJyb3JcIjogXCJwYWNrYWdlIGZvbyBub3QgZm91bmRcIiB9XG4gICAgaWYgKCdlcnJvcicgaW4gZGF0YSkge1xuICAgICAgaWYgKHRocm93RXJyb3JzKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcihgQ291bGQgbm90IHJldHJpZXZlIGRvd25sb2FkIG1ldHJpY3M6ICR7ZGF0YS5lcnJvcn1gKTtcbiAgICAgIH0gZWxzZSB7XG4gICAgICAgIGNvbnNvbGUuZXJyb3IoYENvdWxkIG5vdCByZXRyaWV2ZSBkb3dubG9hZCBtZXRyaWNzOiAke2RhdGEuZXJyb3J9YCk7XG4gICAgICAgIHJldHVybiBuZXcgTWFwKCk7XG4gICAgICB9XG4gICAgfVxuXG4gICAgLy8gb25seSBhIHNpbmdsZSBwYWNrYWdlIHdhcyByZXR1cm5lZFxuICAgIGlmIChpc1NpbmdsZURvd25sb2Fkc0VudHJ5KGRhdGEpKSB7XG4gICAgICByZXR1cm4gbmV3IE1hcChbW3BhY2thZ2VzWzBdLCBkYXRhXV0pO1xuICAgIH1cblxuICAgIC8vIGJ1bGsgcXVlcnkgcmVzdWx0XG4gICAgZm9yIChjb25zdCBrZXkgb2YgT2JqZWN0LmtleXMoZGF0YSkpIHtcbiAgICAgIGlmIChkYXRhW2tleV0gPT09IG51bGwpIHtcbiAgICAgICAgaWYgKHRocm93RXJyb3JzKSB7XG4gICAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBDb3VsZCBub3QgcmV0cmlldmUgZG93bmxvYWQgbWV0cmljcyBmb3IgcGFja2FnZSAke2tleX1gKTtcbiAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICBjb25zb2xlLmVycm9yKGBDb3VsZCBub3QgcmV0cmlldmUgZG93bmxvYWQgbWV0cmljcyBmb3IgcGFja2FnZSAke2tleX1gKTtcbiAgICAgICAgICBkZWxldGUgZGF0YVtrZXldO1xuICAgICAgICB9XG4gICAgICB9XG4gICAgfVxuXG4gICAgLy8gdHlwZXNjcmlwdCBjYW4ndCBmaWd1cmUgb3V0IHRoYXQgd2UgcmVtb3ZlZCBhbGwgbnVsbCB2YWx1ZXNcbiAgICByZXR1cm4gbmV3IE1hcChPYmplY3QuZW50cmllcyhkYXRhKSBhcyBhbnkpO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHJpZXZlcyB0aGUgbnVtYmVyIG9mIGRvd25sb2FkcyBlYWNoIHBhY2thZ2UgaGFzIG9uIG5wbSBpbiB0aGUgbGF0ZXN0IHBlcmlvZC5cbiAgICogT3V0cHV0IGlzIG5vdCBndWFyYW50ZWVkIHRvIGJlIHJldHVybmVkIGluIGEgc3BlY2lmaWMgb3JkZXIuXG4gICAqIElmIHRocm93RXJyb3JzIG9wdGlvbiBpcyBzcGVjaWZpZWQsIGFuIGVycm9yIGlzIHRocm93biB3aGVuIGEgcGFja2FnZSdzXG4gICAqIGRvd25sb2FkIGNvdW50IGlzIHVuYXZhaWxhYmxlIC0gb3RoZXJ3aXNlLCBpdCdzIGp1c3Qgb21pdHRlZCBmcm9tXG4gICAqIHRoZSBvdXRwdXQuXG4gICAqL1xuICBwdWJsaWMgYXN5bmMgZ2V0RG93bmxvYWRzKFxuICAgIHBhY2thZ2VzOiBzdHJpbmdbXSxcbiAgICBvcHRpb25zOiBOcG1Eb3dubG9hZHNPcHRpb25zID0ge30sXG4gICk6IFByb21pc2U8TnBtRG93bmxvYWRzT3V0cHV0PiB7XG4gICAgY29uc3QgcGVyaW9kID0gb3B0aW9ucy5wZXJpb2QgPz8gTnBtRG93bmxvYWRzUGVyaW9kLkxBU1RfV0VFSztcbiAgICBjb25zdCB0aHJvd0Vycm9ycyA9IG9wdGlvbnMudGhyb3dFcnJvcnMgPz8gdHJ1ZTtcblxuICAgIC8vIHNlcGFyYXRlIHNjb3BlZCBhbmQgdW5zY29wZWQgcGFja2FnZXMgc2luY2Ugc2NvcGVkIHBhY2thZ2VzIGFyZSBub3RcbiAgICAvLyBzdXBwb3J0ZWQgYnkgdGhlIGJ1bGsgcXVlcnkgQVBJXG4gICAgY29uc3Qgc2NvcGVkUGFja2FnZXMgPSBbXTtcbiAgICBjb25zdCB1bnNjb3BlZFBhY2thZ2VzID0gW107XG4gICAgZm9yIChjb25zdCBwa2cgb2YgcGFja2FnZXMpIHtcbiAgICAgIGlmICh0aGlzLmlzU2NvcGVkUGFja2FnZShwa2cpKSB7XG4gICAgICAgIHNjb3BlZFBhY2thZ2VzLnB1c2gocGtnKTtcbiAgICAgIH0gZWxzZSB7XG4gICAgICAgIHVuc2NvcGVkUGFja2FnZXMucHVzaChwa2cpO1xuICAgICAgfVxuICAgIH1cblxuICAgIC8vIHdlIGNvdWxkIHBhcmFsbGVsaXplIHRoaXMsIGJ1dCB0aGVuIGl0J3MgbW9yZSBsaWtlbHkgd2UgZ2V0IHRocm90dGxlZFxuICAgIGNvbnN0IG91dHB1dDogTnBtRG93bmxvYWRzT3V0cHV0ID0gbmV3IE1hcCgpO1xuICAgIGZvciAoY29uc3QgcGtnIG9mIHNjb3BlZFBhY2thZ2VzKSB7XG4gICAgICBjb25zdCBwYXJ0aWFsUmVzdWx0cyA9IGF3YWl0IHRoaXMuZ2V0RG93bmxvYWRzUmF3KFtwa2ddLCBwZXJpb2QsIHRocm93RXJyb3JzKTtcbiAgICAgIGZvciAoY29uc3QgW2tleSwgdmFsdWVdIG9mIHBhcnRpYWxSZXN1bHRzKSB7XG4gICAgICAgIG91dHB1dC5zZXQoa2V5LCB2YWx1ZSk7XG4gICAgICB9XG4gICAgfVxuICAgIGZvciAobGV0IGkgPSAwOyBpIDwgdW5zY29wZWRQYWNrYWdlcy5sZW5ndGg7IGkgKz0gTnBtRG93bmxvYWRzQ2xpZW50Lk1BWF9QQUNLQUdFU19QRVJfUVVFUlkpIHtcbiAgICAgIGNvbnN0IGJhdGNoID0gdW5zY29wZWRQYWNrYWdlcy5zbGljZShpLCBpICsgTnBtRG93bmxvYWRzQ2xpZW50Lk1BWF9QQUNLQUdFU19QRVJfUVVFUlkpO1xuICAgICAgY29uc3QgcGFydGlhbFJlc3VsdHMgPSBhd2FpdCB0aGlzLmdldERvd25sb2Fkc1JhdyhiYXRjaCwgcGVyaW9kLCB0aHJvd0Vycm9ycyk7XG4gICAgICBmb3IgKGNvbnN0IFtrZXksIHZhbHVlXSBvZiBwYXJ0aWFsUmVzdWx0cykge1xuICAgICAgICBvdXRwdXQuc2V0KGtleSwgdmFsdWUpO1xuICAgICAgfVxuICAgIH1cblxuICAgIHJldHVybiBvdXRwdXQ7XG4gIH1cblxuICBwcml2YXRlIGlzU2NvcGVkUGFja2FnZShwYWNrYWdlTmFtZTogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHBhY2thZ2VOYW1lLnN0YXJ0c1dpdGgoJ0AnKTtcbiAgfVxufVxuXG4vLyBUeXBlcyBmb3IgdGhlIG91dHB1dCBvZiBOUE0ncyBBUEkuXG50eXBlIE5wbUFwaVJlc3VsdCA9IE5wbUFwaUVycm9yIHwgTnBtQXBpU2luZ2xlUmVzdWx0IHwgTnBtQXBpTXVsdGlwbGVSZXN1bHQ7XG50eXBlIE5wbUFwaUVycm9yID0geyBlcnJvcjogc3RyaW5nIH07XG50eXBlIE5wbUFwaVNpbmdsZVJlc3VsdCA9IE5wbURvd25sb2Fkc0VudHJ5O1xudHlwZSBOcG1BcGlNdWx0aXBsZVJlc3VsdCA9IHsgW2tleTogc3RyaW5nXTogTnBtQXBpU2luZ2xlUmVzdWx0IHwgbnVsbCB9O1xuXG5mdW5jdGlvbiBpc1NpbmdsZURvd25sb2Fkc0VudHJ5KGRhdGE6IGFueSk6IGRhdGEgaXMgTnBtRG93bmxvYWRzRW50cnkge1xuICByZXR1cm4gJ2Rvd25sb2FkcycgaW4gZGF0YSAmJiB0eXBlb2YgZGF0YS5kb3dubG9hZHMgPT09ICdudW1iZXInO1xufVxuIl19