"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const cf = require("@aws-cdk/aws-cloudfront");
const s3 = require("@aws-cdk/aws-s3");
const aws_s3_deployment_1 = require("@aws-cdk/aws-s3-deployment");
const cdk = require("@aws-cdk/core");
const extensions = require("../extensions");
test('minimal usage', () => {
    // GIVEN
    const app = new cdk.App();
    const stack = new cdk.Stack(app, 'demo-stack');
    // WHEN
    // create the cloudfront distribution with extension(s)
    const rewriteUriDemo = new extensions.DefaultDirIndex(stack, 'DefaultDirIndexDemo');
    // create Demo S3 Bucket.
    const bucket = new s3.Bucket(rewriteUriDemo, 'demoBucket', {
        autoDeleteObjects: true,
        removalPolicy: cdk.RemovalPolicy.DESTROY,
        websiteIndexDocument: 'index.html',
        websiteErrorDocument: 'index.html',
    });
    // Put demo Object to Bucket.
    new aws_s3_deployment_1.BucketDeployment(rewriteUriDemo, 'Deployment', {
        sources: [aws_s3_deployment_1.Source.asset(path.join(__dirname, '.'))],
        destinationBucket: bucket,
        retainOnDelete: false,
    });
    // CloudFront OriginAccessIdentity for Bucket
    const originAccessIdentity = new cf.OriginAccessIdentity(rewriteUriDemo, 'OriginAccessIdentity', {
        comment: `CloudFront OriginAccessIdentity for ${bucket.bucketName}`,
    });
    // CloudfrontWebDistribution
    const cloudfrontWebDistribution = new cf.CloudFrontWebDistribution(stack, 'CloudFrontWebDistribution', {
        enableIpV6: false,
        originConfigs: [
            {
                s3OriginSource: {
                    originAccessIdentity,
                    s3BucketSource: bucket,
                },
                behaviors: [{
                        isDefaultBehavior: true,
                        lambdaFunctionAssociations: [rewriteUriDemo],
                    }],
            },
        ],
    });
    new cdk.CfnOutput(stack, 'distributionDomainName', {
        value: cloudfrontWebDistribution.distributionDomainName,
    });
    // THEN
    expect(assert_1.SynthUtils.synthesize(stack).template).toMatchSnapshot();
    expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            DefaultCacheBehavior: {
                AllowedMethods: [
                    'GET',
                    'HEAD',
                ],
                CachedMethods: [
                    'GET',
                    'HEAD',
                ],
                Compress: true,
                ForwardedValues: {
                    Cookies: {
                        Forward: 'none',
                    },
                    QueryString: false,
                },
                LambdaFunctionAssociations: [
                    {
                        EventType: 'origin-request',
                    },
                ],
                TargetOriginId: 'origin1',
                ViewerProtocolPolicy: 'redirect-to-https',
            },
            DefaultRootObject: 'index.html',
        },
    });
});
//# sourceMappingURL=data:application/json;base64,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