import importlib
import inspect
import logging
import platform
import uuid
from datetime import datetime, timezone
from typing import Any

from chaoslib import __version__
from chaoslib.discovery.package import (
    get_discover_function,
    install,
    load_package,
)
from chaoslib.exceptions import DiscoveryFailed
from chaoslib.types import DiscoveredActivities, Discovery

__all__ = [
    "discover",
    "discover_activities",
    "discover_actions",
    "discover_probes",
    "initialize_discovery_result",
    "portable_type_name",
    "portable_type_name_to_python_type",
]
logger = logging.getLogger("chaostoolkit")


def discover(
    package_name: str,
    discover_system: bool = True,
    download_and_install: bool = True,
    keep_activities_arguments: bool = True,
) -> Discovery:
    """
    Discover the capabilities of an extension as well as the system it targets.

    Then apply any post discovery hook that are declared in the chaostoolkit
    settings under the `discovery/post-hook` section.

    By default, returns the arguments for each activity discovered unless
    `keep_activities_arguments` is set to `False`.
    """
    if download_and_install:
        install(package_name)
    package = load_package(package_name)
    discover_func = get_discover_function(package)

    discovery = discover_func(discover_system=discover_system)

    if not keep_activities_arguments:
        for activity in discovery["activities"]:
            activity.pop("arguments", None)
            activity.pop("return_type", None)

    return discovery


def initialize_discovery_result(
    extension_name: str, extension_version: str, discovery_type: str
) -> Discovery:
    """
    Intialize the discovery result payload to fill with activities and system
    discovery.
    """
    plt = platform.uname()
    return {
        "chaoslib_version": __version__,
        "id": str(uuid.uuid4()),
        "target": discovery_type,
        "date": f"{datetime.now(timezone.utc).isoformat()}Z",
        "platform": {
            "system": plt.system,
            "node": plt.node,
            "release": plt.release,
            "version": plt.version,
            "machine": plt.machine,
            "proc": plt.processor,
            "python": platform.python_version(),
        },
        "extension": {
            "name": extension_name,
            "version": extension_version,
        },
        "activities": [],
        "system": None,
    }


def discover_actions(extension_mod_name: str) -> DiscoveredActivities:
    """
    Discover actions from the given extension named `extension_mod_name`.
    """
    logger.debug(f"Searching for actions in {extension_mod_name}")
    return discover_activities(extension_mod_name, "action")


def discover_probes(extension_mod_name: str) -> DiscoveredActivities:
    """
    Discover probes from the given extension named `extension_mod_name`.
    """
    logger.debug(f"Searching for probes in {extension_mod_name}")
    return discover_activities(extension_mod_name, "probe")


def discover_activities(
    extension_mod_name: str,
    activity_type: str,  # noqa: C901
) -> DiscoveredActivities:
    """
    Discover exported activities from the given extension module name.
    """
    try:
        mod = importlib.import_module(extension_mod_name)
    except ImportError:
        raise DiscoveryFailed(
            f"could not import extension module '{extension_mod_name}'"
        )

    activities = []
    try:
        exported = getattr(mod, "__all__")
    except AttributeError:
        logger.warning(
            "'{m}' does not expose the __all__ attribute. "
            "It is required to determine what functions are actually "
            "exported as activities.".format(m=extension_mod_name)
        )
        return activities

    funcs = inspect.getmembers(mod, inspect.isfunction)
    for name, func in funcs:
        if exported and name not in exported:
            # do not return "private" functions
            continue

        sig = inspect.signature(func)
        activity = {
            "type": activity_type,
            "name": name,
            "mod": mod.__name__,
            "doc": inspect.getdoc(func),
            "arguments": [],
        }

        if sig.return_annotation is not inspect.Signature.empty:
            activity["return_type"] = portable_type_name(sig.return_annotation)

        for param in sig.parameters.values():
            if param.kind in (param.KEYWORD_ONLY, param.VAR_KEYWORD):
                continue

            arg = {
                "name": param.name,
            }

            if param.default is not inspect.Parameter.empty:
                arg["default"] = param.default
            if param.annotation is not inspect.Parameter.empty:
                arg["type"] = portable_type_name(param.annotation)
            activity["arguments"].append(arg)

        activities.append(activity)

    return activities


def portable_type_name(python_type: Any) -> str:  # noqa: C901
    """
    Return a fairly portable name for a Python type. The idea is to make it
    easy for consumer to read without caring for actual Python types
    themselves.

    These are not JSON types so don't eval them directly. This function does
    not try to be clever with rich types and will return `"object"` whenever
    it cannot make sense of the provide type.
    """
    if python_type is None:
        return "null"
    elif python_type is bool:
        return "boolean"
    elif python_type is int:
        return "integer"
    elif python_type is float:
        return "number"
    elif python_type is str:
        return "string"
    elif python_type is bytes:
        return "byte"
    elif python_type is set:
        return "set"
    elif python_type is tuple:
        return "tuple"
    elif python_type is list:
        return "list"
    elif python_type is dict:
        return "mapping"
    elif str(python_type).startswith("typing.Dict"):
        return "mapping"
    elif str(python_type).startswith("typing.List"):
        return "list"
    elif str(python_type).startswith("typing.Set"):
        return "set"

    logger.debug(
        f"'{str(python_type)}' could not be ported to something meaningful"
    )

    return "object"


def portable_type_name_to_python_type(name: str) -> Any:  # noqa: C901
    """
    Return the Python type associated to the given portable name.
    """
    if name == "null":
        return None
    elif name == "boolean":
        return bool
    elif name == "integer":
        return int
    elif name == "number":
        return float
    elif name == "string":
        return str
    elif name == "byte":
        return bytes
    elif name == "set":
        return set
    elif name == "list":
        return list
    elif name == "tuple":
        return tuple
    elif name == "mapping":
        return dict

    return object
