# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['datayoga',
 'datayoga.blocks',
 'datayoga.blocks.add_field',
 'datayoga.blocks.add_field.tests',
 'datayoga.blocks.explode',
 'datayoga.blocks.explode.tests',
 'datayoga.blocks.filter',
 'datayoga.blocks.filter.tests',
 'datayoga.blocks.map',
 'datayoga.blocks.remove_field',
 'datayoga.blocks.rename_field',
 'datayoga.blocks.rename_field.tests']

package_data = \
{'': ['*'], 'datayoga': ['resources/schemas/*']}

install_requires = \
['PyYAML>=6.0,<7.0', 'jmespath>=1.0.0,<2.0.0', 'jsonschema>=4.4.0,<5.0.0']

extras_require = \
{'test': ['mock>=4.0.3,<5.0.0',
          'pytest>=7.1.2,<8.0.0',
          'pytest-describe>=2.0.1,<3.0.0',
          'pytest-mock>=3.7.0,<4.0.0',
          'pytest-timeout>=2.1.0,<3.0.0',
          'requests-mock>=1.9.3,<2.0.0']}

setup_kwargs = {
    'name': 'datayoga',
    'version': '0.30.0',
    'description': 'DataYoga for Python',
    'long_description': '# datayoga-py\n\n## Introduction\n\n`datayoga-py` is the transformation engine used in `DataYoga`, a framework for building and generating data pipelines.\n\n## Installation\n\n```bash\npip install datayoga\n```\n\n## Quick Start\n\nThis demonstrates how to transform data using a DataYoga job.\n\n### Create a Job\n\nUse this `example.yaml`:\n\n```yaml\n- uses: add_field\n  with:\n    fields:\n      - field: full_name\n        language: jmespath\n        expression: concat([fname, \' \' , lname])\n      - field: country\n        language: sql\n        expression: country_code || \' - \' || UPPER(country_name)\n- uses: rename_field\n  with:\n    fields:\n      - from_field: fname\n        to_field: first_name\n      - from_field: lname\n        to_field: last_name\n- uses: remove_field\n  with:\n    fields:\n      - field: credit_card\n      - field: country_name\n      - field: country_code\n- uses: map\n  with:\n    expression:\n      {\n        first_name: first_name,\n        last_name: last_name,\n        greeting: "\'Hello \' || CASE WHEN gender = \'F\' THEN \'Ms.\' WHEN gender = \'M\' THEN \'Mr.\' ELSE \'N/A\' END || \' \' || full_name",\n        country: country,\n        full_name: full_name\n      }\n    language: sql\n```\n\n### Transform Data Using `datayoga-py`\n\nUse this code snippet to transform a data record using the job defined [above](#create-a-job):\n\n```python\nimport datayoga as dy\nfrom datayoga.job import Job\nfrom datayoga.utils import read_yaml\n\njob_settings = read_yaml("example.yaml")\njob = dy.compile(job_settings)\n\nassert job.transform({"fname": "jane", "lname": "smith", "country_code": 1, "country_name": "usa", "credit_card": "1234-5678-0000-9999", "gender": "F"}) == {"first_name": "jane", "last_name": "smith", "country": "1 - USA", "full_name": "jane smith", "greeting": "Hello Ms. jane smith"}\n```\n\nAs can be seen, the record has been transformed based on the job:\n\n- `fname` field renamed to `first_name`.\n- `lname` field renamed to `last_name`.\n- `country` field added based on an SQL expression.\n- `full_name` field added based on a [JMESPath](https://jmespath.org/) expression.\n- `greeting` field added based on an SQL expression.\n\n### Examples\n\n- Add a new field `country` out of an SQL expression that concatenates `country_code` and `country_name` fields after upper case the later:\n\n  ```yaml\n  uses: add_field\n  with:\n    field: country\n    language: sql\n    expression: country_code || \' - \' || UPPER(country_name)\n  ```\n\n- Rename `fname` field to `first_name` and `lname` field to `last_name`:\n\n  ```yaml\n  uses: rename_field\n  with:\n    fields:\n      - from_field: fname\n        to_field: first_name\n      - from_field: lname\n        to_field: last_name\n  ```\n\n- Remove `credit_card` field:\n\n  ```yaml\n  uses: remove_field\n  with:\n    field: credit_card\n  ```\n\nFor a full list of supported block types [see reference](https://datayoga-io.github.io/datayoga-py/).\n\n## Expression Language\n\nDataYoga supports both SQL and [JMESPath](https://jmespath.org/) expressions. JMESPath are especially useful to handle nested JSON data, while SQL is more suited to flat row-like structures.\n\n### Notes\n\n- Dot notation in expression represents nesting fields in the object, for example `name.first_name` refers to `{ "name": { "first_name": "John" } }`.\n- In order to refer to a field that contains a dot in its name, escape it, for example `name\\.first_name` refers to `{ "name.first_name": "John" }`.\n\n### JMESPath Custom Functions\n\nDataYoga adds the following custom functions to the standard JMESPath library:\n\n| Function     | Description                                                                                                          | Example                                                                                                                                 | Comments                                                                                                                                        |\n| ------------ | -------------------------------------------------------------------------------------------------------------------- | --------------------------------------------------------------------------------------------------------------------------------------- | ----------------------------------------------------------------------------------------------------------------------------------------------- |\n| `capitalize` | Capitalizes all the words in the string                                                                              | Input: `{"name": "john doe"}` <br /> Expression: `capitalize(name)` <br /> Output: `John Doe`                                           |\n| `concat`     | Concatenates an array of variables or literals                                                                       | Input: `{"fname": "john", "lname": "doe"}` <br /> Expression: `concat([fname, \' \' ,lname])` <br /> Output: `john doe`                   | This is equivalent to the more verbose built-in expression: `\' \'.join([fname,lname])`.                                                          |\n| `left`       | Returns a specified number of characters from the start of a given text string                                       | Input: `{"greeting": "hello world!"}` <br /> Expression: `` left(greeting, `5`) `` <br /> Output: `hello`                               |\n| `lower`      | Converts all uppercase characters in a string into lowercase characters                                              | Input: `{"fname": "John"}` <br /> Expression: `lower(fname)` <br /> Output: `john`                                                      |\n| `mid `       | Returns a specified number of characters from the middle of a given text string                                      | Input: `{"greeting": "hello world!"}` <br /> Expression: `` mid(greeting, `4`, `3`) `` <br /> Output: `o w`                             |\n| `replace`    | Replaces all the occurrences of a substring with a new one                                                           | Input: `{"sentence": "one four three four!"}` <br /> Expression: `replace(sentence, \'four\', \'two\')` <br /> Output: `one two three two!` |\n| `right`      | Returns a specified number of characters from the end of a given text string                                         | Input: `{"greeting": "hello world!"}` <br /> Expression: `` right(greeting, `6`) `` <br /> Output: `world!`                             |\n| `split`      | Splits a string into a list of strings after breaking the given string by the specified delimiter (comma by default) | Input: `{"departments": "finance,hr,r&d"}` <br /> Expression: `split(departments)` <br /> Output: `[\'finance\', \'hr\', \'r&d\']`            | Default delimiter is comma - a different delimiter can be passed to the function as the second argument, for example: `split(departments, \';\')` |\n| `upper`      | Converts all lowercase characters in a string into uppercase characters                                              | Input: `{"fname": "john"}` <br /> Expression: `upper(fname)` <br /> Output: `JOHN`                                                      |\n',
    'author': 'DataYoga',
    'author_email': 'admin@datayoga.io',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'None',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
