"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.kebabCaseKeys = exports.formatAsPythonModule = exports.sorted = exports.dedupArray = exports.deepMerge = exports.isObject = exports.isTruthy = exports.decamelizeKeysRecursively = exports.writeFile = exports.getFilePermissions = exports.execOrUndefined = exports.execCapture = exports.exec = void 0;
const child_process = require("child_process");
const path = require("path");
const fs = require("fs-extra");
const logging = require("./logging");
// eslint-disable-next-line @typescript-eslint/no-require-imports
const decamelize = require('decamelize');
const MAX_BUFFER = 10 * 1024 * 1024;
/**
 * Executes a command with STDOUT > STDERR.
 */
function exec(command, options) {
    logging.verbose(command);
    child_process.execSync(command, {
        stdio: ['inherit', process.stderr, 'pipe'],
        maxBuffer: MAX_BUFFER,
        cwd: options.cwd,
    });
}
exports.exec = exec;
/**
 * Executes command and returns STDOUT. If the command fails (non-zero), throws an error.
 */
function execCapture(command, options) {
    logging.verbose(command);
    return child_process.execSync(command, {
        stdio: ['inherit', 'pipe', 'pipe'],
        maxBuffer: MAX_BUFFER,
        cwd: options.cwd,
    });
}
exports.execCapture = execCapture;
/**
 * Executes `command` and returns its value or undefined if the command failed.
 */
function execOrUndefined(command, options) {
    try {
        const value = child_process.execSync(command, {
            stdio: ['inherit', 'pipe', 'pipe'],
            maxBuffer: MAX_BUFFER,
            cwd: options.cwd,
        }).toString('utf-8').trim();
        if (!value) {
            return undefined;
        } // an empty string is the same as undefined
        return value;
    }
    catch (_a) {
        return undefined;
    }
}
exports.execOrUndefined = execOrUndefined;
function getFilePermissions(options) {
    var _a, _b;
    const readonly = (_a = options.readonly) !== null && _a !== void 0 ? _a : false;
    const executable = (_b = options.executable) !== null && _b !== void 0 ? _b : false;
    if (readonly && executable) {
        return '500';
    }
    else if (readonly) {
        return '400';
    }
    else if (executable) {
        return '755';
    }
    else {
        return '644';
    }
}
exports.getFilePermissions = getFilePermissions;
function writeFile(filePath, data, options = {}) {
    if (fs.existsSync(filePath)) {
        fs.chmodSync(filePath, '600');
    }
    fs.mkdirpSync(path.dirname(filePath));
    fs.writeFileSync(filePath, data);
    fs.chmodSync(filePath, getFilePermissions(options));
}
exports.writeFile = writeFile;
function decamelizeKeysRecursively(input, opt) {
    var _a, _b, _c, _d;
    const shouldAlwaysDecamelize = () => true;
    const shouldDecamelize = (_a = opt === null || opt === void 0 ? void 0 : opt.shouldDecamelize) !== null && _a !== void 0 ? _a : shouldAlwaysDecamelize;
    const separator = (_b = opt === null || opt === void 0 ? void 0 : opt.separator) !== null && _b !== void 0 ? _b : '_';
    const path_ = (_c = opt === null || opt === void 0 ? void 0 : opt.path) !== null && _c !== void 0 ? _c : [];
    const maxDepth = (_d = opt === null || opt === void 0 ? void 0 : opt.maxDepth) !== null && _d !== void 0 ? _d : 10;
    if (path_.length > maxDepth) {
        throw new Error('Decamelled too deeply - check that the input has no circular references');
    }
    if (Array.isArray(input)) {
        return input.map((k, i) => decamelizeKeysRecursively(k, {
            ...opt,
            path: [...path_, i.toString()],
        }));
    }
    if (typeof input === 'object' && input !== null) {
        const mappedObject = {};
        for (const [key, value] of Object.entries(input)) {
            const transformedKey = shouldDecamelize([...path_, key], value)
                ? decamelize(key, separator)
                : key;
            mappedObject[transformedKey] = decamelizeKeysRecursively(value, {
                ...opt,
                path: [...path_, key],
            });
        }
        return mappedObject;
    }
    return input;
}
exports.decamelizeKeysRecursively = decamelizeKeysRecursively;
/**
 * Returns false if value is unset or a falsey value, and true otherwise.
 * @param value an environment variable
 */
function isTruthy(value) {
    return !(value === undefined || ['null', 'undefined', '0', 'false', ''].includes(value.toLocaleLowerCase()));
}
exports.isTruthy = isTruthy;
/**
 * Return whether the given value is an object
 *
 * Even though arrays and instances of classes technically are objects, we
 * usually want to treat them differently, so we return false in those cases.
 */
function isObject(x) {
    return x !== null && typeof x === 'object' && !Array.isArray(x)
        && x.constructor.name === 'Object';
}
exports.isObject = isObject;
/**
 * Recursively merge objects together
 *
 * The leftmost object is mutated and returned. Arrays are not merged
 * but overwritten just like scalars.
 *
 * If an object is merged into a non-object, the non-object is lost.
 *
 * `undefined`s will cause a value to be deleted if destructive is enabled.
 */
function deepMerge(objects, destructive = false) {
    function mergeOne(target, source) {
        for (const key of Object.keys(source)) {
            const value = source[key];
            if (isObject(value)) {
                // if the value at the target is not an object, override it with an
                // object so we can continue the recursion
                if (typeof (target[key]) !== 'object') {
                    target[key] = value;
                }
                mergeOne(target[key], value);
                // if the result of the merge is an empty object, it's because the
                // eventual value we assigned is `undefined`, and there are no
                // sibling concrete values alongside, so we can delete this tree.
                const output = target[key];
                if (typeof (output) === 'object' && Object.keys(output).length === 0 && destructive) {
                    delete target[key];
                }
            }
            else if (value === undefined && destructive) {
                delete target[key];
            }
            else if (typeof value !== 'undefined') {
                target[key] = value;
            }
        }
    }
    const others = objects.filter(x => x != null);
    if (others.length === 0) {
        return {};
    }
    const into = others.splice(0, 1)[0];
    others.forEach(other => mergeOne(into, other));
    return into;
}
exports.deepMerge = deepMerge;
/*
 * Deduplicate values in a list, returning a new array.
 * @param array list of values
 */
function dedupArray(array) {
    return array.filter((val, idx) => array.indexOf(val) === idx);
}
exports.dedupArray = dedupArray;
/**
 * Returns a sorted version of `x` or `undefined` if it is an empty array or object.
 */
function sorted(x) {
    if (x == null) {
        return undefined;
    }
    if (Array.isArray(x)) {
        if (x.length === 0) {
            return undefined;
        }
        return x.sort();
    }
    else if (typeof x === 'object') {
        if (Object.keys(x).length === 0) {
            return undefined;
        }
        const result = {};
        for (const [key, value] of Object.entries(x).sort(([l], [r]) => l.localeCompare(r))) {
            result[key] = value;
        }
        return result;
    }
    else {
        return x;
    }
}
exports.sorted = sorted;
function formatAsPythonModule(name) {
    return name.replace(/-/g, '_').replace(/\./g, '_');
}
exports.formatAsPythonModule = formatAsPythonModule;
function kebabCaseKeys(obj, recursive = true) {
    if (typeof obj !== 'object' || obj == null) {
        return obj;
    }
    if (Array.isArray(obj)) {
        if (recursive) {
            obj = obj.map((v) => kebabCaseKeys(v, recursive));
        }
        return obj;
    }
    const result = {};
    for (let [k, v] of Object.entries(obj)) {
        if (recursive) {
            v = kebabCaseKeys(v, recursive);
        }
        result[decamelize(k).replace(/_/mg, '-')] = v;
    }
    return result;
}
exports.kebabCaseKeys = kebabCaseKeys;
//# sourceMappingURL=data:application/json;base64,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