"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Release = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const component_1 = require("../component");
const github_1 = require("../github");
const workflows_model_1 = require("../github/workflows-model");
const version_1 = require("../version");
const publisher_1 = require("./publisher");
const BUILD_JOBID = 'release';
const GIT_REMOTE_STEPID = 'git_remote';
const LATEST_COMMIT_OUTPUT = 'latest_commit';
/**
 * (experimental) Manages releases (currently through GitHub workflows).
 *
 * By default, no branches are released. To add branches, call `addBranch()`.
 *
 * @experimental
 */
class Release extends component_1.Component {
    /**
     * @experimental
     */
    constructor(project, options) {
        var _b, _c, _d, _e, _f, _g, _h, _j;
        super(project);
        this._branches = new Array();
        this.jobs = {};
        if (Array.isArray(options.releaseBranches)) {
            throw new Error('"releaseBranches" is no longer an array. See type annotations');
        }
        this.github = project.github;
        this.buildTask = options.task;
        this.preBuildSteps = (_b = options.releaseWorkflowSetupSteps) !== null && _b !== void 0 ? _b : [];
        this.postBuildSteps = (_c = options.postBuildSteps) !== null && _c !== void 0 ? _c : [];
        this.antitamper = (_d = options.antitamper) !== null && _d !== void 0 ? _d : true;
        this.artifactsDirectory = (_e = options.artifactsDirectory) !== null && _e !== void 0 ? _e : 'dist';
        this.versionFile = options.versionFile;
        this.releaseSchedule = options.releaseSchedule;
        this.releaseEveryCommit = (_f = options.releaseEveryCommit) !== null && _f !== void 0 ? _f : true;
        this.containerImage = options.workflowContainerImage;
        this.version = new version_1.Version(project, {
            versionInputFile: this.versionFile,
            artifactsDirectory: this.artifactsDirectory,
        });
        this.publisher = new publisher_1.Publisher(project, {
            artifactName: this.artifactsDirectory,
            condition: `needs.${BUILD_JOBID}.outputs.${LATEST_COMMIT_OUTPUT} == github.sha`,
            buildJobId: BUILD_JOBID,
            jsiiReleaseVersion: options.jsiiReleaseVersion,
            failureIssue: options.releaseFailureIssue,
            failureIssueLabel: options.releaseFailureIssueLabel,
        });
        const githubRelease = (_g = options.githubRelease) !== null && _g !== void 0 ? _g : true;
        if (githubRelease) {
            this.publisher.publishToGitHubReleases({
                changelogFile: this.version.changelogFileName,
                versionFile: this.version.versionFileName,
                releaseTagFile: this.version.releaseTagFileName,
            });
        }
        // add the default branch
        this.defaultBranch = {
            name: options.branch,
            prerelease: options.prerelease,
            majorVersion: options.majorVersion,
            workflowName: (_h = options.releaseWorkflowName) !== null && _h !== void 0 ? _h : 'release',
            tagPrefix: options.releaseTagPrefix,
        };
        this._branches.push(this.defaultBranch);
        for (const [name, opts] of Object.entries((_j = options.releaseBranches) !== null && _j !== void 0 ? _j : {})) {
            this.addBranch(name, opts);
        }
    }
    /**
     * (experimental) Adds a release branch.
     *
     * It is a git branch from which releases are published. If a project has more than one release
     * branch, we require that `majorVersion` is also specified for the primary branch in order to
     * ensure branches always release the correct version.
     *
     * @param branch The branch to monitor (e.g. `main`, `v2.x`).
     * @param options Branch definition.
     * @experimental
     */
    addBranch(branch, options) {
        if (this._branches.find(b => b.name === branch)) {
            throw new Error(`The release branch ${branch} is already defined`);
        }
        // if we add a branch, we require that the default branch will also define a
        // major version.
        if (this.defaultBranch.majorVersion === undefined) {
            throw new Error('you must specify "majorVersion" for the default branch when adding multiple release branches');
        }
        this._branches.push({
            name: branch,
            ...options,
        });
    }
    /**
     * (experimental) Adds jobs to all release workflows.
     *
     * @param jobs The jobs to add (name => job).
     * @experimental
     */
    addJobs(jobs) {
        for (const [name, job] of Object.entries(jobs)) {
            this.jobs[name] = job;
        }
    }
    // render a workflow per branch and all the jobs to it
    /**
     * (experimental) Called before synthesis.
     *
     * @experimental
     */
    preSynthesize() {
        for (const branch of this._branches) {
            const workflow = this.createWorkflow(branch);
            if (workflow) {
                workflow.addJobs(this.publisher.render());
                workflow.addJobs(this.jobs);
            }
        }
    }
    /**
     * (experimental) Retrieve all release branch names.
     *
     * @experimental
     */
    get branches() {
        return this._branches.map(b => b.name);
    }
    /**
     * @returns a workflow or `undefined` if github integration is disabled.
     */
    createWorkflow(branch) {
        var _b;
        const workflowName = (_b = branch.workflowName) !== null && _b !== void 0 ? _b : `release-${branch.name}`;
        // to avoid race conditions between two commits trying to release the same
        // version, we check if the head sha is identical to the remote sha. if
        // not, we will skip the release and just finish the build.
        const noNewCommits = `\${{ steps.${GIT_REMOTE_STEPID}.outputs.${LATEST_COMMIT_OUTPUT} == github.sha }}`;
        // The arrays are being cloned to avoid accumulating values from previous branches
        const preBuildSteps = [...this.preBuildSteps];
        const env = {
            RELEASE: 'true',
        };
        if (branch.majorVersion !== undefined) {
            env.MAJOR = branch.majorVersion.toString();
        }
        if (branch.prerelease) {
            env.PRERELEASE = branch.prerelease;
        }
        if (branch.tagPrefix) {
            env.PREFIX = branch.tagPrefix;
        }
        // the "release" task prepares a release but does not publish anything. the
        // output of the release task is: `dist`, `.version.txt`, and
        // `.changelog.md`. this is what publish tasks expect.
        // if this is the release for "main" or "master", just call it "release".
        // otherwise, "release:BRANCH"
        const releaseTaskName = (branch.name === 'main' || branch.name === 'master') ? 'release' : `release:${branch.name}`;
        const releaseTask = this.project.addTask(releaseTaskName, {
            description: `Prepare a release from "${branch.name}" branch`,
            env,
        });
        releaseTask.exec(`rm -fr ${this.artifactsDirectory}`);
        releaseTask.spawn(this.version.bumpTask);
        releaseTask.spawn(this.buildTask);
        releaseTask.spawn(this.version.unbumpTask);
        // anti-tamper check (fails if there were changes to committed files)
        // this will identify any non-committed files generated during build (e.g. test snapshots)
        if (this.antitamper) {
            releaseTask.exec('git diff --ignore-space-at-eol --exit-code');
        }
        const postBuildSteps = [...this.postBuildSteps];
        // check if new commits were pushed to the repo while we were building.
        // if new commits have been pushed, we will cancel this release
        postBuildSteps.push({
            name: 'Check for new commits',
            id: GIT_REMOTE_STEPID,
            run: `echo ::set-output name=${LATEST_COMMIT_OUTPUT}::"$(git ls-remote origin -h \${{ github.ref }} | cut -f1)"`,
        });
        postBuildSteps.push({
            name: 'Upload artifact',
            if: noNewCommits,
            uses: 'actions/upload-artifact@v2.1.1',
            with: {
                name: this.artifactsDirectory,
                path: this.artifactsDirectory,
            },
        });
        if (this.github) {
            return new github_1.TaskWorkflow(this.github, {
                name: workflowName,
                jobId: BUILD_JOBID,
                outputs: {
                    latest_commit: {
                        stepId: GIT_REMOTE_STEPID,
                        outputName: LATEST_COMMIT_OUTPUT,
                    },
                },
                triggers: {
                    schedule: this.releaseSchedule ? [{ cron: this.releaseSchedule }] : undefined,
                    push: (this.releaseEveryCommit) ? { branches: [branch.name] } : undefined,
                },
                container: this.containerImage ? { image: this.containerImage } : undefined,
                env: {
                    CI: 'true',
                },
                permissions: {
                    contents: workflows_model_1.JobPermission.WRITE,
                },
                checkoutWith: {
                    // we must use 'fetch-depth=0' in order to fetch all tags
                    // otherwise tags are not checked out
                    'fetch-depth': 0,
                },
                preBuildSteps,
                task: releaseTask,
                postBuildSteps,
            });
        }
        else {
            return undefined;
        }
    }
}
exports.Release = Release;
_a = JSII_RTTI_SYMBOL_1;
Release[_a] = { fqn: "projen.release.Release", version: "0.29.3" };
//# sourceMappingURL=data:application/json;base64,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