"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * Resolves the latest version from git tags and uses `standard-version` to bump
 * to the next version based on commits.
 *
 * This expects `standard-version` to be installed in the path.
 *
 * Environment variables:
 *
 * - OUTFILE: (required) the name of the JSON output file (the "version" field
 *   will be updated with the latest version)
 * - PRERELEASE: (optional) a prerelease tag to use (e.g. "beta")
 * - MAJOR: major version number NN to filter (tags are filtered by "vNN."
 *   prefix). if not specified, the last major version is selected
 * - CHANGELOG: name of changelog file to create
 * - PREFIX: (optional) a prefix to apply to the release tag
 *
 */
const bump_version_1 = require("./bump-version");
const versionFile = process.env.OUTFILE;
const prerelease = process.env.PRERELEASE;
const major = process.env.MAJOR;
const changelog = process.env.CHANGELOG;
const bumpFile = process.env.BUMPFILE;
const releaseTagFile = process.env.RELEASETAG;
const prefix = process.env.PREFIX;
if (!versionFile) {
    throw new Error('OUTFILE is required');
}
if (!changelog) {
    throw new Error('CHANGELOG is required');
}
if (!bumpFile) {
    throw new Error('BUMPFILE is required');
}
if (!releaseTagFile) {
    throw new Error('RELEASETAG is required');
}
const majorVersion = (major == null || major === '') ? undefined : parseInt(major);
if (majorVersion === NaN) {
    throw new Error(`MAJOR must be a number: ${majorVersion}`);
}
const opts = {
    versionFile: versionFile,
    changelog: changelog,
    majorVersion: majorVersion,
    prerelease: prerelease,
    bumpFile: bumpFile,
    releaseTagFile: releaseTagFile,
    tagPrefix: prefix,
};
bump_version_1.bump(process.cwd(), opts).catch((e) => {
    console.log(e.stack);
    process.exit(1);
});
//# sourceMappingURL=data:application/json;base64,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