"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.bump = void 0;
const path_1 = require("path");
const fs_extra_1 = require("fs-extra");
const logging = require("../logging");
const util_1 = require("../util");
/**
 * Resolves the latest version from git tags and uses `standard-version` to bump
 * to the next version based on commits.
 *
 * This expects `standard-version` to be installed in the path.
 *
 * @param cwd working directory (git repository)
 * @param options options
 */
async function bump(cwd, options) {
    var _a;
    const versionFile = path_1.join(cwd, options.versionFile);
    const prerelease = options.prerelease;
    const major = options.majorVersion;
    const prefix = (_a = options.tagPrefix) !== null && _a !== void 0 ? _a : '';
    const bumpFile = path_1.join(cwd, options.bumpFile);
    const changelogFile = path_1.join(cwd, options.changelog);
    const releaseTagFile = path_1.join(cwd, options.releaseTagFile);
    await fs_extra_1.mkdirp(path_1.dirname(bumpFile));
    await fs_extra_1.mkdirp(path_1.dirname(changelogFile));
    await fs_extra_1.mkdirp(path_1.dirname(releaseTagFile));
    const { latestVersion, latestTag, isFirstRelease } = determineLatestTag({ cwd, major, prerelease, prefix });
    const content = await tryReadVersionFile(versionFile);
    // update version
    content.version = latestVersion;
    logging.info(`Update ${versionFile} to latest resolved version: ${latestVersion}`);
    await fs_extra_1.writeFile(versionFile, JSON.stringify(content, undefined, 2));
    // check if the latest commit already has a version tag
    const currentTags = util_1.execCapture('git tag --points-at HEAD', { cwd }).toString('utf8').split('\n');
    logging.info(`Tags listed on current commit: ${currentTags}`);
    let skipBump = false;
    if (currentTags.includes(latestTag)) {
        logging.info('Skipping bump...');
        skipBump = true;
        // delete the existing tag (locally)
        // if we don't do this, standard-version generates an empty changelog
        util_1.exec(`git tag --delete ${latestTag}`, { cwd });
    }
    // create a standard-version configuration file
    const rcfile = path_1.join(cwd, '.versionrc.json');
    await fs_extra_1.writeFile(rcfile, JSON.stringify({
        packageFiles: [{
                filename: versionFile,
                type: 'json',
            }],
        bumpFiles: [{
                filename: versionFile,
                type: 'json',
            }],
        commitAll: false,
        infile: changelogFile,
        prerelease: prerelease,
        header: '',
        skip: {
            commit: true,
            tag: true,
            bump: skipBump,
        },
    }, undefined, 2));
    const cmd = ['npx', 'standard-version@^9'];
    if (isFirstRelease) {
        cmd.push('--first-release');
    }
    util_1.exec(cmd.join(' '), { cwd });
    // add the tag back if it was previously removed
    if (currentTags.includes(latestTag)) {
        util_1.exec(`git tag ${latestTag}`, { cwd });
    }
    await fs_extra_1.remove(rcfile);
    const newVersion = (await tryReadVersionFile(versionFile)).version;
    if (!newVersion) {
        throw new Error(`bump failed: ${versionFile} does not have a version set`);
    }
    // if MAJOR is defined, ensure that the new version is within the same major version
    if (major) {
        if (!newVersion.startsWith(`${major}.`)) {
            throw new Error(`bump failed: this branch is configured to only publish v${major} releases - bump resulted in ${newVersion}`);
        }
    }
    await fs_extra_1.writeFile(bumpFile, newVersion);
    const newTag = `${prefix}v${newVersion}`;
    await fs_extra_1.writeFile(releaseTagFile, newTag);
}
exports.bump = bump;
async function tryReadVersionFile(versionFile) {
    if (!(await fs_extra_1.pathExists(versionFile))) {
        return {};
    }
    return JSON.parse(await fs_extra_1.readFile(versionFile, 'utf8'));
}
/**
 * Determines the latest release tag.
 * @param major (optional) A major version line to select from
 * @param prerelease (optional) A pre-release suffix.
 * @returns the latest tag, and whether it is the first release or not
 */
function determineLatestTag(options) {
    const { cwd, major, prerelease, prefix } = options;
    // filter only tags for this prefix and major version if specified (start with "vNN.").
    const prefixFilter = major ? `${prefix}v${major}.*` : `${prefix}v*`;
    const listGitTags = [
        'git',
        '-c "versionsort.suffix=-"',
        'tag',
        '--sort="-version:refname"',
        '--list',
        `"${prefixFilter}"`,
    ].join(' ');
    const stdout = util_1.execCapture(listGitTags, { cwd }).toString('utf8');
    let tags = stdout === null || stdout === void 0 ? void 0 : stdout.split('\n');
    // if "pre" is set, filter versions that end with "-PRE.ddd".
    if (prerelease) {
        tags = tags.filter(x => new RegExp(`-${prerelease}\.[0-9]+$`).test(x));
    }
    tags = tags.filter(x => x);
    // if a pre-release tag is used, then add it to the initial version
    let isFirstRelease = false;
    let latestTag;
    if (tags.length > 0) {
        latestTag = tags[0];
    }
    else {
        const initial = `${prefix}v${major !== null && major !== void 0 ? major : 0}.0.0`;
        latestTag = prerelease ? `${initial}-${prerelease}.0` : initial;
        isFirstRelease = true;
    }
    // remove tag prefix (if exists)
    let latestVersion = latestTag;
    if (prefix && latestVersion.startsWith(prefix)) {
        latestVersion = latestVersion.substr(prefix.length);
    }
    // remove "v" prefix (if exists)
    if (latestVersion.startsWith('v')) {
        latestVersion = latestVersion.substr(1);
    }
    return { latestVersion, latestTag, isFirstRelease };
}
//# sourceMappingURL=data:application/json;base64,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