"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.renderJavaScriptOptions = exports.resolveNewProject = exports.renderProjenNewOptions = void 0;
const inventory = require("../inventory");
const option_hints_1 = require("../option-hints");
const PROJEN_NEW = '__new__';
const TAB = makePadding(2);
/**
 * Renders options as if the project was created via `projen new` (embeds the __new__ field).
 */
function renderProjenNewOptions(fqn, args, comments = option_hints_1.NewProjectOptionHints.NONE) {
    return {
        ...args,
        [PROJEN_NEW]: { fqn, args, comments },
    };
}
exports.renderProjenNewOptions = renderProjenNewOptions;
function resolveNewProject(opts) {
    const f = opts[PROJEN_NEW];
    if (!f) {
        return undefined;
    }
    const type = inventory.resolveProjectType(f.fqn);
    if (!type) {
        throw new Error(`unable to resolve project type for ${f.fqn}`);
    }
    return {
        args: f.args,
        fqn: f.fqn,
        type: type,
        comments: f.comments,
    };
}
exports.resolveNewProject = resolveNewProject;
/**
 * Prints all parameters that can be used in a project type, alongside their descriptions.
 *
 * Parameters in `params` that aren't undefined are rendered as defaults,
 * while all other parameters are rendered as commented out.
 *
 * Returns the printed output and a set of required imports as an object
 * in the form { options, imports }.
 */
function renderJavaScriptOptions(opts) {
    var _a, _b, _c;
    const renders = {};
    const optionsWithDefaults = [];
    const useSingleQuotes = (str) => str === null || str === void 0 ? void 0 : str.replace(/"(.+)"/, '\'$1\'');
    const imports = new Set();
    for (const option of opts.type.options) {
        if (option.deprecated) {
            continue;
        }
        const optionName = option.name;
        if (opts.args[optionName] !== undefined) {
            const arg = opts.args[optionName];
            const { js, importName } = renderArgAsJavaScript(arg, option);
            if (importName)
                imports.add(importName);
            renders[optionName] = `${optionName}: ${useSingleQuotes(js)},`;
            optionsWithDefaults.push(optionName);
        }
        else {
            const defaultValue = ((_a = option.default) === null || _a === void 0 ? void 0 : _a.startsWith('-')) ? undefined : ((_b = option.default) !== null && _b !== void 0 ? _b : undefined);
            renders[optionName] = `// ${optionName}: ${useSingleQuotes(defaultValue)},`;
        }
    }
    const bootstrap = (_c = opts.bootstrap) !== null && _c !== void 0 ? _c : false;
    if (bootstrap) {
        renders[PROJEN_NEW] = `${PROJEN_NEW}: ${JSON.stringify({ args: opts.args, fqn: opts.type.fqn, comments: opts.comments })},`;
        optionsWithDefaults.push(PROJEN_NEW);
    }
    // generate rendering
    const result = [];
    result.push('{');
    // render options with defaults
    optionsWithDefaults.sort();
    for (const optionName of optionsWithDefaults) {
        result.push(`${TAB}${renders[optionName]}`);
    }
    if (result.length > 1) {
        result.push('');
    }
    // render options without defaults as comments
    if (opts.comments === option_hints_1.NewProjectOptionHints.ALL) {
        const options = opts.type.options.filter((opt) => !opt.deprecated && opts.args[opt.name] === undefined);
        result.push(...renderCommentedOptionsByModule(renders, options));
    }
    else if (opts.comments === option_hints_1.NewProjectOptionHints.FEATURED) {
        const options = opts.type.options.filter((opt) => !opt.deprecated && opts.args[opt.name] === undefined && opt.featured);
        result.push(...renderCommentedOptionsInOrder(renders, options));
    }
    else if (opts.comments === option_hints_1.NewProjectOptionHints.NONE) {
        // don't render any extra options
    }
    if (result[result.length - 1] === '') {
        result.pop();
    }
    result.push('}');
    return { renderedOptions: result.join('\n'), imports };
}
exports.renderJavaScriptOptions = renderJavaScriptOptions;
function renderCommentedOptionsByModule(renders, options) {
    var _a;
    const optionsByModule = {};
    for (const option of options) {
        const parentModule = option.parent;
        optionsByModule[parentModule] = (_a = optionsByModule[parentModule]) !== null && _a !== void 0 ? _a : [];
        optionsByModule[parentModule].push(option);
    }
    for (const parentModule in optionsByModule) {
        optionsByModule[parentModule].sort((o1, o2) => o1.name.localeCompare(o2.name));
    }
    const result = [];
    const marginSize = Math.max(...options.map((opt) => renders[opt.name].length));
    for (const [moduleName, optionGroup] of Object.entries(optionsByModule).sort()) {
        result.push(`${TAB}/* ${moduleName} */`);
        for (const option of optionGroup) {
            const paramRender = renders[option.name];
            const docstring = option.docs || 'No documentation found.';
            result.push(`${TAB}${paramRender}${makePadding(marginSize - paramRender.length + 2)}/* ${docstring} */`);
        }
        result.push('');
    }
    return result;
}
function renderCommentedOptionsInOrder(renders, options) {
    const result = [];
    const marginSize = Math.max(...options.map((opt) => renders[opt.name].length));
    for (const option of options) {
        const paramRender = renders[option.name];
        const docstring = option.docs || 'No documentation found.';
        result.push(`${TAB}${paramRender}${makePadding(marginSize - paramRender.length + 2)}/* ${docstring} */`);
    }
    return result;
}
/**
 * Renders a CLI argument as a basic JavaScript value. It must either be a
 * string, number, boolean, or enum.
 *
 * Returns a string and the name of any needed imports if needed as an
 * object in the form { js, import }.
 */
function renderArgAsJavaScript(arg, option) {
    // devDeps added as an exception to handle bootstrapping projects from external modules
    if (['string', 'number', 'boolean'].includes(option.type) || option.name === 'devDeps') {
        return { js: JSON.stringify(arg) };
    }
    else if (option.kind === 'enum') {
        if (!option.fqn) {
            throw new Error(`fqn field is missing from enum option ${option.name}`);
        }
        const parts = option.fqn.split('.'); // -> ['projen', 'web', 'MyEnum']
        const enumChoice = String(arg).toUpperCase().replace(/-/g, '_'); // custom-value -> CUSTOM_VALUE
        const js = `${parts.slice(1).join('.')}.${enumChoice}`; // -> web.MyEnum.CUSTOM_VALUE
        const importName = parts[1]; // -> web
        return { js, importName: importName };
    }
    else {
        throw new Error(`Unexpected option ${option.name} of kind: ${option.kind}`);
    }
}
function makePadding(paddingLength) {
    return ' '.repeat(paddingLength);
}
//# sourceMappingURL=data:application/json;base64,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