"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.readJsiiManifest = exports.resolveProjectType = exports.discover = void 0;
const path = require("path");
const fs = require("fs-extra");
// eslint-disable-next-line @typescript-eslint/no-require-imports
const decamelize = require('decamelize');
const PROJEN_MODULE_ROOT = path.join(__dirname, '..');
const PROJECT_BASE_FQN = 'projen.Project';
/**
 * Returns a list of project types exported the modules defined in `moduleDirs`.
 * This list will always also include the built-in projen project types.
 * Modules without a .jsii manifest are skipped.
 *
 * @param moduleDirs A list of npm module directories
 */
function discover(...moduleDirs) {
    const jsii = discoverJsiiTypes(...moduleDirs);
    const result = new Array();
    for (const fqn of Object.keys(jsii)) {
        const p = toProjectType(jsii, fqn);
        if (!p) {
            continue;
        }
        result.push(p);
    }
    return result.sort((r1, r2) => r1.pjid.localeCompare(r2.pjid));
}
exports.discover = discover;
/**
 * Resolve all jsii types from @modulesDirs.
 * When a jsii module is found it will recusively list the types from the dependant module as well
 *
 * @param moduleDirs
 * @returns
 */
function discoverJsiiTypes(...moduleDirs) {
    const jsii = {};
    const discoveredManifests = [];
    const discoverJsii = (dir) => {
        const jsiiFile = path.join(dir, '.jsii');
        if (!fs.existsSync(jsiiFile)) {
            return;
        } // no jsii manifest
        const manifest = fs.readJsonSync(jsiiFile);
        if (discoveredManifests.includes(manifest.fingerprint)) {
            return;
        }
        discoveredManifests.push(manifest.fingerprint);
        for (const [fqn, type] of Object.entries(manifest.types)) {
            jsii[fqn] = {
                ...type,
            };
        }
        // Also search recursively in nested project dependencies. If the requested module is an external module
        // this will also end-up in the projen module and add the projen types
        if (manifest.dependencies) {
            for (const dependency of Object.keys(manifest.dependencies)) {
                const nestedDependencyFolder = path.dirname(require.resolve(`${dependency}/package.json`, {
                    paths: [dir],
                }));
                if (fs.existsSync(nestedDependencyFolder)) {
                    discoverJsii(nestedDependencyFolder);
                }
            }
        }
    };
    // read all .jsii manifests from all requested modules and merge
    // them all into a single map of fqn->type.
    for (const dir of [...moduleDirs, PROJEN_MODULE_ROOT]) {
        discoverJsii(dir);
        // Read from scoped packages
        if (dir.includes('@') && fs.lstatSync(dir).isDirectory()) {
            const childDirs = fs.readdirSync(dir).map(file => path.join(dir, file));
            for (const child of childDirs) {
                discoverJsii(child);
            }
        }
    }
    return jsii;
}
function resolveProjectType(projectFqn) {
    let [moduleName] = projectFqn.split('.');
    if (moduleName === 'projen') {
        moduleName = PROJEN_MODULE_ROOT;
    }
    // try picking the manifest. We only need the base folder but this is directly a nice check if we request from a valid jsii package
    const jsiiManifestFile = require.resolve(`${moduleName}/.jsii`, {
        paths: [
            process.cwd(),
        ],
    });
    const moduleFolder = path.dirname(jsiiManifestFile);
    // Read all jsii types that can be loaded from this project type
    const jsii = discoverJsiiTypes(moduleFolder);
    return toProjectType(jsii, projectFqn);
}
exports.resolveProjectType = resolveProjectType;
function toProjectType(jsii, fqn) {
    var _a, _b, _c, _d;
    if (!isProjectType(jsii, fqn)) {
        return undefined;
    }
    const typeinfo = jsii[fqn];
    // projen.web.ReactProject -> web.ReactProject
    const typename = fqn.substring(fqn.indexOf('.') + 1);
    const docsurl = `https://github.com/projen/projen/blob/main/API.md#projen-${typename.toLocaleLowerCase().replace(/\./g, '-')}`;
    let pjid = (_c = (_b = (_a = typeinfo.docs) === null || _a === void 0 ? void 0 : _a.custom) === null || _b === void 0 ? void 0 : _b.pjid) !== null && _c !== void 0 ? _c : decamelize(typename).replace(/_project$/, '');
    return {
        moduleName: typeinfo.assembly,
        typename,
        pjid,
        fqn,
        options: discoverOptions(jsii, fqn).sort((o1, o2) => o1.name.localeCompare(o2.name)),
        docs: (_d = typeinfo.docs) === null || _d === void 0 ? void 0 : _d.summary,
        docsurl,
    };
}
function readJsiiManifest(jsiiFqn) {
    let [moduleName] = jsiiFqn.split('.');
    if (moduleName === 'projen') {
        moduleName = PROJEN_MODULE_ROOT;
    }
    const jsiiManifestFile = require.resolve(`${moduleName}/.jsii`);
    return fs.readJsonSync(jsiiManifestFile);
}
exports.readJsiiManifest = readJsiiManifest;
function discoverOptions(jsii, fqn) {
    var _a, _b, _c, _d;
    const options = {};
    const params = (_c = (_b = (_a = jsii[fqn]) === null || _a === void 0 ? void 0 : _a.initializer) === null || _b === void 0 ? void 0 : _b.parameters) !== null && _c !== void 0 ? _c : [];
    const optionsParam = params[0];
    const optionsTypeFqn = (_d = optionsParam === null || optionsParam === void 0 ? void 0 : optionsParam.type) === null || _d === void 0 ? void 0 : _d.fqn;
    if (params.length > 1 || (params.length === 1 && (optionsParam === null || optionsParam === void 0 ? void 0 : optionsParam.name) !== 'options')) {
        throw new Error(`constructor for project ${fqn} must have a single "options" argument of a struct type. got ${JSON.stringify(params)}`);
    }
    addOptions(optionsTypeFqn);
    const opts = Object.values(options);
    return opts.sort((a, b) => a.switch.localeCompare(b.switch));
    function addOptions(ofqn, basePath = [], optional = false) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o;
        if (!ofqn) {
            return;
        }
        const struct = jsii[ofqn];
        if (!struct) {
            throw new Error(`unable to find options type ${ofqn} for project ${fqn}`);
        }
        for (const prop of (_a = struct.properties) !== null && _a !== void 0 ? _a : []) {
            const propPath = [...basePath, prop.name];
            // protect against double-booking
            if (prop.name in options) {
                throw new Error(`duplicate option "${prop.name}" in ${fqn} (already declared in ${options[prop.name].parent})`);
            }
            let typeName;
            let jsiiKind;
            if ((_b = prop.type) === null || _b === void 0 ? void 0 : _b.primitive) {
                typeName = (_c = prop.type) === null || _c === void 0 ? void 0 : _c.primitive; // e.g. 'string', 'boolean', 'number'
            }
            else if ((_d = prop.type) === null || _d === void 0 ? void 0 : _d.fqn) {
                typeName = (_e = prop.type) === null || _e === void 0 ? void 0 : _e.fqn.split('.').pop(); // projen.NodeProjectOptions -> NodeProjectOptions
                jsiiKind = jsii[(_f = prop.type) === null || _f === void 0 ? void 0 : _f.fqn].kind; // e.g. 'class', 'interface', 'enum'
            }
            else { // any other types such as collection types
                typeName = 'unknown';
            }
            const isOptional = optional || prop.optional;
            let defaultValue = (_g = prop.docs) === null || _g === void 0 ? void 0 : _g.default;
            if (defaultValue === 'undefined') {
                defaultValue = undefined;
            }
            // if this is a mandatory option and we have a default value, it has to be JSON-parsable to the correct type
            if (!isOptional && defaultValue) {
                if (!((_h = prop.type) === null || _h === void 0 ? void 0 : _h.primitive)) {
                    throw new Error(`required option "${prop.name}" with a @default must use primitive types (string, number or boolean). type found is: ${typeName}`);
                }
                checkDefaultIsParsable(prop.name, defaultValue, (_j = prop.type) === null || _j === void 0 ? void 0 : _j.primitive);
            }
            options[prop.name] = filterUndefined({
                path: propPath,
                parent: struct.name,
                name: prop.name,
                fqn: (_k = prop.type) === null || _k === void 0 ? void 0 : _k.fqn,
                docs: prop.docs.summary,
                type: typeName,
                kind: jsiiKind,
                switch: propPath.map(p => decamelize(p).replace(/_/g, '-')).join('-'),
                default: defaultValue,
                optional: isOptional,
                featured: ((_m = (_l = prop.docs) === null || _l === void 0 ? void 0 : _l.custom) === null || _m === void 0 ? void 0 : _m.featured) === 'true',
                deprecated: prop.docs.stability === 'deprecated' ? true : undefined,
            });
        }
        for (const ifc of (_o = struct.interfaces) !== null && _o !== void 0 ? _o : []) {
            addOptions(ifc);
        }
    }
}
function filterUndefined(obj) {
    const ret = {};
    for (const [k, v] of Object.entries(obj)) {
        if (v !== undefined) {
            ret[k] = v;
        }
    }
    return ret;
}
function isProjectType(jsii, fqn) {
    var _a;
    const type = jsii[fqn];
    if (type.kind !== 'class') {
        return false;
    }
    if (type.abstract) {
        return false;
    }
    if ((_a = type.docs) === null || _a === void 0 ? void 0 : _a.deprecated) {
        return false;
    }
    let curr = type;
    while (true) {
        if (curr.fqn === PROJECT_BASE_FQN) {
            return true;
        }
        if (!curr.base) {
            return false;
        }
        curr = jsii[curr.base];
        if (!curr) {
            return false;
        }
    }
}
function checkDefaultIsParsable(prop, value, type) {
    // macros are pass-through
    if (value.startsWith('$')) {
        return;
    }
    try {
        const parsed = JSON.parse(value);
        if (typeof (parsed) !== type) {
            throw new Error(`cannot parse @default value for mandatory option ${prop} as a ${type}: ${parsed}`);
        }
    }
    catch (e) {
        throw new Error(`unable to JSON.parse() value "${value}" specified as @default for mandatory option "${prop}": ${e.message}`);
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW52ZW50b3J5LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vc3JjL2ludmVudG9yeS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSw2QkFBNkI7QUFDN0IsK0JBQStCO0FBRS9CLGlFQUFpRTtBQUNqRSxNQUFNLFVBQVUsR0FBRyxPQUFPLENBQUMsWUFBWSxDQUFDLENBQUM7QUFDekMsTUFBTSxrQkFBa0IsR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxJQUFJLENBQUMsQ0FBQztBQUN0RCxNQUFNLGdCQUFnQixHQUFHLGdCQUFnQixDQUFDO0FBbUUxQzs7Ozs7O0dBTUc7QUFDSCxTQUFnQixRQUFRLENBQUMsR0FBRyxVQUFvQjtJQUU5QyxNQUFNLElBQUksR0FBRyxpQkFBaUIsQ0FBQyxHQUFHLFVBQVUsQ0FBQyxDQUFDO0lBRTlDLE1BQU0sTUFBTSxHQUFHLElBQUksS0FBSyxFQUFlLENBQUM7SUFFeEMsS0FBSyxNQUFNLEdBQUcsSUFBSSxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxFQUFFO1FBQ25DLE1BQU0sQ0FBQyxHQUFHLGFBQWEsQ0FBQyxJQUFJLEVBQUUsR0FBRyxDQUFDLENBQUM7UUFDbkMsSUFBSSxDQUFDLENBQUMsRUFBRTtZQUNOLFNBQVM7U0FDVjtRQUVELE1BQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUM7S0FDaEI7SUFFRCxPQUFPLE1BQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQyxFQUFFLEVBQUUsRUFBRSxFQUFFLEVBQUUsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQztBQUNqRSxDQUFDO0FBaEJELDRCQWdCQztBQUVEOzs7Ozs7R0FNRztBQUNILFNBQVMsaUJBQWlCLENBQUMsR0FBRyxVQUFvQjtJQUNoRCxNQUFNLElBQUksR0FBYyxFQUFFLENBQUM7SUFDM0IsTUFBTSxtQkFBbUIsR0FBa0IsRUFBRSxDQUFDO0lBRTlDLE1BQU0sWUFBWSxHQUFHLENBQUMsR0FBVyxFQUFFLEVBQUU7UUFDbkMsTUFBTSxRQUFRLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxHQUFHLEVBQUUsT0FBTyxDQUFDLENBQUM7UUFDekMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxVQUFVLENBQUMsUUFBUSxDQUFDLEVBQUU7WUFBRSxPQUFPO1NBQUUsQ0FBQyxtQkFBbUI7UUFFN0QsTUFBTSxRQUFRLEdBQUcsRUFBRSxDQUFDLFlBQVksQ0FBQyxRQUFRLENBQUMsQ0FBQztRQUUzQyxJQUFJLG1CQUFtQixDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsV0FBVyxDQUFDLEVBQUU7WUFDdEQsT0FBTztTQUNSO1FBQ0QsbUJBQW1CLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxXQUFXLENBQUMsQ0FBQztRQUUvQyxLQUFLLE1BQU0sQ0FBQyxHQUFHLEVBQUUsSUFBSSxDQUFDLElBQUksTUFBTSxDQUFDLE9BQU8sQ0FBQyxRQUFRLENBQUMsS0FBa0IsQ0FBQyxFQUFFO1lBQ3JFLElBQUksQ0FBQyxHQUFHLENBQUMsR0FBRztnQkFDVixHQUFHLElBQUk7YUFDUixDQUFDO1NBQ0g7UUFFRCx3R0FBd0c7UUFDeEcsc0VBQXNFO1FBQ3RFLElBQUksUUFBUSxDQUFDLFlBQVksRUFBRTtZQUN6QixLQUFLLE1BQU0sVUFBVSxJQUFJLE1BQU0sQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLFlBQVksQ0FBQyxFQUFFO2dCQUMzRCxNQUFNLHNCQUFzQixHQUFHLElBQUksQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxHQUFHLFVBQVUsZUFBZSxFQUFFO29CQUN4RixLQUFLLEVBQUUsQ0FBQyxHQUFHLENBQUM7aUJBQ2IsQ0FBQyxDQUFDLENBQUM7Z0JBRUosSUFBSSxFQUFFLENBQUMsVUFBVSxDQUFDLHNCQUFzQixDQUFDLEVBQUU7b0JBQ3pDLFlBQVksQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO2lCQUN0QzthQUNGO1NBQ0Y7SUFDSCxDQUFDLENBQUM7SUFFRixnRUFBZ0U7SUFDaEUsMkNBQTJDO0lBQzNDLEtBQUssTUFBTSxHQUFHLElBQUksQ0FBQyxHQUFHLFVBQVUsRUFBRSxrQkFBa0IsQ0FBQyxFQUFFO1FBQ3JELFlBQVksQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUVsQiw0QkFBNEI7UUFDNUIsSUFBSSxHQUFHLENBQUMsUUFBUSxDQUFDLEdBQUcsQ0FBQyxJQUFJLEVBQUUsQ0FBQyxTQUFTLENBQUMsR0FBRyxDQUFDLENBQUMsV0FBVyxFQUFFLEVBQUU7WUFDeEQsTUFBTSxTQUFTLEdBQUcsRUFBRSxDQUFDLFdBQVcsQ0FBQyxHQUFHLENBQUMsQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLEdBQUcsRUFBRSxJQUFJLENBQUMsQ0FBQyxDQUFDO1lBQ3hFLEtBQUssTUFBTSxLQUFLLElBQUksU0FBUyxFQUFFO2dCQUM3QixZQUFZLENBQUMsS0FBSyxDQUFDLENBQUM7YUFDckI7U0FDRjtLQUNGO0lBRUQsT0FBTyxJQUFJLENBQUM7QUFDZCxDQUFDO0FBRUQsU0FBZ0Isa0JBQWtCLENBQUMsVUFBa0I7SUFDbkQsSUFBSSxDQUFDLFVBQVUsQ0FBQyxHQUFHLFVBQVUsQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDekMsSUFBSSxVQUFVLEtBQUssUUFBUSxFQUFFO1FBQzNCLFVBQVUsR0FBRyxrQkFBa0IsQ0FBQztLQUNqQztJQUVELG1JQUFtSTtJQUNuSSxNQUFNLGdCQUFnQixHQUFHLE9BQU8sQ0FBQyxPQUFPLENBQUMsR0FBRyxVQUFVLFFBQVEsRUFBRTtRQUM5RCxLQUFLLEVBQUU7WUFDTCxPQUFPLENBQUMsR0FBRyxFQUFFO1NBQ2Q7S0FDRixDQUFDLENBQUM7SUFDSCxNQUFNLFlBQVksR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLGdCQUFnQixDQUFDLENBQUM7SUFFcEQsZ0VBQWdFO0lBQ2hFLE1BQU0sSUFBSSxHQUFHLGlCQUFpQixDQUFDLFlBQVksQ0FBQyxDQUFDO0lBQzdDLE9BQU8sYUFBYSxDQUFDLElBQUksRUFBRSxVQUFVLENBQUMsQ0FBQztBQUN6QyxDQUFDO0FBakJELGdEQWlCQztBQUVELFNBQVMsYUFBYSxDQUFDLElBQWUsRUFBRSxHQUFXOztJQUNqRCxJQUFJLENBQUMsYUFBYSxDQUFDLElBQUksRUFBRSxHQUFHLENBQUMsRUFBRTtRQUM3QixPQUFPLFNBQVMsQ0FBQztLQUNsQjtJQUVELE1BQU0sUUFBUSxHQUFHLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUUzQiw4Q0FBOEM7SUFDOUMsTUFBTSxRQUFRLEdBQUcsR0FBRyxDQUFDLFNBQVMsQ0FBQyxHQUFHLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDO0lBRXJELE1BQU0sT0FBTyxHQUFHLDREQUE0RCxRQUFRLENBQUMsaUJBQWlCLEVBQUUsQ0FBQyxPQUFPLENBQUMsS0FBSyxFQUFFLEdBQUcsQ0FBQyxFQUFFLENBQUM7SUFDL0gsSUFBSSxJQUFJLHFCQUFHLFFBQVEsQ0FBQyxJQUFJLDBDQUFFLE1BQU0sMENBQUUsSUFBSSxtQ0FBSSxVQUFVLENBQUMsUUFBUSxDQUFDLENBQUMsT0FBTyxDQUFDLFdBQVcsRUFBRSxFQUFFLENBQUMsQ0FBQztJQUN4RixPQUFPO1FBQ0wsVUFBVSxFQUFFLFFBQVEsQ0FBQyxRQUFRO1FBQzdCLFFBQVE7UUFDUixJQUFJO1FBQ0osR0FBRztRQUNILE9BQU8sRUFBRSxlQUFlLENBQUMsSUFBSSxFQUFFLEdBQUcsQ0FBQyxDQUFDLElBQUksQ0FBQyxDQUFDLEVBQUUsRUFBRSxFQUFFLEVBQUUsRUFBRSxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUNwRixJQUFJLFFBQUUsUUFBUSxDQUFDLElBQUksMENBQUUsT0FBTztRQUM1QixPQUFPO0tBQ08sQ0FBQztBQUNuQixDQUFDO0FBRUQsU0FBZ0IsZ0JBQWdCLENBQUMsT0FBZTtJQUM5QyxJQUFJLENBQUMsVUFBVSxDQUFDLEdBQUcsT0FBTyxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QyxJQUFJLFVBQVUsS0FBSyxRQUFRLEVBQUU7UUFDM0IsVUFBVSxHQUFHLGtCQUFrQixDQUFDO0tBQ2pDO0lBRUQsTUFBTSxnQkFBZ0IsR0FBRyxPQUFPLENBQUMsT0FBTyxDQUFDLEdBQUcsVUFBVSxRQUFRLENBQUMsQ0FBQztJQUNoRSxPQUFPLEVBQUUsQ0FBQyxZQUFZLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztBQUMzQyxDQUFDO0FBUkQsNENBUUM7QUFFRCxTQUFTLGVBQWUsQ0FBQyxJQUFlLEVBQUUsR0FBVzs7SUFDbkQsTUFBTSxPQUFPLEdBQXNDLEVBQUUsQ0FBQztJQUN0RCxNQUFNLE1BQU0scUJBQUcsSUFBSSxDQUFDLEdBQUcsQ0FBQywwQ0FBRSxXQUFXLDBDQUFFLFVBQVUsbUNBQUksRUFBRSxDQUFDO0lBQ3hELE1BQU0sWUFBWSxHQUFHLE1BQU0sQ0FBQyxDQUFDLENBQUMsQ0FBQztJQUMvQixNQUFNLGNBQWMsU0FBRyxZQUFZLGFBQVosWUFBWSx1QkFBWixZQUFZLENBQUUsSUFBSSwwQ0FBRSxHQUFHLENBQUM7SUFFL0MsSUFBSSxNQUFNLENBQUMsTUFBTSxHQUFHLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxNQUFNLEtBQUssQ0FBQyxJQUFJLENBQUEsWUFBWSxhQUFaLFlBQVksdUJBQVosWUFBWSxDQUFFLElBQUksTUFBSyxTQUFTLENBQUMsRUFBRTtRQUNsRixNQUFNLElBQUksS0FBSyxDQUFDLDJCQUEyQixHQUFHLGdFQUFnRSxJQUFJLENBQUMsU0FBUyxDQUFDLE1BQU0sQ0FBQyxFQUFFLENBQUMsQ0FBQztLQUN6STtJQUVELFVBQVUsQ0FBQyxjQUFjLENBQUMsQ0FBQztJQUUzQixNQUFNLElBQUksR0FBRyxNQUFNLENBQUMsTUFBTSxDQUFDLE9BQU8sQ0FBQyxDQUFDO0lBRXBDLE9BQU8sSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQyxNQUFNLENBQUMsYUFBYSxDQUFDLENBQUMsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDO0lBRTdELFNBQVMsVUFBVSxDQUFDLElBQWEsRUFBRSxXQUFxQixFQUFFLEVBQUUsUUFBUSxHQUFHLEtBQUs7O1FBQzFFLElBQUksQ0FBQyxJQUFJLEVBQUU7WUFDVCxPQUFPO1NBQ1I7UUFFRCxNQUFNLE1BQU0sR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDMUIsSUFBSSxDQUFDLE1BQU0sRUFBRTtZQUNYLE1BQU0sSUFBSSxLQUFLLENBQUMsK0JBQStCLElBQUksZ0JBQWdCLEdBQUcsRUFBRSxDQUFDLENBQUM7U0FDM0U7UUFFRCxLQUFLLE1BQU0sSUFBSSxVQUFJLE1BQU0sQ0FBQyxVQUFVLG1DQUFJLEVBQUUsRUFBRTtZQUMxQyxNQUFNLFFBQVEsR0FBRyxDQUFDLEdBQUcsUUFBUSxFQUFFLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztZQUUxQyxpQ0FBaUM7WUFDakMsSUFBSSxJQUFJLENBQUMsSUFBSSxJQUFJLE9BQU8sRUFBRTtnQkFDeEIsTUFBTSxJQUFJLEtBQUssQ0FBQyxxQkFBcUIsSUFBSSxDQUFDLElBQUksUUFBUSxHQUFHLHlCQUF5QixPQUFPLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUM7YUFDakg7WUFFRCxJQUFJLFFBQVEsQ0FBQztZQUNiLElBQUksUUFBUSxDQUFDO1lBQ2IsVUFBSSxJQUFJLENBQUMsSUFBSSwwQ0FBRSxTQUFTLEVBQUU7Z0JBQ3hCLFFBQVEsU0FBRyxJQUFJLENBQUMsSUFBSSwwQ0FBRSxTQUFTLENBQUMsQ0FBQyxxQ0FBcUM7YUFDdkU7aUJBQU0sVUFBSSxJQUFJLENBQUMsSUFBSSwwQ0FBRSxHQUFHLEVBQUU7Z0JBQ3pCLFFBQVEsU0FBRyxJQUFJLENBQUMsSUFBSSwwQ0FBRSxHQUFHLENBQUMsS0FBSyxDQUFDLEdBQUcsRUFBRSxHQUFHLEVBQUUsQ0FBQyxDQUFDLGtEQUFrRDtnQkFDOUYsUUFBUSxHQUFHLElBQUksT0FBQyxJQUFJLENBQUMsSUFBSSwwQ0FBRSxHQUFHLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxvQ0FBb0M7YUFDM0U7aUJBQU0sRUFBRSwyQ0FBMkM7Z0JBQ2xELFFBQVEsR0FBRyxTQUFTLENBQUM7YUFDdEI7WUFFRCxNQUFNLFVBQVUsR0FBRyxRQUFRLElBQUksSUFBSSxDQUFDLFFBQVEsQ0FBQztZQUM3QyxJQUFJLFlBQVksU0FBRyxJQUFJLENBQUMsSUFBSSwwQ0FBRSxPQUFPLENBQUM7WUFFdEMsSUFBSSxZQUFZLEtBQUssV0FBVyxFQUFFO2dCQUNoQyxZQUFZLEdBQUcsU0FBUyxDQUFDO2FBQzFCO1lBRUQsNEdBQTRHO1lBQzVHLElBQUksQ0FBQyxVQUFVLElBQUksWUFBWSxFQUFFO2dCQUMvQixJQUFJLFFBQUMsSUFBSSxDQUFDLElBQUksMENBQUUsU0FBUyxDQUFBLEVBQUU7b0JBQ3pCLE1BQU0sSUFBSSxLQUFLLENBQUMsb0JBQW9CLElBQUksQ0FBQyxJQUFJLDBGQUEwRixRQUFRLEVBQUUsQ0FBQyxDQUFDO2lCQUNwSjtnQkFFRCxzQkFBc0IsQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLFlBQVksUUFBRSxJQUFJLENBQUMsSUFBSSwwQ0FBRSxTQUFTLENBQUMsQ0FBQzthQUN2RTtZQUVELE9BQU8sQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLEdBQUcsZUFBZSxDQUFDO2dCQUNuQyxJQUFJLEVBQUUsUUFBUTtnQkFDZCxNQUFNLEVBQUUsTUFBTSxDQUFDLElBQUk7Z0JBQ25CLElBQUksRUFBRSxJQUFJLENBQUMsSUFBSTtnQkFDZixHQUFHLFFBQUUsSUFBSSxDQUFDLElBQUksMENBQUUsR0FBRztnQkFDbkIsSUFBSSxFQUFFLElBQUksQ0FBQyxJQUFJLENBQUMsT0FBTztnQkFDdkIsSUFBSSxFQUFFLFFBQVE7Z0JBQ2QsSUFBSSxFQUFFLFFBQVE7Z0JBQ2QsTUFBTSxFQUFFLFFBQVEsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDLElBQUksRUFBRSxHQUFHLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUM7Z0JBQ3JFLE9BQU8sRUFBRSxZQUFZO2dCQUNyQixRQUFRLEVBQUUsVUFBVTtnQkFDcEIsUUFBUSxFQUFFLGFBQUEsSUFBSSxDQUFDLElBQUksMENBQUUsTUFBTSwwQ0FBRSxRQUFRLE1BQUssTUFBTTtnQkFDaEQsVUFBVSxFQUFFLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxLQUFLLFlBQVksQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxTQUFTO2FBQ3BFLENBQUMsQ0FBQztTQUNKO1FBRUQsS0FBSyxNQUFNLEdBQUcsVUFBSSxNQUFNLENBQUMsVUFBVSxtQ0FBSSxFQUFFLEVBQUU7WUFDekMsVUFBVSxDQUFDLEdBQUcsQ0FBQyxDQUFDO1NBQ2pCO0lBQ0gsQ0FBQztBQUNILENBQUM7QUFFRCxTQUFTLGVBQWUsQ0FBQyxHQUFRO0lBQy9CLE1BQU0sR0FBRyxHQUFRLEVBQUUsQ0FBQztJQUNwQixLQUFLLE1BQU0sQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLElBQUksTUFBTSxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsRUFBRTtRQUN4QyxJQUFJLENBQUMsS0FBSyxTQUFTLEVBQUU7WUFDbkIsR0FBRyxDQUFDLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQztTQUNaO0tBQ0Y7SUFDRCxPQUFPLEdBQUcsQ0FBQztBQUNiLENBQUM7QUFFRCxTQUFTLGFBQWEsQ0FBQyxJQUFlLEVBQUUsR0FBVzs7SUFDakQsTUFBTSxJQUFJLEdBQUcsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDO0lBRXZCLElBQUksSUFBSSxDQUFDLElBQUksS0FBSyxPQUFPLEVBQUU7UUFDekIsT0FBTyxLQUFLLENBQUM7S0FDZDtJQUNELElBQUksSUFBSSxDQUFDLFFBQVEsRUFBRTtRQUNqQixPQUFPLEtBQUssQ0FBQztLQUNkO0lBRUQsVUFBSSxJQUFJLENBQUMsSUFBSSwwQ0FBRSxVQUFVLEVBQUU7UUFDekIsT0FBTyxLQUFLLENBQUM7S0FDZDtJQUVELElBQUksSUFBSSxHQUFHLElBQUksQ0FBQztJQUNoQixPQUFPLElBQUksRUFBRTtRQUNYLElBQUksSUFBSSxDQUFDLEdBQUcsS0FBSyxnQkFBZ0IsRUFBRTtZQUNqQyxPQUFPLElBQUksQ0FBQztTQUNiO1FBRUQsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUU7WUFDZCxPQUFPLEtBQUssQ0FBQztTQUNkO1FBRUQsSUFBSSxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDdkIsSUFBSSxDQUFDLElBQUksRUFBRTtZQUNULE9BQU8sS0FBSyxDQUFDO1NBQ2Q7S0FDRjtBQUNILENBQUM7QUFFRCxTQUFTLHNCQUFzQixDQUFDLElBQVksRUFBRSxLQUFhLEVBQUUsSUFBWTtJQUN2RSwwQkFBMEI7SUFDMUIsSUFBSSxLQUFLLENBQUMsVUFBVSxDQUFDLEdBQUcsQ0FBQyxFQUFFO1FBQ3pCLE9BQU87S0FDUjtJQUNELElBQUk7UUFDRixNQUFNLE1BQU0sR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQ2pDLElBQUksT0FBTSxDQUFDLE1BQU0sQ0FBQyxLQUFLLElBQUksRUFBRTtZQUMzQixNQUFNLElBQUksS0FBSyxDQUFDLG9EQUFvRCxJQUFJLFNBQVMsSUFBSSxLQUFLLE1BQU0sRUFBRSxDQUFDLENBQUM7U0FDckc7S0FFRjtJQUFDLE9BQU8sQ0FBQyxFQUFFO1FBQ1YsTUFBTSxJQUFJLEtBQUssQ0FBQyxpQ0FBaUMsS0FBSyxpREFBaUQsSUFBSSxNQUFNLENBQUMsQ0FBQyxPQUFPLEVBQUUsQ0FBQyxDQUFDO0tBQy9IO0FBQ0gsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIHBhdGggZnJvbSAncGF0aCc7XG5pbXBvcnQgKiBhcyBmcyBmcm9tICdmcy1leHRyYSc7XG5cbi8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZSBAdHlwZXNjcmlwdC1lc2xpbnQvbm8tcmVxdWlyZS1pbXBvcnRzXG5jb25zdCBkZWNhbWVsaXplID0gcmVxdWlyZSgnZGVjYW1lbGl6ZScpO1xuY29uc3QgUFJPSkVOX01PRFVMRV9ST09UID0gcGF0aC5qb2luKF9fZGlybmFtZSwgJy4uJyk7XG5jb25zdCBQUk9KRUNUX0JBU0VfRlFOID0gJ3Byb2plbi5Qcm9qZWN0JztcblxudHlwZSBKc2lpVHlwZXMgPSB7IFtuYW1lOiBzdHJpbmddOiBKc2lpVHlwZSB9O1xuXG5leHBvcnQgaW50ZXJmYWNlIFByb2plY3RPcHRpb24ge1xuICBwYXRoOiBzdHJpbmdbXTtcbiAgbmFtZTogc3RyaW5nO1xuICBmcW4/OiBzdHJpbmc7XG4gIHN3aXRjaDogc3RyaW5nO1xuICB0eXBlOiBzdHJpbmc7XG4gIGtpbmQ/OiBzdHJpbmc7XG4gIHBhcmVudDogc3RyaW5nO1xuICBkb2NzPzogc3RyaW5nO1xuICBkZWZhdWx0Pzogc3RyaW5nO1xuICBvcHRpb25hbD86IGJvb2xlYW47XG4gIGRlcHJlY2F0ZWQ/OiBib29sZWFuO1xuICBmZWF0dXJlZD86IGJvb2xlYW47XG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgUHJvamVjdFR5cGUge1xuICBtb2R1bGVOYW1lOiBzdHJpbmc7XG4gIHBqaWQ6IHN0cmluZztcbiAgZnFuOiBzdHJpbmc7XG4gIHR5cGVuYW1lOiBzdHJpbmc7XG4gIG9wdGlvbnM6IFByb2plY3RPcHRpb25bXTtcbiAgZG9jcz86IHN0cmluZztcbiAgZG9jc3VybDogc3RyaW5nO1xufVxuXG5pbnRlcmZhY2UgSnNpaVR5cGUge1xuICBuYW1lOiBzdHJpbmc7XG4gIGFzc2VtYmx5OiBzdHJpbmc7XG4gIGtpbmQ6IHN0cmluZztcbiAgYWJzdHJhY3Q/OiBib29sZWFuO1xuICBiYXNlPzogc3RyaW5nO1xuICBmcW46IHN0cmluZztcbiAgaW50ZXJmYWNlcz86IHN0cmluZ1tdO1xuICBpbml0aWFsaXplcj86IHtcbiAgICBwYXJhbWV0ZXJzPzogQXJyYXk8e1xuICAgICAgbmFtZTogc3RyaW5nO1xuICAgICAgdHlwZT86IHsgZnFuPzogc3RyaW5nIH07XG4gICAgfT47XG4gIH07XG4gIHByb3BlcnRpZXM/OiBBcnJheTx7XG4gICAgbmFtZTogc3RyaW5nO1xuICAgIGRvY3M6IHtcbiAgICAgIHN1bW1hcnk/OiBzdHJpbmc7XG4gICAgICBkZWZhdWx0Pzogc3RyaW5nO1xuICAgICAgZGVwcmVjYXRlZD86IHN0cmluZztcbiAgICAgIHN0YWJpbGl0eT86IHN0cmluZztcbiAgICAgIGN1c3RvbT86IHsgW25hbWU6IHN0cmluZ106IHN0cmluZyB9O1xuICAgIH07XG4gICAgb3B0aW9uYWw/OiBib29sZWFuO1xuICAgIHR5cGU/OiB7XG4gICAgICBwcmltaXRpdmU/OiBzdHJpbmc7XG4gICAgICBmcW4/OiBzdHJpbmc7XG4gICAgfTtcbiAgfT47XG4gIGRvY3M/OiB7XG4gICAgc3VtbWFyeT86IHN0cmluZztcbiAgICBkZXByZWNhdGVkPzogc3RyaW5nO1xuICAgIGN1c3RvbT86IHtcbiAgICAgIHBqaWQ/OiBzdHJpbmc7XG4gICAgfTtcbiAgfTtcbn1cblxuLyoqXG4gKiBSZXR1cm5zIGEgbGlzdCBvZiBwcm9qZWN0IHR5cGVzIGV4cG9ydGVkIHRoZSBtb2R1bGVzIGRlZmluZWQgaW4gYG1vZHVsZURpcnNgLlxuICogVGhpcyBsaXN0IHdpbGwgYWx3YXlzIGFsc28gaW5jbHVkZSB0aGUgYnVpbHQtaW4gcHJvamVuIHByb2plY3QgdHlwZXMuXG4gKiBNb2R1bGVzIHdpdGhvdXQgYSAuanNpaSBtYW5pZmVzdCBhcmUgc2tpcHBlZC5cbiAqXG4gKiBAcGFyYW0gbW9kdWxlRGlycyBBIGxpc3Qgb2YgbnBtIG1vZHVsZSBkaXJlY3Rvcmllc1xuICovXG5leHBvcnQgZnVuY3Rpb24gZGlzY292ZXIoLi4ubW9kdWxlRGlyczogc3RyaW5nW10pIHtcblxuICBjb25zdCBqc2lpID0gZGlzY292ZXJKc2lpVHlwZXMoLi4ubW9kdWxlRGlycyk7XG5cbiAgY29uc3QgcmVzdWx0ID0gbmV3IEFycmF5PFByb2plY3RUeXBlPigpO1xuXG4gIGZvciAoY29uc3QgZnFuIG9mIE9iamVjdC5rZXlzKGpzaWkpKSB7XG4gICAgY29uc3QgcCA9IHRvUHJvamVjdFR5cGUoanNpaSwgZnFuKTtcbiAgICBpZiAoIXApIHtcbiAgICAgIGNvbnRpbnVlO1xuICAgIH1cblxuICAgIHJlc3VsdC5wdXNoKHApO1xuICB9XG5cbiAgcmV0dXJuIHJlc3VsdC5zb3J0KChyMSwgcjIpID0+IHIxLnBqaWQubG9jYWxlQ29tcGFyZShyMi5wamlkKSk7XG59XG5cbi8qKlxuICogUmVzb2x2ZSBhbGwganNpaSB0eXBlcyBmcm9tIEBtb2R1bGVzRGlycy5cbiAqIFdoZW4gYSBqc2lpIG1vZHVsZSBpcyBmb3VuZCBpdCB3aWxsIHJlY3VzaXZlbHkgbGlzdCB0aGUgdHlwZXMgZnJvbSB0aGUgZGVwZW5kYW50IG1vZHVsZSBhcyB3ZWxsXG4gKlxuICogQHBhcmFtIG1vZHVsZURpcnNcbiAqIEByZXR1cm5zXG4gKi9cbmZ1bmN0aW9uIGRpc2NvdmVySnNpaVR5cGVzKC4uLm1vZHVsZURpcnM6IHN0cmluZ1tdKSB7XG4gIGNvbnN0IGpzaWk6IEpzaWlUeXBlcyA9IHt9O1xuICBjb25zdCBkaXNjb3ZlcmVkTWFuaWZlc3RzOiBBcnJheTxzdHJpbmc+ID0gW107XG5cbiAgY29uc3QgZGlzY292ZXJKc2lpID0gKGRpcjogc3RyaW5nKSA9PiB7XG4gICAgY29uc3QganNpaUZpbGUgPSBwYXRoLmpvaW4oZGlyLCAnLmpzaWknKTtcbiAgICBpZiAoIWZzLmV4aXN0c1N5bmMoanNpaUZpbGUpKSB7IHJldHVybjsgfSAvLyBubyBqc2lpIG1hbmlmZXN0XG5cbiAgICBjb25zdCBtYW5pZmVzdCA9IGZzLnJlYWRKc29uU3luYyhqc2lpRmlsZSk7XG5cbiAgICBpZiAoZGlzY292ZXJlZE1hbmlmZXN0cy5pbmNsdWRlcyhtYW5pZmVzdC5maW5nZXJwcmludCkpIHtcbiAgICAgIHJldHVybjtcbiAgICB9XG4gICAgZGlzY292ZXJlZE1hbmlmZXN0cy5wdXNoKG1hbmlmZXN0LmZpbmdlcnByaW50KTtcblxuICAgIGZvciAoY29uc3QgW2ZxbiwgdHlwZV0gb2YgT2JqZWN0LmVudHJpZXMobWFuaWZlc3QudHlwZXMgYXMgSnNpaVR5cGVzKSkge1xuICAgICAganNpaVtmcW5dID0ge1xuICAgICAgICAuLi50eXBlLFxuICAgICAgfTtcbiAgICB9XG5cbiAgICAvLyBBbHNvIHNlYXJjaCByZWN1cnNpdmVseSBpbiBuZXN0ZWQgcHJvamVjdCBkZXBlbmRlbmNpZXMuIElmIHRoZSByZXF1ZXN0ZWQgbW9kdWxlIGlzIGFuIGV4dGVybmFsIG1vZHVsZVxuICAgIC8vIHRoaXMgd2lsbCBhbHNvIGVuZC11cCBpbiB0aGUgcHJvamVuIG1vZHVsZSBhbmQgYWRkIHRoZSBwcm9qZW4gdHlwZXNcbiAgICBpZiAobWFuaWZlc3QuZGVwZW5kZW5jaWVzKSB7XG4gICAgICBmb3IgKGNvbnN0IGRlcGVuZGVuY3kgb2YgT2JqZWN0LmtleXMobWFuaWZlc3QuZGVwZW5kZW5jaWVzKSkge1xuICAgICAgICBjb25zdCBuZXN0ZWREZXBlbmRlbmN5Rm9sZGVyID0gcGF0aC5kaXJuYW1lKHJlcXVpcmUucmVzb2x2ZShgJHtkZXBlbmRlbmN5fS9wYWNrYWdlLmpzb25gLCB7XG4gICAgICAgICAgcGF0aHM6IFtkaXJdLFxuICAgICAgICB9KSk7XG5cbiAgICAgICAgaWYgKGZzLmV4aXN0c1N5bmMobmVzdGVkRGVwZW5kZW5jeUZvbGRlcikpIHtcbiAgICAgICAgICBkaXNjb3ZlckpzaWkobmVzdGVkRGVwZW5kZW5jeUZvbGRlcik7XG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9XG4gIH07XG5cbiAgLy8gcmVhZCBhbGwgLmpzaWkgbWFuaWZlc3RzIGZyb20gYWxsIHJlcXVlc3RlZCBtb2R1bGVzIGFuZCBtZXJnZVxuICAvLyB0aGVtIGFsbCBpbnRvIGEgc2luZ2xlIG1hcCBvZiBmcW4tPnR5cGUuXG4gIGZvciAoY29uc3QgZGlyIG9mIFsuLi5tb2R1bGVEaXJzLCBQUk9KRU5fTU9EVUxFX1JPT1RdKSB7XG4gICAgZGlzY292ZXJKc2lpKGRpcik7XG5cbiAgICAvLyBSZWFkIGZyb20gc2NvcGVkIHBhY2thZ2VzXG4gICAgaWYgKGRpci5pbmNsdWRlcygnQCcpICYmIGZzLmxzdGF0U3luYyhkaXIpLmlzRGlyZWN0b3J5KCkpIHtcbiAgICAgIGNvbnN0IGNoaWxkRGlycyA9IGZzLnJlYWRkaXJTeW5jKGRpcikubWFwKGZpbGUgPT4gcGF0aC5qb2luKGRpciwgZmlsZSkpO1xuICAgICAgZm9yIChjb25zdCBjaGlsZCBvZiBjaGlsZERpcnMpIHtcbiAgICAgICAgZGlzY292ZXJKc2lpKGNoaWxkKTtcbiAgICAgIH1cbiAgICB9XG4gIH1cblxuICByZXR1cm4ganNpaTtcbn1cblxuZXhwb3J0IGZ1bmN0aW9uIHJlc29sdmVQcm9qZWN0VHlwZShwcm9qZWN0RnFuOiBzdHJpbmcpIHtcbiAgbGV0IFttb2R1bGVOYW1lXSA9IHByb2plY3RGcW4uc3BsaXQoJy4nKTtcbiAgaWYgKG1vZHVsZU5hbWUgPT09ICdwcm9qZW4nKSB7XG4gICAgbW9kdWxlTmFtZSA9IFBST0pFTl9NT0RVTEVfUk9PVDtcbiAgfVxuXG4gIC8vIHRyeSBwaWNraW5nIHRoZSBtYW5pZmVzdC4gV2Ugb25seSBuZWVkIHRoZSBiYXNlIGZvbGRlciBidXQgdGhpcyBpcyBkaXJlY3RseSBhIG5pY2UgY2hlY2sgaWYgd2UgcmVxdWVzdCBmcm9tIGEgdmFsaWQganNpaSBwYWNrYWdlXG4gIGNvbnN0IGpzaWlNYW5pZmVzdEZpbGUgPSByZXF1aXJlLnJlc29sdmUoYCR7bW9kdWxlTmFtZX0vLmpzaWlgLCB7XG4gICAgcGF0aHM6IFtcbiAgICAgIHByb2Nlc3MuY3dkKCksXG4gICAgXSxcbiAgfSk7XG4gIGNvbnN0IG1vZHVsZUZvbGRlciA9IHBhdGguZGlybmFtZShqc2lpTWFuaWZlc3RGaWxlKTtcblxuICAvLyBSZWFkIGFsbCBqc2lpIHR5cGVzIHRoYXQgY2FuIGJlIGxvYWRlZCBmcm9tIHRoaXMgcHJvamVjdCB0eXBlXG4gIGNvbnN0IGpzaWkgPSBkaXNjb3ZlckpzaWlUeXBlcyhtb2R1bGVGb2xkZXIpO1xuICByZXR1cm4gdG9Qcm9qZWN0VHlwZShqc2lpLCBwcm9qZWN0RnFuKTtcbn1cblxuZnVuY3Rpb24gdG9Qcm9qZWN0VHlwZShqc2lpOiBKc2lpVHlwZXMsIGZxbjogc3RyaW5nKSB7XG4gIGlmICghaXNQcm9qZWN0VHlwZShqc2lpLCBmcW4pKSB7XG4gICAgcmV0dXJuIHVuZGVmaW5lZDtcbiAgfVxuXG4gIGNvbnN0IHR5cGVpbmZvID0ganNpaVtmcW5dO1xuXG4gIC8vIHByb2plbi53ZWIuUmVhY3RQcm9qZWN0IC0+IHdlYi5SZWFjdFByb2plY3RcbiAgY29uc3QgdHlwZW5hbWUgPSBmcW4uc3Vic3RyaW5nKGZxbi5pbmRleE9mKCcuJykgKyAxKTtcblxuICBjb25zdCBkb2NzdXJsID0gYGh0dHBzOi8vZ2l0aHViLmNvbS9wcm9qZW4vcHJvamVuL2Jsb2IvbWFpbi9BUEkubWQjcHJvamVuLSR7dHlwZW5hbWUudG9Mb2NhbGVMb3dlckNhc2UoKS5yZXBsYWNlKC9cXC4vZywgJy0nKX1gO1xuICBsZXQgcGppZCA9IHR5cGVpbmZvLmRvY3M/LmN1c3RvbT8ucGppZCA/PyBkZWNhbWVsaXplKHR5cGVuYW1lKS5yZXBsYWNlKC9fcHJvamVjdCQvLCAnJyk7XG4gIHJldHVybiB7XG4gICAgbW9kdWxlTmFtZTogdHlwZWluZm8uYXNzZW1ibHksXG4gICAgdHlwZW5hbWUsXG4gICAgcGppZCxcbiAgICBmcW4sXG4gICAgb3B0aW9uczogZGlzY292ZXJPcHRpb25zKGpzaWksIGZxbikuc29ydCgobzEsIG8yKSA9PiBvMS5uYW1lLmxvY2FsZUNvbXBhcmUobzIubmFtZSkpLFxuICAgIGRvY3M6IHR5cGVpbmZvLmRvY3M/LnN1bW1hcnksXG4gICAgZG9jc3VybCxcbiAgfSBhcyBQcm9qZWN0VHlwZTtcbn1cblxuZXhwb3J0IGZ1bmN0aW9uIHJlYWRKc2lpTWFuaWZlc3QoanNpaUZxbjogc3RyaW5nKTogYW55IHtcbiAgbGV0IFttb2R1bGVOYW1lXSA9IGpzaWlGcW4uc3BsaXQoJy4nKTtcbiAgaWYgKG1vZHVsZU5hbWUgPT09ICdwcm9qZW4nKSB7XG4gICAgbW9kdWxlTmFtZSA9IFBST0pFTl9NT0RVTEVfUk9PVDtcbiAgfVxuXG4gIGNvbnN0IGpzaWlNYW5pZmVzdEZpbGUgPSByZXF1aXJlLnJlc29sdmUoYCR7bW9kdWxlTmFtZX0vLmpzaWlgKTtcbiAgcmV0dXJuIGZzLnJlYWRKc29uU3luYyhqc2lpTWFuaWZlc3RGaWxlKTtcbn1cblxuZnVuY3Rpb24gZGlzY292ZXJPcHRpb25zKGpzaWk6IEpzaWlUeXBlcywgZnFuOiBzdHJpbmcpOiBQcm9qZWN0T3B0aW9uW10ge1xuICBjb25zdCBvcHRpb25zOiB7IFtuYW1lOiBzdHJpbmddOiBQcm9qZWN0T3B0aW9uIH0gPSB7fTtcbiAgY29uc3QgcGFyYW1zID0ganNpaVtmcW5dPy5pbml0aWFsaXplcj8ucGFyYW1ldGVycyA/PyBbXTtcbiAgY29uc3Qgb3B0aW9uc1BhcmFtID0gcGFyYW1zWzBdO1xuICBjb25zdCBvcHRpb25zVHlwZUZxbiA9IG9wdGlvbnNQYXJhbT8udHlwZT8uZnFuO1xuXG4gIGlmIChwYXJhbXMubGVuZ3RoID4gMSB8fCAocGFyYW1zLmxlbmd0aCA9PT0gMSAmJiBvcHRpb25zUGFyYW0/Lm5hbWUgIT09ICdvcHRpb25zJykpIHtcbiAgICB0aHJvdyBuZXcgRXJyb3IoYGNvbnN0cnVjdG9yIGZvciBwcm9qZWN0ICR7ZnFufSBtdXN0IGhhdmUgYSBzaW5nbGUgXCJvcHRpb25zXCIgYXJndW1lbnQgb2YgYSBzdHJ1Y3QgdHlwZS4gZ290ICR7SlNPTi5zdHJpbmdpZnkocGFyYW1zKX1gKTtcbiAgfVxuXG4gIGFkZE9wdGlvbnMob3B0aW9uc1R5cGVGcW4pO1xuXG4gIGNvbnN0IG9wdHMgPSBPYmplY3QudmFsdWVzKG9wdGlvbnMpO1xuXG4gIHJldHVybiBvcHRzLnNvcnQoKGEsIGIpID0+IGEuc3dpdGNoLmxvY2FsZUNvbXBhcmUoYi5zd2l0Y2gpKTtcblxuICBmdW5jdGlvbiBhZGRPcHRpb25zKG9mcW4/OiBzdHJpbmcsIGJhc2VQYXRoOiBzdHJpbmdbXSA9IFtdLCBvcHRpb25hbCA9IGZhbHNlKSB7XG4gICAgaWYgKCFvZnFuKSB7XG4gICAgICByZXR1cm47XG4gICAgfVxuXG4gICAgY29uc3Qgc3RydWN0ID0ganNpaVtvZnFuXTtcbiAgICBpZiAoIXN0cnVjdCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGB1bmFibGUgdG8gZmluZCBvcHRpb25zIHR5cGUgJHtvZnFufSBmb3IgcHJvamVjdCAke2Zxbn1gKTtcbiAgICB9XG5cbiAgICBmb3IgKGNvbnN0IHByb3Agb2Ygc3RydWN0LnByb3BlcnRpZXMgPz8gW10pIHtcbiAgICAgIGNvbnN0IHByb3BQYXRoID0gWy4uLmJhc2VQYXRoLCBwcm9wLm5hbWVdO1xuXG4gICAgICAvLyBwcm90ZWN0IGFnYWluc3QgZG91YmxlLWJvb2tpbmdcbiAgICAgIGlmIChwcm9wLm5hbWUgaW4gb3B0aW9ucykge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYGR1cGxpY2F0ZSBvcHRpb24gXCIke3Byb3AubmFtZX1cIiBpbiAke2Zxbn0gKGFscmVhZHkgZGVjbGFyZWQgaW4gJHtvcHRpb25zW3Byb3AubmFtZV0ucGFyZW50fSlgKTtcbiAgICAgIH1cblxuICAgICAgbGV0IHR5cGVOYW1lO1xuICAgICAgbGV0IGpzaWlLaW5kO1xuICAgICAgaWYgKHByb3AudHlwZT8ucHJpbWl0aXZlKSB7XG4gICAgICAgIHR5cGVOYW1lID0gcHJvcC50eXBlPy5wcmltaXRpdmU7IC8vIGUuZy4gJ3N0cmluZycsICdib29sZWFuJywgJ251bWJlcidcbiAgICAgIH0gZWxzZSBpZiAocHJvcC50eXBlPy5mcW4pIHtcbiAgICAgICAgdHlwZU5hbWUgPSBwcm9wLnR5cGU/LmZxbi5zcGxpdCgnLicpLnBvcCgpOyAvLyBwcm9qZW4uTm9kZVByb2plY3RPcHRpb25zIC0+IE5vZGVQcm9qZWN0T3B0aW9uc1xuICAgICAgICBqc2lpS2luZCA9IGpzaWlbcHJvcC50eXBlPy5mcW5dLmtpbmQ7IC8vIGUuZy4gJ2NsYXNzJywgJ2ludGVyZmFjZScsICdlbnVtJ1xuICAgICAgfSBlbHNlIHsgLy8gYW55IG90aGVyIHR5cGVzIHN1Y2ggYXMgY29sbGVjdGlvbiB0eXBlc1xuICAgICAgICB0eXBlTmFtZSA9ICd1bmtub3duJztcbiAgICAgIH1cblxuICAgICAgY29uc3QgaXNPcHRpb25hbCA9IG9wdGlvbmFsIHx8IHByb3Aub3B0aW9uYWw7XG4gICAgICBsZXQgZGVmYXVsdFZhbHVlID0gcHJvcC5kb2NzPy5kZWZhdWx0O1xuXG4gICAgICBpZiAoZGVmYXVsdFZhbHVlID09PSAndW5kZWZpbmVkJykge1xuICAgICAgICBkZWZhdWx0VmFsdWUgPSB1bmRlZmluZWQ7XG4gICAgICB9XG5cbiAgICAgIC8vIGlmIHRoaXMgaXMgYSBtYW5kYXRvcnkgb3B0aW9uIGFuZCB3ZSBoYXZlIGEgZGVmYXVsdCB2YWx1ZSwgaXQgaGFzIHRvIGJlIEpTT04tcGFyc2FibGUgdG8gdGhlIGNvcnJlY3QgdHlwZVxuICAgICAgaWYgKCFpc09wdGlvbmFsICYmIGRlZmF1bHRWYWx1ZSkge1xuICAgICAgICBpZiAoIXByb3AudHlwZT8ucHJpbWl0aXZlKSB7XG4gICAgICAgICAgdGhyb3cgbmV3IEVycm9yKGByZXF1aXJlZCBvcHRpb24gXCIke3Byb3AubmFtZX1cIiB3aXRoIGEgQGRlZmF1bHQgbXVzdCB1c2UgcHJpbWl0aXZlIHR5cGVzIChzdHJpbmcsIG51bWJlciBvciBib29sZWFuKS4gdHlwZSBmb3VuZCBpczogJHt0eXBlTmFtZX1gKTtcbiAgICAgICAgfVxuXG4gICAgICAgIGNoZWNrRGVmYXVsdElzUGFyc2FibGUocHJvcC5uYW1lLCBkZWZhdWx0VmFsdWUsIHByb3AudHlwZT8ucHJpbWl0aXZlKTtcbiAgICAgIH1cblxuICAgICAgb3B0aW9uc1twcm9wLm5hbWVdID0gZmlsdGVyVW5kZWZpbmVkKHtcbiAgICAgICAgcGF0aDogcHJvcFBhdGgsXG4gICAgICAgIHBhcmVudDogc3RydWN0Lm5hbWUsXG4gICAgICAgIG5hbWU6IHByb3AubmFtZSxcbiAgICAgICAgZnFuOiBwcm9wLnR5cGU/LmZxbixcbiAgICAgICAgZG9jczogcHJvcC5kb2NzLnN1bW1hcnksXG4gICAgICAgIHR5cGU6IHR5cGVOYW1lLFxuICAgICAgICBraW5kOiBqc2lpS2luZCxcbiAgICAgICAgc3dpdGNoOiBwcm9wUGF0aC5tYXAocCA9PiBkZWNhbWVsaXplKHApLnJlcGxhY2UoL18vZywgJy0nKSkuam9pbignLScpLFxuICAgICAgICBkZWZhdWx0OiBkZWZhdWx0VmFsdWUsXG4gICAgICAgIG9wdGlvbmFsOiBpc09wdGlvbmFsLFxuICAgICAgICBmZWF0dXJlZDogcHJvcC5kb2NzPy5jdXN0b20/LmZlYXR1cmVkID09PSAndHJ1ZScsXG4gICAgICAgIGRlcHJlY2F0ZWQ6IHByb3AuZG9jcy5zdGFiaWxpdHkgPT09ICdkZXByZWNhdGVkJyA/IHRydWUgOiB1bmRlZmluZWQsXG4gICAgICB9KTtcbiAgICB9XG5cbiAgICBmb3IgKGNvbnN0IGlmYyBvZiBzdHJ1Y3QuaW50ZXJmYWNlcyA/PyBbXSkge1xuICAgICAgYWRkT3B0aW9ucyhpZmMpO1xuICAgIH1cbiAgfVxufVxuXG5mdW5jdGlvbiBmaWx0ZXJVbmRlZmluZWQob2JqOiBhbnkpIHtcbiAgY29uc3QgcmV0OiBhbnkgPSB7fTtcbiAgZm9yIChjb25zdCBbaywgdl0gb2YgT2JqZWN0LmVudHJpZXMob2JqKSkge1xuICAgIGlmICh2ICE9PSB1bmRlZmluZWQpIHtcbiAgICAgIHJldFtrXSA9IHY7XG4gICAgfVxuICB9XG4gIHJldHVybiByZXQ7XG59XG5cbmZ1bmN0aW9uIGlzUHJvamVjdFR5cGUoanNpaTogSnNpaVR5cGVzLCBmcW46IHN0cmluZykge1xuICBjb25zdCB0eXBlID0ganNpaVtmcW5dO1xuXG4gIGlmICh0eXBlLmtpbmQgIT09ICdjbGFzcycpIHtcbiAgICByZXR1cm4gZmFsc2U7XG4gIH1cbiAgaWYgKHR5cGUuYWJzdHJhY3QpIHtcbiAgICByZXR1cm4gZmFsc2U7XG4gIH1cblxuICBpZiAodHlwZS5kb2NzPy5kZXByZWNhdGVkKSB7XG4gICAgcmV0dXJuIGZhbHNlO1xuICB9XG5cbiAgbGV0IGN1cnIgPSB0eXBlO1xuICB3aGlsZSAodHJ1ZSkge1xuICAgIGlmIChjdXJyLmZxbiA9PT0gUFJPSkVDVF9CQVNFX0ZRTikge1xuICAgICAgcmV0dXJuIHRydWU7XG4gICAgfVxuXG4gICAgaWYgKCFjdXJyLmJhc2UpIHtcbiAgICAgIHJldHVybiBmYWxzZTtcbiAgICB9XG5cbiAgICBjdXJyID0ganNpaVtjdXJyLmJhc2VdO1xuICAgIGlmICghY3Vycikge1xuICAgICAgcmV0dXJuIGZhbHNlO1xuICAgIH1cbiAgfVxufVxuXG5mdW5jdGlvbiBjaGVja0RlZmF1bHRJc1BhcnNhYmxlKHByb3A6IHN0cmluZywgdmFsdWU6IHN0cmluZywgdHlwZTogc3RyaW5nKSB7XG4gIC8vIG1hY3JvcyBhcmUgcGFzcy10aHJvdWdoXG4gIGlmICh2YWx1ZS5zdGFydHNXaXRoKCckJykpIHtcbiAgICByZXR1cm47XG4gIH1cbiAgdHJ5IHtcbiAgICBjb25zdCBwYXJzZWQgPSBKU09OLnBhcnNlKHZhbHVlKTtcbiAgICBpZiAodHlwZW9mKHBhcnNlZCkgIT09IHR5cGUpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgY2Fubm90IHBhcnNlIEBkZWZhdWx0IHZhbHVlIGZvciBtYW5kYXRvcnkgb3B0aW9uICR7cHJvcH0gYXMgYSAke3R5cGV9OiAke3BhcnNlZH1gKTtcbiAgICB9XG5cbiAgfSBjYXRjaCAoZSkge1xuICAgIHRocm93IG5ldyBFcnJvcihgdW5hYmxlIHRvIEpTT04ucGFyc2UoKSB2YWx1ZSBcIiR7dmFsdWV9XCIgc3BlY2lmaWVkIGFzIEBkZWZhdWx0IGZvciBtYW5kYXRvcnkgb3B0aW9uIFwiJHtwcm9wfVwiOiAke2UubWVzc2FnZX1gKTtcbiAgfVxufVxuIl19