"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Eslint = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const common_1 = require("./common");
const component_1 = require("./component");
const json_1 = require("./json");
/**
 * @experimental
 */
class Eslint extends component_1.Component {
    /**
     * @experimental
     */
    constructor(project, options) {
        var _b, _c, _d, _e, _f, _g;
        super(project);
        project.addDevDeps('eslint', '@typescript-eslint/eslint-plugin', '@typescript-eslint/parser', 'eslint-import-resolver-node', 'eslint-import-resolver-typescript', 'eslint-plugin-import', 'json-schema');
        if (options.prettier) {
            project.addDevDeps('prettier', 'eslint-plugin-prettier', 'eslint-config-prettier');
        }
        if (options.aliasMap) {
            project.addDevDeps('eslint-import-resolver-alias');
        }
        const devdirs = (_b = options.devdirs) !== null && _b !== void 0 ? _b : [];
        const dirs = [...options.dirs, ...devdirs];
        const fileExtensions = (_c = options.fileExtensions) !== null && _c !== void 0 ? _c : ['.ts'];
        this._allowDevDeps = (devdirs !== null && devdirs !== void 0 ? devdirs : []).map(dir => `**/${dir}/**`);
        const lintProjenRc = (_d = options.lintProjenRc) !== null && _d !== void 0 ? _d : true;
        const eslint = project.addTask('eslint', {
            description: 'Runs eslint against the codebase',
            exec: [
                'eslint',
                `--ext ${fileExtensions.join(',')}`,
                '--fix',
                '--no-error-on-unmatched-pattern',
                ...dirs,
                ...lintProjenRc ? [common_1.PROJEN_RC] : [],
            ].join(' '),
        });
        project.testTask.spawn(eslint);
        // exclude some files
        (_e = project.npmignore) === null || _e === void 0 ? void 0 : _e.exclude('/.eslintrc.json');
        const formattingRules = options.prettier ? {
            'prettier/prettier': ['error'],
        } : {
            // see https://github.com/typescript-eslint/typescript-eslint/blob/master/packages/eslint-plugin/docs/rules/indent.md
            'indent': ['off'],
            '@typescript-eslint/indent': ['error', 2],
            // Style
            'quotes': ['error', 'single', { avoidEscape: true }],
            'comma-dangle': ['error', 'always-multiline'],
            'comma-spacing': ['error', { before: false, after: true }],
            'no-multi-spaces': ['error', { ignoreEOLComments: false }],
            'array-bracket-spacing': ['error', 'never'],
            'array-bracket-newline': ['error', 'consistent'],
            'object-curly-spacing': ['error', 'always'],
            'object-curly-newline': ['error', { multiline: true, consistent: true }],
            'object-property-newline': ['error', { allowAllPropertiesOnSameLine: true }],
            'keyword-spacing': ['error'],
            'brace-style': ['error', '1tbs', { allowSingleLine: true }],
            'space-before-blocks': ['error'],
            'curly': ['error', 'multi-line', 'consistent'],
            '@typescript-eslint/member-delimiter-style': ['error'],
            // Require semicolons
            'semi': ['error', 'always'],
            // Max line lengths
            'max-len': ['error', {
                    code: 150,
                    ignoreUrls: true,
                    ignoreStrings: true,
                    ignoreTemplateLiterals: true,
                    ignoreComments: true,
                    ignoreRegExpLiterals: true,
                }],
            // Don't unnecessarily quote properties
            'quote-props': ['error', 'consistent-as-needed'],
        };
        this.rules = {
            ...formattingRules,
            // Require use of the `import { foo } from 'bar';` form instead of `import foo = require('bar');`
            '@typescript-eslint/no-require-imports': ['error'],
            // Require all imported dependencies are actually declared in package.json
            'import/no-extraneous-dependencies': [
                'error',
                {
                    // Only allow importing devDependencies from "devdirs".
                    devDependencies: () => this.renderDevDepsAllowList(),
                    optionalDependencies: false,
                    peerDependencies: true,
                },
            ],
            // Require all imported libraries actually resolve (!!required for import/no-extraneous-dependencies to work!!)
            'import/no-unresolved': ['error'],
            // Require an ordering on all imports
            'import/order': ['warn', {
                    groups: ['builtin', 'external'],
                    alphabetize: { order: 'asc', caseInsensitive: true },
                }],
            // Cannot import from the same module twice
            'no-duplicate-imports': ['error'],
            // Cannot shadow names
            'no-shadow': ['off'],
            '@typescript-eslint/no-shadow': ['error'],
            // Required spacing in property declarations (copied from TSLint, defaults are good)
            'key-spacing': ['error'],
            // No multiple empty lines
            'no-multiple-empty-lines': ['error'],
            // One of the easiest mistakes to make
            '@typescript-eslint/no-floating-promises': ['error'],
            // Make sure that inside try/catch blocks, promises are 'return await'ed
            // (must disable the base rule as it can report incorrect errors)
            'no-return-await': ['off'],
            '@typescript-eslint/return-await': ['error'],
            // Useless diff results
            'no-trailing-spaces': ['error'],
            // Must use foo.bar instead of foo['bar'] if possible
            'dot-notation': ['error'],
            // Are you sure | is not a typo for || ?
            'no-bitwise': ['error'],
            // Member ordering
            '@typescript-eslint/member-ordering': ['error', {
                    default: [
                        'public-static-field',
                        'public-static-method',
                        'protected-static-field',
                        'protected-static-method',
                        'private-static-field',
                        'private-static-method',
                        'field',
                        // Constructors
                        'constructor',
                        // Methods
                        'method',
                    ],
                }],
        };
        // Overrides for .projenrc.js
        this.overrides = [
            {
                files: [common_1.PROJEN_RC],
                rules: {
                    '@typescript-eslint/no-require-imports': 'off',
                    'import/no-extraneous-dependencies': 'off',
                },
            },
        ];
        this.ignorePatterns = (_f = options.ignorePatterns) !== null && _f !== void 0 ? _f : [
            '*.js',
            `!${common_1.PROJEN_RC}`,
            '*.d.ts',
            'node_modules/',
            '*.generated.ts',
            'coverage',
        ];
        const tsconfig = (_g = options.tsconfigPath) !== null && _g !== void 0 ? _g : './tsconfig.json';
        const plugins = [
            '@typescript-eslint',
            'import',
            ...(options.prettier ? ['prettier'] : []),
        ];
        const extendsConf = [
            'plugin:import/typescript',
            ...(options.prettier ? [
                'prettier',
                'plugin:prettier/recommended',
            ] : []),
        ];
        this.config = {
            env: {
                jest: true,
                node: true,
            },
            root: true,
            plugins,
            parser: '@typescript-eslint/parser',
            parserOptions: {
                ecmaVersion: 2018,
                sourceType: 'module',
                project: tsconfig,
            },
            extends: extendsConf,
            settings: {
                'import/parsers': {
                    '@typescript-eslint/parser': ['.ts', '.tsx'],
                },
                'import/resolver': {
                    ...(options.aliasMap && {
                        alias: {
                            map: Object.entries(options.aliasMap).map(([k, v]) => [k, v]),
                            extensions: options.aliasExtensions,
                        },
                    }),
                    node: {},
                    typescript: {
                        project: tsconfig,
                    },
                },
            },
            ignorePatterns: this.ignorePatterns,
            rules: this.rules,
            overrides: this.overrides,
        };
        new json_1.JsonFile(project, '.eslintrc.json', { obj: this.config, marker: false });
    }
    /**
     * (experimental) Add an eslint rule.
     *
     * @experimental
     */
    addRules(rules) {
        for (const [k, v] of Object.entries(rules)) {
            this.rules[k] = v;
        }
    }
    /**
     * (experimental) Add an eslint override.
     *
     * @experimental
     */
    addOverride(override) {
        this.overrides.push(override);
    }
    /**
     * (experimental) Do not lint these files.
     *
     * @experimental
     */
    addIgnorePattern(pattern) {
        this.ignorePatterns.push(pattern);
    }
    /**
     * (experimental) Add a glob file pattern which allows importing dev dependencies.
     *
     * @param pattern glob pattern.
     * @experimental
     */
    allowDevDeps(pattern) {
        this._allowDevDeps.push(pattern);
    }
    renderDevDepsAllowList() {
        return this._allowDevDeps;
    }
}
exports.Eslint = Eslint;
_a = JSII_RTTI_SYMBOL_1;
Eslint[_a] = { fqn: "projen.Eslint", version: "0.29.3" };
//# sourceMappingURL=data:application/json;base64,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