"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.discoverTaskCommands = void 0;
const chalk = require("chalk");
const logging = require("../logging");
/**
 * Reads .projen/tasks.json and adds CLI commands for all tasks.
 * @param ya yargs
 */
function discoverTaskCommands(runtime, ya) {
    var _a, _b;
    const tasks = (_a = runtime.manifest.tasks) !== null && _a !== void 0 ? _a : {};
    for (const task of Object.values(tasks)) {
        ya.command(task.name, (_b = task.description) !== null && _b !== void 0 ? _b : '', taskCommandHandler(task.name));
    }
    function taskCommandHandler(taskName) {
        return (args) => {
            args.option('inspect', { alias: 'i', desc: 'show all steps in this task' });
            const argv = args.argv;
            if (argv.inspect) {
                return inspectTask(taskName);
            }
            else {
                try {
                    runtime.runTask(taskName);
                }
                catch (e) {
                    logging.error(e.message);
                    process.exit(1);
                }
            }
        };
    }
    function inspectTask(name, indent = 0) {
        var _a, _b;
        const writeln = (s) => console.log(' '.repeat(indent) + s);
        const task = runtime.tryFindTask(name);
        if (!task) {
            throw new Error(`${name}: unable to resolve subtask with name "${name}"`);
        }
        if (task.description) {
            writeln(`${chalk.underline('description')}: ${task.description}`);
        }
        for (const [k, v] of Object.entries((_a = task.env) !== null && _a !== void 0 ? _a : {})) {
            writeln(`${chalk.underline('env')}: ${k}=${v}`);
        }
        for (const step of (_b = task.steps) !== null && _b !== void 0 ? _b : []) {
            if (step.spawn) {
                writeln(`- ${chalk.bold(step.spawn)}`);
                inspectTask(step.spawn, indent + 2);
            }
            else if (step.exec) {
                writeln(`- exec: ${step.exec}`);
            }
            else if (step.builtin) {
                writeln(`- builtin: ${step.builtin}`);
            }
        }
    }
}
exports.discoverTaskCommands = discoverTaskCommands;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGFza3MuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi9zcmMvY2xpL3Rhc2tzLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLCtCQUErQjtBQUUvQixzQ0FBc0M7QUFHdEM7OztHQUdHO0FBQ0gsU0FBZ0Isb0JBQW9CLENBQUMsT0FBb0IsRUFBRSxFQUFjOztJQUN2RSxNQUFNLEtBQUssU0FBRyxPQUFPLENBQUMsUUFBUSxDQUFDLEtBQUssbUNBQUksRUFBRSxDQUFDO0lBQzNDLEtBQUssTUFBTSxJQUFJLElBQUksTUFBTSxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsRUFBRTtRQUN2QyxFQUFFLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxJQUFJLFFBQUUsSUFBSSxDQUFDLFdBQVcsbUNBQUksRUFBRSxFQUFFLGtCQUFrQixDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDO0tBQzlFO0lBRUQsU0FBUyxrQkFBa0IsQ0FBQyxRQUFnQjtRQUMxQyxPQUFPLENBQUMsSUFBZ0IsRUFBRSxFQUFFO1lBQzFCLElBQUksQ0FBQyxNQUFNLENBQUMsU0FBUyxFQUFFLEVBQUUsS0FBSyxFQUFFLEdBQUcsRUFBRSxJQUFJLEVBQUUsNkJBQTZCLEVBQUUsQ0FBQyxDQUFDO1lBRTVFLE1BQU0sSUFBSSxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUM7WUFFdkIsSUFBSSxJQUFJLENBQUMsT0FBTyxFQUFFO2dCQUNoQixPQUFPLFdBQVcsQ0FBQyxRQUFRLENBQUMsQ0FBQzthQUM5QjtpQkFBTTtnQkFDTCxJQUFJO29CQUNGLE9BQU8sQ0FBQyxPQUFPLENBQUMsUUFBUSxDQUFDLENBQUM7aUJBQzNCO2dCQUFDLE9BQU8sQ0FBQyxFQUFFO29CQUNWLE9BQU8sQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxDQUFDO29CQUN6QixPQUFPLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDO2lCQUNqQjthQUNGO1FBQ0gsQ0FBQyxDQUFDO0lBQ0osQ0FBQztJQUVELFNBQVMsV0FBVyxDQUFDLElBQVksRUFBRSxNQUFNLEdBQUcsQ0FBQzs7UUFDM0MsTUFBTSxPQUFPLEdBQUcsQ0FBQyxDQUFTLEVBQUUsRUFBRSxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsR0FBRyxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQztRQUVuRSxNQUFNLElBQUksR0FBRyxPQUFPLENBQUMsV0FBVyxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQ3ZDLElBQUksQ0FBQyxJQUFJLEVBQUU7WUFDVCxNQUFNLElBQUksS0FBSyxDQUFDLEdBQUcsSUFBSSwwQ0FBMEMsSUFBSSxHQUFHLENBQUMsQ0FBQztTQUMzRTtRQUVELElBQUksSUFBSSxDQUFDLFdBQVcsRUFBRTtZQUNwQixPQUFPLENBQUMsR0FBRyxLQUFLLENBQUMsU0FBUyxDQUFDLGFBQWEsQ0FBQyxLQUFLLElBQUksQ0FBQyxXQUFXLEVBQUUsQ0FBQyxDQUFDO1NBQ25FO1FBRUQsS0FBSyxNQUFNLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxJQUFJLE1BQU0sQ0FBQyxPQUFPLE9BQUMsSUFBSSxDQUFDLEdBQUcsbUNBQUksRUFBRSxDQUFDLEVBQUU7WUFDbkQsT0FBTyxDQUFDLEdBQUcsS0FBSyxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsQ0FBQztTQUNqRDtRQUVELEtBQUssTUFBTSxJQUFJLFVBQUksSUFBSSxDQUFDLEtBQUssbUNBQUksRUFBRSxFQUFFO1lBQ25DLElBQUksSUFBSSxDQUFDLEtBQUssRUFBRTtnQkFDZCxPQUFPLENBQUMsS0FBSyxLQUFLLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDLENBQUM7Z0JBQ3ZDLFdBQVcsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLE1BQU0sR0FBRyxDQUFDLENBQUMsQ0FBQzthQUNyQztpQkFBTSxJQUFJLElBQUksQ0FBQyxJQUFJLEVBQUU7Z0JBQ3BCLE9BQU8sQ0FBQyxXQUFXLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQyxDQUFDO2FBQ2pDO2lCQUFNLElBQUksSUFBSSxDQUFDLE9BQU8sRUFBRTtnQkFDdkIsT0FBTyxDQUFDLGNBQWMsSUFBSSxDQUFDLE9BQU8sRUFBRSxDQUFDLENBQUM7YUFDdkM7U0FDRjtJQUNILENBQUM7QUFDSCxDQUFDO0FBcERELG9EQW9EQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGNoYWxrIGZyb20gJ2NoYWxrJztcbmltcG9ydCAqIGFzIHlhcmdzIGZyb20gJ3lhcmdzJztcbmltcG9ydCAqIGFzIGxvZ2dpbmcgZnJvbSAnLi4vbG9nZ2luZyc7XG5pbXBvcnQgeyBUYXNrUnVudGltZSB9IGZyb20gJy4uL3Rhc2tzJztcblxuLyoqXG4gKiBSZWFkcyAucHJvamVuL3Rhc2tzLmpzb24gYW5kIGFkZHMgQ0xJIGNvbW1hbmRzIGZvciBhbGwgdGFza3MuXG4gKiBAcGFyYW0geWEgeWFyZ3NcbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIGRpc2NvdmVyVGFza0NvbW1hbmRzKHJ1bnRpbWU6IFRhc2tSdW50aW1lLCB5YTogeWFyZ3MuQXJndikge1xuICBjb25zdCB0YXNrcyA9IHJ1bnRpbWUubWFuaWZlc3QudGFza3MgPz8ge307XG4gIGZvciAoY29uc3QgdGFzayBvZiBPYmplY3QudmFsdWVzKHRhc2tzKSkge1xuICAgIHlhLmNvbW1hbmQodGFzay5uYW1lLCB0YXNrLmRlc2NyaXB0aW9uID8/ICcnLCB0YXNrQ29tbWFuZEhhbmRsZXIodGFzay5uYW1lKSk7XG4gIH1cblxuICBmdW5jdGlvbiB0YXNrQ29tbWFuZEhhbmRsZXIodGFza05hbWU6IHN0cmluZykge1xuICAgIHJldHVybiAoYXJnczogeWFyZ3MuQXJndikgPT4ge1xuICAgICAgYXJncy5vcHRpb24oJ2luc3BlY3QnLCB7IGFsaWFzOiAnaScsIGRlc2M6ICdzaG93IGFsbCBzdGVwcyBpbiB0aGlzIHRhc2snIH0pO1xuXG4gICAgICBjb25zdCBhcmd2ID0gYXJncy5hcmd2O1xuXG4gICAgICBpZiAoYXJndi5pbnNwZWN0KSB7XG4gICAgICAgIHJldHVybiBpbnNwZWN0VGFzayh0YXNrTmFtZSk7XG4gICAgICB9IGVsc2Uge1xuICAgICAgICB0cnkge1xuICAgICAgICAgIHJ1bnRpbWUucnVuVGFzayh0YXNrTmFtZSk7XG4gICAgICAgIH0gY2F0Y2ggKGUpIHtcbiAgICAgICAgICBsb2dnaW5nLmVycm9yKGUubWVzc2FnZSk7XG4gICAgICAgICAgcHJvY2Vzcy5leGl0KDEpO1xuICAgICAgICB9XG4gICAgICB9XG4gICAgfTtcbiAgfVxuXG4gIGZ1bmN0aW9uIGluc3BlY3RUYXNrKG5hbWU6IHN0cmluZywgaW5kZW50ID0gMCkge1xuICAgIGNvbnN0IHdyaXRlbG4gPSAoczogc3RyaW5nKSA9PiBjb25zb2xlLmxvZygnICcucmVwZWF0KGluZGVudCkgKyBzKTtcblxuICAgIGNvbnN0IHRhc2sgPSBydW50aW1lLnRyeUZpbmRUYXNrKG5hbWUpO1xuICAgIGlmICghdGFzaykge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGAke25hbWV9OiB1bmFibGUgdG8gcmVzb2x2ZSBzdWJ0YXNrIHdpdGggbmFtZSBcIiR7bmFtZX1cImApO1xuICAgIH1cblxuICAgIGlmICh0YXNrLmRlc2NyaXB0aW9uKSB7XG4gICAgICB3cml0ZWxuKGAke2NoYWxrLnVuZGVybGluZSgnZGVzY3JpcHRpb24nKX06ICR7dGFzay5kZXNjcmlwdGlvbn1gKTtcbiAgICB9XG5cbiAgICBmb3IgKGNvbnN0IFtrLCB2XSBvZiBPYmplY3QuZW50cmllcyh0YXNrLmVudiA/PyB7fSkpIHtcbiAgICAgIHdyaXRlbG4oYCR7Y2hhbGsudW5kZXJsaW5lKCdlbnYnKX06ICR7a309JHt2fWApO1xuICAgIH1cblxuICAgIGZvciAoY29uc3Qgc3RlcCBvZiB0YXNrLnN0ZXBzID8/IFtdKSB7XG4gICAgICBpZiAoc3RlcC5zcGF3bikge1xuICAgICAgICB3cml0ZWxuKGAtICR7Y2hhbGsuYm9sZChzdGVwLnNwYXduKX1gKTtcbiAgICAgICAgaW5zcGVjdFRhc2soc3RlcC5zcGF3biwgaW5kZW50ICsgMik7XG4gICAgICB9IGVsc2UgaWYgKHN0ZXAuZXhlYykge1xuICAgICAgICB3cml0ZWxuKGAtIGV4ZWM6ICR7c3RlcC5leGVjfWApO1xuICAgICAgfSBlbHNlIGlmIChzdGVwLmJ1aWx0aW4pIHtcbiAgICAgICAgd3JpdGVsbihgLSBidWlsdGluOiAke3N0ZXAuYnVpbHRpbn1gKTtcbiAgICAgIH1cbiAgICB9XG4gIH1cbn0iXX0=