"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.createProject = void 0;
const path = require("path");
const vm = require("vm");
const render_options_1 = require("../javascript/render-options");
/**
 * Creates a new project with defaults.
 *
 * This function creates the project type in-process (with in VM) and calls
 * `.synth()` on it (if `options.synth` is not `false`).
 *
 * At the moment, it also generates a `.projenrc.js` file with the same code
 * that was just executed. In the future, this will also be done by the project
 * type, so we can easily support multiple languages of projenrc.
 */
function createProject(opts) {
    // Default project resolution location
    let mod = '../index';
    // External projects need to load the module from the modules directory
    if (opts.type.moduleName !== 'projen') {
        try {
            mod = path.dirname(require.resolve(path.join(opts.type.moduleName, 'package.json'), { paths: [process.cwd()] }));
        }
        catch (err) {
            throw new Error(`External project module '${opts.type.moduleName}' could not be resolved.`);
        }
    }
    // pass the FQN of the project type to the project initializer so it can
    // generate the projenrc file.
    const { renderedOptions } = render_options_1.renderJavaScriptOptions({
        bootstrap: true,
        comments: opts.comments,
        type: opts.type,
        args: opts.params,
    });
    const newProjectCode = `const project = new ${opts.type.typename}(${renderedOptions});`;
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const module = require(mod);
    const ctx = vm.createContext(module);
    process.env.PROJEN_DISABLE_POST = (!opts.post).toString();
    vm.runInContext([
        newProjectCode,
        opts.synth ? 'project.synth();' : '',
    ].join('\n'), ctx);
}
exports.createProject = createProject;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY3JlYXRlLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vc3JjL2NsaS9jcmVhdGUudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEsNkJBQTZCO0FBQzdCLHlCQUF5QjtBQUV6QixpRUFBdUU7QUFxQ3ZFOzs7Ozs7Ozs7R0FTRztBQUNILFNBQWdCLGFBQWEsQ0FBQyxJQUEwQjtJQUN0RCxzQ0FBc0M7SUFDdEMsSUFBSSxHQUFHLEdBQUcsVUFBVSxDQUFDO0lBRXJCLHVFQUF1RTtJQUN2RSxJQUFJLElBQUksQ0FBQyxJQUFJLENBQUMsVUFBVSxLQUFLLFFBQVEsRUFBRTtRQUNyQyxJQUFJO1lBQ0YsR0FBRyxHQUFHLElBQUksQ0FBQyxPQUFPLENBQ2hCLE9BQU8sQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFVBQVUsRUFBRSxjQUFjLENBQUMsRUFBRSxFQUFFLEtBQUssRUFBRSxDQUFDLE9BQU8sQ0FBQyxHQUFHLEVBQUUsQ0FBQyxFQUFFLENBQUMsQ0FDN0YsQ0FBQztTQUNIO1FBQUMsT0FBTyxHQUFHLEVBQUU7WUFDWixNQUFNLElBQUksS0FBSyxDQUFDLDRCQUE0QixJQUFJLENBQUMsSUFBSSxDQUFDLFVBQVUsMEJBQTBCLENBQUMsQ0FBQztTQUM3RjtLQUNGO0lBRUQsd0VBQXdFO0lBQ3hFLDhCQUE4QjtJQUM5QixNQUFNLEVBQUUsZUFBZSxFQUFFLEdBQUcsd0NBQXVCLENBQUM7UUFDbEQsU0FBUyxFQUFFLElBQUk7UUFDZixRQUFRLEVBQUUsSUFBSSxDQUFDLFFBQVE7UUFDdkIsSUFBSSxFQUFFLElBQUksQ0FBQyxJQUFJO1FBQ2YsSUFBSSxFQUFFLElBQUksQ0FBQyxNQUFNO0tBQ2xCLENBQUMsQ0FBQztJQUVILE1BQU0sY0FBYyxHQUFHLHVCQUF1QixJQUFJLENBQUMsSUFBSSxDQUFDLFFBQVEsSUFBSSxlQUFlLElBQUksQ0FBQztJQUV4RixpRUFBaUU7SUFDakUsTUFBTSxNQUFNLEdBQUcsT0FBTyxDQUFDLEdBQUcsQ0FBQyxDQUFDO0lBQzVCLE1BQU0sR0FBRyxHQUFHLEVBQUUsQ0FBQyxhQUFhLENBQUMsTUFBTSxDQUFDLENBQUM7SUFFckMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxtQkFBbUIsR0FBRyxDQUFDLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsRUFBRSxDQUFDO0lBQzFELEVBQUUsQ0FBQyxZQUFZLENBQUM7UUFDZCxjQUFjO1FBQ2QsSUFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsa0JBQWtCLENBQUMsQ0FBQyxDQUFDLEVBQUU7S0FDckMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLEVBQUUsR0FBRyxDQUFDLENBQUM7QUFDckIsQ0FBQztBQW5DRCxzQ0FtQ0MiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBwYXRoIGZyb20gJ3BhdGgnO1xuaW1wb3J0ICogYXMgdm0gZnJvbSAndm0nO1xuaW1wb3J0ICogYXMgaW52ZW50b3J5IGZyb20gJy4uL2ludmVudG9yeSc7XG5pbXBvcnQgeyByZW5kZXJKYXZhU2NyaXB0T3B0aW9ucyB9IGZyb20gJy4uL2phdmFzY3JpcHQvcmVuZGVyLW9wdGlvbnMnO1xuaW1wb3J0IHsgTmV3UHJvamVjdE9wdGlvbkhpbnRzIH0gZnJvbSAnLi4vb3B0aW9uLWhpbnRzJztcblxuZXhwb3J0IGludGVyZmFjZSBDcmVhdGVQcm9qZWN0T3B0aW9ucyB7XG4gIC8qKlxuICAgKiBQcm9qZWN0IGRpcmVjdG9yeS5cbiAgICovXG4gIGRpcjogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBQcm9qZWN0IHR5cGUgZnJvbSB0aGUgaW52ZW50b3J5LlxuICAgKi9cbiAgdHlwZTogaW52ZW50b3J5LlByb2plY3RUeXBlO1xuXG4gIC8qKlxuICAgKiBPcHRpb24gdmFsdWVzLlxuICAgKi9cbiAgcGFyYW1zOiBSZWNvcmQ8c3RyaW5nLCBzdHJpbmc+O1xuXG4gIC8qKlxuICAgKiBTaG91bGQgd2UgcmVuZGVyIGNvbW1lbnRlZC1vdXQgZGVmYXVsdCBvcHRpb25zIGluIHRoZSBwcm9qZW5yYyBmaWxlP1xuICAgKiBEb2VzIG5vdCBhcHBseSB0byBwcm9qZW5yYy5qc29uIGZpbGVzLlxuICAgKi9cbiAgY29tbWVudHM6IE5ld1Byb2plY3RPcHRpb25IaW50cztcblxuICAvKipcbiAgICogU2hvdWxkIHdlIGNhbGwgYHByb2plY3Quc3ludGgoKWAgb3IgaW5zdGFudGlhdGUgdGhlIHByb2plY3QgKGNvdWxkIHN0aWxsXG4gICAqIGhhdmUgc2lkZS1lZmZlY3RzKSBhbmQgcmVuZGVyIHRoZSAucHJvamVucmMgZmlsZS5cbiAgICovXG4gIHN5bnRoOiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBTaG91bGQgd2UgZXhlY3V0ZSBwb3N0IHN5bnRoZXNpcyBob29rcz8gKHVzdWFsbHkgcGFja2FnZSBtYW5hZ2VyIGluc3RhbGwpLlxuICAgKi9cbiAgcG9zdDogYm9vbGVhbjtcbn1cblxuLyoqXG4gKiBDcmVhdGVzIGEgbmV3IHByb2plY3Qgd2l0aCBkZWZhdWx0cy5cbiAqXG4gKiBUaGlzIGZ1bmN0aW9uIGNyZWF0ZXMgdGhlIHByb2plY3QgdHlwZSBpbi1wcm9jZXNzICh3aXRoIGluIFZNKSBhbmQgY2FsbHNcbiAqIGAuc3ludGgoKWAgb24gaXQgKGlmIGBvcHRpb25zLnN5bnRoYCBpcyBub3QgYGZhbHNlYCkuXG4gKlxuICogQXQgdGhlIG1vbWVudCwgaXQgYWxzbyBnZW5lcmF0ZXMgYSBgLnByb2plbnJjLmpzYCBmaWxlIHdpdGggdGhlIHNhbWUgY29kZVxuICogdGhhdCB3YXMganVzdCBleGVjdXRlZC4gSW4gdGhlIGZ1dHVyZSwgdGhpcyB3aWxsIGFsc28gYmUgZG9uZSBieSB0aGUgcHJvamVjdFxuICogdHlwZSwgc28gd2UgY2FuIGVhc2lseSBzdXBwb3J0IG11bHRpcGxlIGxhbmd1YWdlcyBvZiBwcm9qZW5yYy5cbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIGNyZWF0ZVByb2plY3Qob3B0czogQ3JlYXRlUHJvamVjdE9wdGlvbnMpIHtcbiAgLy8gRGVmYXVsdCBwcm9qZWN0IHJlc29sdXRpb24gbG9jYXRpb25cbiAgbGV0IG1vZCA9ICcuLi9pbmRleCc7XG5cbiAgLy8gRXh0ZXJuYWwgcHJvamVjdHMgbmVlZCB0byBsb2FkIHRoZSBtb2R1bGUgZnJvbSB0aGUgbW9kdWxlcyBkaXJlY3RvcnlcbiAgaWYgKG9wdHMudHlwZS5tb2R1bGVOYW1lICE9PSAncHJvamVuJykge1xuICAgIHRyeSB7XG4gICAgICBtb2QgPSBwYXRoLmRpcm5hbWUoXG4gICAgICAgIHJlcXVpcmUucmVzb2x2ZShwYXRoLmpvaW4ob3B0cy50eXBlLm1vZHVsZU5hbWUsICdwYWNrYWdlLmpzb24nKSwgeyBwYXRoczogW3Byb2Nlc3MuY3dkKCldIH0pLFxuICAgICAgKTtcbiAgICB9IGNhdGNoIChlcnIpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgRXh0ZXJuYWwgcHJvamVjdCBtb2R1bGUgJyR7b3B0cy50eXBlLm1vZHVsZU5hbWV9JyBjb3VsZCBub3QgYmUgcmVzb2x2ZWQuYCk7XG4gICAgfVxuICB9XG5cbiAgLy8gcGFzcyB0aGUgRlFOIG9mIHRoZSBwcm9qZWN0IHR5cGUgdG8gdGhlIHByb2plY3QgaW5pdGlhbGl6ZXIgc28gaXQgY2FuXG4gIC8vIGdlbmVyYXRlIHRoZSBwcm9qZW5yYyBmaWxlLlxuICBjb25zdCB7IHJlbmRlcmVkT3B0aW9ucyB9ID0gcmVuZGVySmF2YVNjcmlwdE9wdGlvbnMoe1xuICAgIGJvb3RzdHJhcDogdHJ1ZSxcbiAgICBjb21tZW50czogb3B0cy5jb21tZW50cyxcbiAgICB0eXBlOiBvcHRzLnR5cGUsXG4gICAgYXJnczogb3B0cy5wYXJhbXMsXG4gIH0pO1xuXG4gIGNvbnN0IG5ld1Byb2plY3RDb2RlID0gYGNvbnN0IHByb2plY3QgPSBuZXcgJHtvcHRzLnR5cGUudHlwZW5hbWV9KCR7cmVuZGVyZWRPcHRpb25zfSk7YDtcblxuICAvLyBlc2xpbnQtZGlzYWJsZS1uZXh0LWxpbmUgQHR5cGVzY3JpcHQtZXNsaW50L25vLXJlcXVpcmUtaW1wb3J0c1xuICBjb25zdCBtb2R1bGUgPSByZXF1aXJlKG1vZCk7XG4gIGNvbnN0IGN0eCA9IHZtLmNyZWF0ZUNvbnRleHQobW9kdWxlKTtcblxuICBwcm9jZXNzLmVudi5QUk9KRU5fRElTQUJMRV9QT1NUID0gKCFvcHRzLnBvc3QpLnRvU3RyaW5nKCk7XG4gIHZtLnJ1bkluQ29udGV4dChbXG4gICAgbmV3UHJvamVjdENvZGUsXG4gICAgb3B0cy5zeW50aCA/ICdwcm9qZWN0LnN5bnRoKCk7JyA6ICcnLFxuICBdLmpvaW4oJ1xcbicpLCBjdHgpO1xufVxuIl19