"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const fs = require("fs-extra");
const yargs = require("yargs");
const inventory = require("../../inventory");
const logging = require("../../logging");
const option_hints_1 = require("../../option-hints");
const util_1 = require("../../util");
const create_1 = require("../create");
const macros_1 = require("../macros");
class Command {
    constructor() {
        this.command = 'new [PROJECT-TYPE-NAME] [OPTIONS]';
        this.describe = 'Creates a new projen project';
    }
    builder(args) {
        var _a;
        args.positional('PROJECT-TYPE-NAME', { describe: 'optional only when --from is used and there is a single project type in the external module', type: 'string' });
        args.option('synth', { type: 'boolean', default: true, desc: 'Synthesize after creating .projenrc.js' });
        args.option('comments', { type: 'boolean', default: true, desc: 'Include commented out options in .projenrc.js (use --no-comments to disable)' });
        args.option('from', { type: 'string', alias: 'f', desc: 'External jsii npm module to create project from. Supports any package spec supported by npm (such as "my-pack@^2.0")' });
        args.option('git', { type: 'boolean', default: true, desc: 'Run `git init` and create an initial commit (use --no-git to disable)' });
        args.example('projen new awscdk-app-ts', 'Creates a new project of built-in type "awscdk-app-ts"');
        args.example('projen new --from projen-vue@^2', 'Creates a new project from an external module "projen-vue" with the specified version');
        for (const type of inventory.discover()) {
            args.command(type.pjid, (_a = type.docs) !== null && _a !== void 0 ? _a : '', {
                builder: cargs => {
                    var _a, _b, _c;
                    cargs.showHelpOnFail(false);
                    for (const option of (_a = type.options) !== null && _a !== void 0 ? _a : []) {
                        if (option.type !== 'string' && option.type !== 'number' && option.type !== 'boolean' && option.kind !== 'enum') {
                            continue; // we only support primitive and enum fields as command line options
                        }
                        let desc = [(_c = (_b = option.docs) === null || _b === void 0 ? void 0 : _b.replace(/\ *\.$/, '')) !== null && _c !== void 0 ? _c : ''];
                        const required = !option.optional;
                        let defaultValue;
                        if (option.default && option.default !== 'undefined') {
                            if (!required) {
                                // if the field is not required, just describe the default but don't actually assign a value
                                desc.push(`[default: ${option.default.replace(/^\ *-/, '').replace(/\.$/, '').trim()}]`);
                            }
                            else {
                                // if the field is required and we have a @default, then assign
                                // the value here so it appears in `--help`
                                defaultValue = renderDefault(process.cwd(), option.default);
                            }
                        }
                        const argType = option.kind === 'enum' ? 'string' : option.type;
                        cargs.option(option.switch, {
                            group: required ? 'Required:' : 'Optional:',
                            type: argType,
                            description: desc.join(' '),
                            default: defaultValue,
                            required,
                        });
                    }
                    return cargs;
                },
                handler: argv => newProject(process.cwd(), type, argv),
            });
        }
        return args;
    }
    async handler(args) {
        // handle --from which means we want to first install a jsii module and then
        // create a project defined within this module.
        if (args.from) {
            return newProjectFromModule(process.cwd(), args.from, args);
        }
        // project type is defined but was not matched by yargs, so print the list of supported types
        if (args.projectTypeName) {
            console.log(`Invalid project type ${args.projectTypeName}. Supported types:`);
            for (const pjid of inventory.discover().map(x => x.pjid)) {
                console.log(`  ${pjid}`);
            }
            return;
        }
        // Handles the use case that nothing was specified since PROJECT-TYPE is now an optional positional parameter
        yargs.showHelp();
    }
}
/**
 * Given a value from "@default", processes macros and returns a stringified
 * (quoted) result.
 *
 * @returns a javascript primitive (could be a string, number or boolean)
 */
function renderDefault(cwd, value) {
    var _a;
    return (_a = macros_1.tryProcessMacro(cwd, value)) !== null && _a !== void 0 ? _a : JSON.parse(value);
}
/**
 * Converts yargs command line switches to project type props.
 * @param type Project type
 * @param argv Command line switches
 */
function commandLineToProps(cwd, type, argv) {
    var _a;
    const props = {};
    // initialize props with default values
    for (const prop of type.options) {
        if (prop.default && prop.default !== 'undefined' && !prop.optional) {
            props[prop.name] = renderDefault(cwd, prop.default);
        }
    }
    for (const [arg, value] of Object.entries(argv)) {
        for (const prop of type.options) {
            if (prop.switch === arg) {
                let curr = props;
                const queue = [...prop.path];
                while (true) {
                    const p = queue.shift();
                    if (!p) {
                        break;
                    }
                    if (queue.length === 0) {
                        curr[p] = value;
                    }
                    else {
                        curr[p] = (_a = curr[p]) !== null && _a !== void 0 ? _a : {};
                        curr = curr[p];
                    }
                }
            }
        }
    }
    return props;
}
/**
 * Generates a new project from an external module.
 *
 * @param spec The name of the external module to load
 * @param args Command line arguments (incl. project type)
 */
async function newProjectFromModule(baseDir, spec, args) {
    var _a, _b;
    const projenVersion = (_a = args.projenVersion) !== null && _a !== void 0 ? _a : 'latest';
    const installCommand = renderInstallCommand(baseDir, `projen@${projenVersion}`);
    if (args.projenVersion) {
        util_1.exec(installCommand, { cwd: baseDir });
    }
    else {
        // do not overwrite existing installation
        util_1.exec(`npm ls --prefix=${baseDir} --depth=0 --pattern projen || ${installCommand}`, { cwd: baseDir });
    }
    const moduleName = installPackage(baseDir, spec);
    // Find the just installed package and discover the rest recursively from this package folder
    const moduleDir = path.dirname(require.resolve(`${moduleName}/.jsii`, {
        paths: [
            baseDir,
        ],
    }));
    // Only leave projects from the main (requested) package
    const projects = inventory
        .discover(moduleDir)
        .filter(x => x.moduleName === moduleName); // Only list project types from the requested 'from' module
    if (projects.length < 1) {
        throw new Error(`No projects found after installing ${spec}. The module must export at least one class which extends projen.Project`);
    }
    const requested = args.projectTypeName;
    const types = projects.map(p => p.pjid);
    // if user did not specify a project type but the module has more than one, we need them to tell us which one...
    if (!requested && projects.length > 1) {
        throw new Error(`Multiple projects found after installing ${spec}: ${types.join(',')}. Please specify a project name.\nExample: npx projen new --from ${spec} ${types[0]}`);
    }
    // if user did not specify a type (and we know we have only one), the select it. otherwise, search by pjid.
    const type = !requested ? projects[0] : projects.find(p => p.pjid === requested);
    if (!type) {
        throw new Error(`Project type ${requested} not found. Found ${types.join(',')}`);
    }
    for (const option of (_b = type.options) !== null && _b !== void 0 ? _b : []) {
        if (option.type !== 'string' && option.type !== 'number' && option.type !== 'boolean') {
            continue; // we don't support non-primitive fields as command line options
        }
        if (args[option.name] !== undefined) {
            if (option.type === 'number') {
                args[option.name] = parseInt(args[option.name]);
                args[option.switch] = args[option.name];
            }
            else if (option.type === 'boolean') {
                const raw = args[option.name];
                const safe = typeof raw === 'string' ? util_1.isTruthy(raw) : raw;
                args[option.name] = safe;
                args[option.switch] = safe;
            }
            continue; // do not overwrite passed arguments
        }
        if (option.default && option.default !== 'undefined') {
            if (!option.optional) {
                const defaultValue = renderDefault(baseDir, option.default);
                args[option.name] = defaultValue;
                args[option.switch] = defaultValue;
            }
        }
    }
    // include a dev dependency for the external module
    await newProject(baseDir, type, args, {
        devDeps: [spec],
    });
}
/**
 * Generates a new project.
 * @param type Project type
 * @param args Command line arguments
 * @param additionalProps Additional parameters to include in .projenrc.js
 */
async function newProject(baseDir, type, args, additionalProps) {
    // convert command line arguments to project props using type information
    const props = commandLineToProps(baseDir, type, args);
    // merge in additional props if specified
    for (const [k, v] of Object.entries(additionalProps !== null && additionalProps !== void 0 ? additionalProps : {})) {
        props[k] = v;
    }
    create_1.createProject({
        dir: baseDir,
        type,
        params: props,
        comments: args.comments ? option_hints_1.NewProjectOptionHints.FEATURED : option_hints_1.NewProjectOptionHints.NONE,
        synth: args.synth,
        post: args.post,
    });
    if (args.git) {
        const git = (cmd) => util_1.exec(`git ${cmd}`, { cwd: baseDir });
        git('init');
        git('add .');
        git('commit --allow-empty -m "chore: project created with projen"');
        git('branch -M main');
    }
}
/**
 * Installs the npm module (through `npm install`) to node_modules under `projectDir`.
 * @param spec The npm package spec (e.g. `foo@^1.2` or `foo@/var/folders/8k/qcw0ls5pv_ph0000gn/T/projen-RYurCw/pkg.tgz`)
 * @returns The installed package name (e.g. `@foo/bar`)
 */
function installPackage(baseDir, spec) {
    const packageJsonPath = path.join(baseDir, 'package.json');
    const packageJsonExisted = fs.existsSync(packageJsonPath);
    if (!packageJsonExisted) {
        // Make sure we have a package.json to read from later
        util_1.exec('npm init --yes', { cwd: baseDir });
    }
    logging.info(`installing external module ${spec}...`);
    util_1.exec(renderInstallCommand(baseDir, spec), { cwd: baseDir });
    // Get the true installed package name
    const packageJson = fs.readJsonSync(packageJsonPath);
    const packageName = Object.keys(packageJson.devDependencies).find(name => name !== 'projen');
    if (!packageName) {
        throw new Error(`Unable to resolve package name from spec ${spec}`);
    }
    // if package.json did not exist before calling `npm install`, we should remove it
    // so we can start off clean.
    if (!packageJsonExisted) {
        fs.removeSync(packageJsonPath);
    }
    return packageName;
}
/**
 * Render a command to install an npm package.
 *
 * Engine checks are ignorred at this point so that the module can be installed
 * regardless of the environment. This was needed to unblock the upgrade of the
 * minimum node version of projen, but also okay generally because engine checks
 * will be performed later and for all eternety.
 *
 * @param dir Base directory
 * @param module The module to install (e.g. foo@^1.2)
 * @returns The string that includes the install command ("npm install ...")
 */
function renderInstallCommand(dir, module) {
    return `npm install --save-dev -f --no-package-lock --prefix=${dir} ${module}`;
}
module.exports = new Command();
//# sourceMappingURL=data:application/json;base64,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