import pandas as pd
from pandas.util.testing import network
import pytest
import pytz

from pvlib.iotools import midc
from conftest import DATA_DIR, RERUNS, RERUNS_DELAY


@pytest.fixture
def test_mapping():
    return {
        'Direct Normal [W/m^2]': 'dni',
        'Global PSP [W/m^2]': 'ghi',
        'Rel Humidity [%]': 'relative_humidity',
        'Temperature @ 2m [deg C]': 'temp_air',
        'Non Existant': 'variable',
    }


MIDC_TESTFILE = DATA_DIR / 'midc_20181014.txt'
MIDC_RAW_TESTFILE = DATA_DIR / 'midc_raw_20181018.txt'
MIDC_RAW_SHORT_HEADER_TESTFILE = (
    DATA_DIR / 'midc_raw_short_header_20191115.txt')

# TODO: not used, remove?
# midc_network_testfile = ('https://midcdmz.nrel.gov/apps/data_api.pl'
#                          '?site=UAT&begin=20181018&end=20181019')


def test_midc_format_index():
    data = pd.read_csv(MIDC_TESTFILE)
    data = midc.format_index(data)
    start = pd.Timestamp("20181014 00:00")
    start = start.tz_localize("MST")
    end = pd.Timestamp("20181014 23:59")
    end = end.tz_localize("MST")
    assert type(data.index) == pd.DatetimeIndex
    assert data.index[0] == start
    assert data.index[-1] == end


def test_midc_format_index_tz_conversion():
    data = pd.read_csv(MIDC_TESTFILE)
    data = data.rename(columns={'MST': 'PST'})
    data = midc.format_index(data)
    assert data.index[0].tz == pytz.timezone('Etc/GMT+8')


def test_midc_format_index_raw():
    data = pd.read_csv(MIDC_RAW_TESTFILE)
    data = midc.format_index_raw(data)
    start = pd.Timestamp('20181018 00:00')
    start = start.tz_localize('MST')
    end = pd.Timestamp('20181018 23:59')
    end = end.tz_localize('MST')
    assert data.index[0] == start
    assert data.index[-1] == end


def test_read_midc_var_mapping_as_arg(test_mapping):
    data = midc.read_midc(MIDC_TESTFILE, variable_map=test_mapping)
    assert 'ghi' in data.columns
    assert 'temp_air' in data.columns


@network
@pytest.mark.remote_data
@pytest.mark.flaky(reruns=RERUNS, reruns_delay=RERUNS_DELAY)
def test_read_midc_raw_data_from_nrel():
    start_ts = pd.Timestamp('20181018')
    end_ts = pd.Timestamp('20181019')
    var_map = midc.MIDC_VARIABLE_MAP['UAT']
    data = midc.read_midc_raw_data_from_nrel('UAT', start_ts, end_ts, var_map)
    for k, v in var_map.items():
        assert v in data.columns
    assert data.index.size == 2880


def test_read_midc_header_length_mismatch(mocker):
    mock_data = mocker.MagicMock()
    with MIDC_RAW_SHORT_HEADER_TESTFILE.open() as f:
        mock_data.text = f.read()
    mocker.patch('pvlib.iotools.midc.requests.get',
                 return_value=mock_data)
    start = pd.Timestamp('2019-11-15T00:00:00-06:00')
    end = pd.Timestamp('2019-11-15T23:59:00-06:00')
    data = midc.read_midc_raw_data_from_nrel('', start, end)
    assert isinstance(data.index, pd.DatetimeIndex)
    assert data.index[0] == start
    assert data.index[-1] == end
