# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/01_datasets.ipynb.

# %% ../nbs/01_datasets.ipynb 2
from __future__ import annotations
from .import_essentials import *
from .utils import load_json, validate_configs
from sklearn.preprocessing import StandardScaler,MinMaxScaler,OneHotEncoder
from urllib.request import urlretrieve

# %% auto 0
__all__ = ['backend2dataloader', 'DEFAULT_DATA_CONFIGS', 'Dataset', 'BaseDataLoader', 'DataLoaderPytorch', 'DataLoaderJax',
           'DataLoader', 'find_imutable_idx_list', 'DataModuleConfigs', 'TabularDataModule', 'load_default_data']

# %% ../nbs/01_datasets.ipynb 3
try:
    import torch.utils.data as torch_data
except ModuleNotFoundError:
    torch_data = None

# %% ../nbs/01_datasets.ipynb 5
class Dataset:
    def __init__(self, X, y):
        self.X = X
        self.y = y
        assert self.X.shape[0] == self.y.shape[0]

    def __len__(self):
        return len(self.X)

    def __getitem__(self, idx):
        return self.X[idx], self.y[idx]

# %% ../nbs/01_datasets.ipynb 6
class BaseDataLoader(ABC):
    def __init__(
        self, 
        dataset,
        backend: str,
        *,
        batch_size: int = 1,  # batch size
        shuffle: bool = False,  # if true, dataloader shuffles before sampling each batch
        num_workers: int = 0,
        drop_last: bool = False,
        **kwargs
    ):
        pass 
    
    def __len__(self):
        raise NotImplementedError
    
    def __next__(self):
        raise NotImplementedError
    
    def __iter__(self):
        raise NotImplementedError

# %% ../nbs/01_datasets.ipynb 8
# copy from https://jax.readthedocs.io/en/latest/notebooks/Neural_Network_and_Data_Loading.html#data-loading-with-pytorch
def _numpy_collate(batch):
    if isinstance(batch[0], np.ndarray):
        return np.stack(batch)
    elif isinstance(batch[0], (tuple, list)):
        transposed = zip(*batch)
        return [_numpy_collate(samples) for samples in transposed]
    else:
        return np.array(batch)

def _convert_dataset_pytorch(dataset: Dataset):
    class DatasetPytorch(torch_data.Dataset):
        def __init__(self, dataset: Dataset): self.dataset = dataset
        def __len__(self): return len(self.dataset)
        def __getitem__(self, idx): return self.dataset[idx]
    
    return DatasetPytorch(dataset)

# %% ../nbs/01_datasets.ipynb 9
class DataLoaderPytorch(BaseDataLoader):
    def __init__(
        self, 
        dataset: Dataset,
        backend: str = 'pytorch', # positional argument
        *,
        batch_size: int = 1,  # batch size
        shuffle: bool = False,  # if true, dataloader shuffles before sampling each batch
        num_workers: int = 0,
        drop_last: bool = False,
        **kwargs
    ):
        if torch_data is None:
            raise ModuleNotFoundError("`pytorch` library needs to be installed. Try `pip install torch`."
            "Please refer to pytorch documentation for details: https://pytorch.org/get-started/.")
        
        dataset = _convert_dataset_pytorch(dataset)
        self.dataloader = torch_data.DataLoader(
            dataset, 
            batch_size=batch_size, 
            shuffle=shuffle, 
            num_workers=num_workers, 
            drop_last=drop_last,
            collate_fn=_numpy_collate,
            **kwargs
        ) 

    def __len__(self):
        return len(self.dataloader)

    def __next__(self):
        return next(self.dataloader)

    def __iter__(self):
        return self.dataloader.__iter__()

# %% ../nbs/01_datasets.ipynb 11
class DataLoaderJax(BaseDataLoader):
    def __init__(
        self, 
        dataset: Dataset,
        backend: str,
        *,
        batch_size: int = 1,  # batch size
        shuffle: bool = False,  # if true, dataloader shuffles before sampling each batch
        num_workers: int = 0,
        drop_last: bool = False,
        **kwargs
    ):
        # Attributes from pytorch data loader (implemented)
        self.dataset = dataset
        self.batch_size = batch_size
        self.shuffle = shuffle
        self.drop_last = drop_last

        self.data_len: int = len(dataset)  # Length of the dataset
        self.indices: np.ndarray = np.arange(self.data_len) # available indices in the dataset
        self.pose: int = 0  # record the current position in the dataset

    def __len__(self):
        if self.drop_last:
            batches = len(self.dataset) // self.batch_size  # get the floor of division
        else:
            batches = -(
                len(self.dataset) // -self.batch_size
            )  # get the ceil of division
        return batches

    def __next__(self):
        if self.pose <= self.data_len:
            if self.shuffle:
                self.indices = np.random.permutation(self.indices)
            batch_data = self.dataset[self.indices[: self.batch_size]]
            self.indices = self.indices[self.batch_size :]
            if self.drop_last and len(self.indices) < self.batch_size:
                self.pose = 0
                self.indices = np.arange(self.data_len)
                raise StopIteration
            self.pose += self.batch_size
            return batch_data
        else:
            self.pose = 0
            self.indices = np.arange(self.data_len)
            raise StopIteration

    def __iter__(self):
        return self

# %% ../nbs/01_datasets.ipynb 13
backend2dataloader = {
    'jax': DataLoaderJax,
    'pytorch': DataLoaderPytorch,
    'tensorflow': None,
    'merlin': None,
}

# %% ../nbs/01_datasets.ipynb 14
def _dispatch_datalaoder(backend: str):
    dataloader_backends = backend2dataloader.keys()
    if not backend in dataloader_backends:
        raise ValueError(f"backend=`{backend}` is an invalid backend for dataloader. "
            f"Should be one of {dataloader_backends}.")
    
    dataloader_cls = backend2dataloader[backend]
    if dataloader_cls is None:
        raise NotImplementedError(f'backend=`{backend}` is not supported yet.')
    return dataloader_cls


# %% ../nbs/01_datasets.ipynb 15
class DataLoader(BaseDataLoader):
    @deprecated(removed_in='0.1.0', deprecated_in='0.0.11',
            details='Use `cfnet.data.DataLoader` instead.')
    def __init__(
        self,
        dataset,
        backend,
        *,
        batch_size: int = 1,  # batch size
        shuffle: bool = False,  # if true, dataloader shuffles before sampling each batch
        num_workers: int = 0,
        drop_last: bool = False,
        **kwargs
    ):
        self.__class__ = _dispatch_datalaoder(backend)
        self.__init__(
            dataset=dataset, 
            backend=backend, 
            batch_size=batch_size, 
            shuffle=shuffle, 
            num_workers=num_workers,
            drop_last=drop_last,
            **kwargs
        )

# %% ../nbs/01_datasets.ipynb 24
@deprecated(removed_in='0.1.0', deprecated_in='0.0.11',
            details='Use `cfnet.data.find_imutable_idx_list` instead.')
def find_imutable_idx_list(
    imutable_col_names: List[str],
    discrete_col_names: List[str],
    continuous_col_names: List[str],
    cat_arrays: List[List[str]],
) -> List[int]:
    imutable_idx_list = []
    for idx, col_name in enumerate(continuous_col_names):
        if col_name in imutable_col_names:
            imutable_idx_list.append(idx)

    cat_idx = len(continuous_col_names)

    for i, (col_name, cols) in enumerate(zip(discrete_col_names, cat_arrays)):
        cat_end_idx = cat_idx + len(cols)
        if col_name in imutable_col_names:
            imutable_idx_list += list(range(cat_idx, cat_end_idx))
        cat_idx = cat_end_idx
    return imutable_idx_list

# %% ../nbs/01_datasets.ipynb 25
class DataModuleConfigs(BaseParser):
    data_dir: str
    data_name: str
    discret_cols: List[str] = []
    continous_cols: List[str] = []
    imutable_cols: List[str] = []
    normalizer: Optional[Any] = None
    encoder: Optional[Any] = None
    sample_frac: Optional[float] = None
    backend: str = 'jax'

# %% ../nbs/01_datasets.ipynb 26
class TabularDataModule:
    discret_cols: List[str] = []
    continous_cols: List[str] = []
    imutable_cols: List[str] = []
    normalizer: Optional[Any] = None
    encoder: Optional[OneHotEncoder] = None
    cat_arrays: List[str] = []
    data: Optional[pd.DataFrame] = None
    sample_frac: Optional[float] = None
    backend: str = 'jax'
    data_name: str = ""

    @deprecated(removed_in='0.1.0', deprecated_in='0.0.11',
            details='Use `cfnet.data.TabularDataModule` instead.')
    def __init__(
        self, 
        data_config: dict | DataModuleConfigs, # Configurator of `TabularDataModule`
        df: pd.DataFrame = None # Dataframe which overrides `data_dir` in `data_config` (if not None)
    ):
        self.configs: DataModuleConfigs = validate_configs(
            data_config, DataModuleConfigs
        )
        if df is None:
            self.data = pd.read_csv(self.configs.data_dir)
        elif isinstance(df, pd.DataFrame):
            self.data = df
        else:
            raise ValueError(f"{type(df).__name__} is not supported as an input type for `TabularDataModule`.")

        # update configs
        self._update_configs(self.configs.dict())
        self.check_cols()
        # update cat_idx
        self.cat_idx = len(self.continous_cols)
        # prepare data
        self.prepare_data()

    def check_cols(self):
        self.data = self.data.astype({col: float for col in self.continous_cols})
        # check imutable cols
        cols = self.continous_cols + self.discret_cols
        for col in self.imutable_cols:
            assert (
                 col in cols
             ), f"imutable_cols=[{col}] is not specified in `continous_cols` or `discret_cols`."

    def _update_configs(self, configs):
        for k, v in configs.items():
            setattr(self, k, v)

    def prepare_data(self):
        def split_x_and_y(data: pd.DataFrame):
            X = data[data.columns[:-1]]
            y = data[[data.columns[-1]]]
            return X, y

        X, y = split_x_and_y(self.data)

        # preprocessing
        if self.normalizer:
            X_cont = self.normalizer.transform(X[self.continous_cols])
        else:
            self.normalizer = MinMaxScaler()
            X_cont = (
                 self.normalizer.fit_transform(X[self.continous_cols])
                 if self.continous_cols
                 else np.array([[] for _ in range(len(X))])
             )

        if self.encoder:
            X_cat = self.encoder.transform(X[self.discret_cols])
        else:
            self.encoder = OneHotEncoder(sparse=False)
            X_cat = (
                 self.encoder.fit_transform(X[self.discret_cols])
                 if self.discret_cols
                 else np.array([[] for _ in range(len(X))])
             )
        X = np.concatenate((X_cont, X_cat), axis=1)
        # get categorical arrays
        self.cat_arrays = self.encoder.categories_ if self.discret_cols else []
        self.imutable_idx_list = find_imutable_idx_list(
            imutable_col_names=self.imutable_cols,
            discrete_col_names=self.discret_cols,
            continuous_col_names=self.continous_cols,
            cat_arrays=self.cat_arrays,
        )

        # prepare train & test
        train_test_tuple = train_test_split(X, y.to_numpy(), shuffle=False)
        train_X, test_X, train_y, test_y = map(
             lambda x: x.astype(float), train_test_tuple
         )
        if self.sample_frac:
            train_size = int(len(train_X) * self.sample_frac)
            train_X, train_y = train_X[:train_size], train_y[:train_size]
        self.train_dataset = Dataset(train_X, train_y)
        self.val_dataset = Dataset(test_X, test_y)
        self.test_dataset = self.val_dataset

    def train_dataloader(self, batch_size):
        return DataLoader(self.train_dataset, self.backend, 
            batch_size=batch_size, shuffle=True, num_workers=0, drop_last=False
        )

    def val_dataloader(self, batch_size):
        return DataLoader(self.val_dataset, self.backend,
            batch_size=batch_size, shuffle=True, num_workers=0, drop_last=False
        )

    def test_dataloader(self, batch_size):
        return DataLoader(self.val_dataset, self.backend,
            batch_size=batch_size, shuffle=True, num_workers=0, drop_last=False
        )

    def get_sample_X(self, frac: float | None = None):
        train_X, _ = self.get_samples(frac)
        return train_X

    def get_samples(self, frac: float | None = None):
        if frac is None:
            frac = 0.1
        train_X, train_y = self.train_dataset[:]
        train_size = int(len(train_X) * frac)
        return train_X[:train_size], train_y[:train_size]

# %% ../nbs/01_datasets.ipynb 30
DEFAULT_DATA_CONFIGS = {
    'adult': {
        'data' :'assets/data/s_adult.csv',
        'conf' :'assets/configs/adult.json',
    },
    'heloc': {
        'data': 'assets/data/s_home.csv',
        'conf': 'assets/configs/home.json'
    },
    'oulad': {
        'data': 'assets/data/s_student.csv',
        'conf': 'assets/configs/student.json'
    }
}

# %% ../nbs/01_datasets.ipynb 31
def _validate_dataname(data_name: str):
    if data_name not in DEFAULT_DATA_CONFIGS.keys():
        raise ValueError(f'`data_name` must be one of {DEFAULT_DATA_CONFIGS.keys()}, '
            f'but got data_name={data_name}.')

# %% ../nbs/01_datasets.ipynb 32
@deprecated(removed_in='0.1.0', deprecated_in='0.0.11',
            details='Use `cfnet.data.load_data` instead.')
def load_default_data(data_name: str) -> TabularDataModule:
    _validate_dataname(data_name)

    # get data/config urls
    _data_path = DEFAULT_DATA_CONFIGS[data_name]['data']
    _conf_path = DEFAULT_DATA_CONFIGS[data_name]['conf']
    
    data_url = f"https://github.com/BirkhoffG/cfnet/raw/master/{_data_path}"
    conf_url = f"https://github.com/BirkhoffG/cfnet/raw/master/{_conf_path}"

    # create new dir
    data_dir = Path(os.getcwd()) / "cf_data"
    if not data_dir.exists():
        os.makedirs(data_dir)
    data_path = data_dir / f'{data_name}.csv'
    conf_path = data_dir / f'{data_name}.json'

    # download data/configs
    if not data_path.is_file():
        urlretrieve(data_url, data_path)    
    if not conf_path.is_file():
        urlretrieve(conf_url, conf_path)

    # read config
    config = load_json(conf_path)
    config['data_dir'] = str(data_path)

    data_module = TabularDataModule(config)
    return data_module
