# AUTOGENERATED! DO NOT EDIT! File to edit: ../../nbs/01_data.loader.ipynb.

# %% ../../nbs/01_data.loader.ipynb 3
from __future__ import annotations
from ..import_essentials import *


# %% auto 0
__all__ = ['backend2dataloader', 'Dataset', 'BaseDataLoader', 'DataLoaderJax', 'DataLoaderPytorch', 'DataLoader']

# %% ../../nbs/01_data.loader.ipynb 4
try:
    import torch.utils.data as torch_data
except ModuleNotFoundError:
    torch_data = None

# %% ../../nbs/01_data.loader.ipynb 5
class Dataset:
    """A simple pytorch-like Numpy Dataset."""
    def __init__(self, X: jnp.DeviceArray, y: jnp.DeviceArray):
        self.X = X
        self.y = y
        assert self.X.shape[0] == self.y.shape[0]

    def __len__(self):
        return len(self.X)

    def __getitem__(self, idx):
        return self.X[idx], self.y[idx]

# %% ../../nbs/01_data.loader.ipynb 9
class BaseDataLoader(ABC):
    """Dataloader Interface"""
    def __init__(
        self, 
        dataset,
        backend: str,
        *,
        batch_size: int = 1,  # batch size
        shuffle: bool = False,  # if true, dataloader shuffles before sampling each batch
        num_workers: int = 0,
        drop_last: bool = False,
        **kwargs
    ):
        pass 
    
    def __len__(self):
        raise NotImplementedError
    
    def __next__(self):
        raise NotImplementedError
    
    def __iter__(self):
        raise NotImplementedError

# %% ../../nbs/01_data.loader.ipynb 11
class DataLoaderJax(BaseDataLoader):
    """Dataloder in Vanilla Jax"""
    def __init__(
        self, 
        dataset: Dataset,
        backend: str = 'jax', # positional argument
        batch_size: int = 1,  # batch size
        shuffle: bool = False,  # if true, dataloader shuffles before sampling each batch
        num_workers: int = 0, # positional argument; 
        drop_last: bool = False, # drop last batches or not
        **kwargs
    ):
        # Attributes from pytorch data loader (implemented)
        self.dataset = dataset
        self.batch_size = batch_size
        self.shuffle = shuffle
        self.drop_last = drop_last

        self.data_len: int = len(dataset)  # Length of the dataset
        self.indices: np.ndarray = np.arange(self.data_len) # available indices in the dataset
        self.pose: int = 0  # record the current position in the dataset

    def __len__(self):
        if self.drop_last:
            batches = len(self.dataset) // self.batch_size  # get the floor of division
        else:
            batches = -(
                len(self.dataset) // -self.batch_size
            )  # get the ceil of division
        return batches

    def __next__(self):
        if self.pose <= self.data_len:
            if self.shuffle:
                self.indices = np.random.permutation(self.indices)
            batch_data = self.dataset[self.indices[: self.batch_size]]
            self.indices = self.indices[self.batch_size :]
            if self.drop_last and len(self.indices) < self.batch_size:
                self.pose = 0
                self.indices = np.arange(self.data_len)
                raise StopIteration
            self.pose += self.batch_size
            return batch_data
        else:
            self.pose = 0
            self.indices = np.arange(self.data_len)
            raise StopIteration

    def __iter__(self):
        return self

# %% ../../nbs/01_data.loader.ipynb 13
# copy from https://jax.readthedocs.io/en/latest/notebooks/Neural_Network_and_Data_Loading.html#data-loading-with-pytorch
def _numpy_collate(batch):
    if isinstance(batch[0], np.ndarray):
        return np.stack(batch)
    elif isinstance(batch[0], (tuple, list)):
        transposed = zip(*batch)
        return [_numpy_collate(samples) for samples in transposed]
    else:
        return np.array(batch)

def _convert_dataset_pytorch(dataset: Dataset):
    class DatasetPytorch(torch_data.Dataset):
        def __init__(self, dataset: Dataset): self.dataset = dataset
        def __len__(self): return len(self.dataset)
        def __getitem__(self, idx): return self.dataset[idx]
    
    return DatasetPytorch(dataset)

# %% ../../nbs/01_data.loader.ipynb 14
class DataLoaderPytorch(BaseDataLoader):
    """Pytorch Dataloader"""
    def __init__(
        self, 
        dataset: Dataset,
        backend: str = 'pytorch', # positional argument
        batch_size: int = 1,  # batch size
        shuffle: bool = False,  # if true, dataloader shuffles before sampling each batch
        num_workers: int = 0, # number of workers
        drop_last: bool = False, # drop last batch or not
        **kwargs
    ):
        if torch_data is None:
            raise ModuleNotFoundError("`pytorch` library needs to be installed. Try `pip install torch`."
            "Please refer to pytorch documentation for details: https://pytorch.org/get-started/.")
        
        dataset = _convert_dataset_pytorch(dataset)
        self.dataloader = torch_data.DataLoader(
            dataset, 
            batch_size=batch_size, 
            shuffle=shuffle, 
            num_workers=num_workers, 
            drop_last=drop_last,
            collate_fn=_numpy_collate,
            **kwargs
        ) 

    def __len__(self):
        return len(self.dataloader)

    def __next__(self):
        return next(self.dataloader)

    def __iter__(self):
        return self.dataloader.__iter__()

# %% ../../nbs/01_data.loader.ipynb 16
backend2dataloader = {
    'jax': DataLoaderJax,
    'pytorch': DataLoaderPytorch,
    'tensorflow': None,
    'merlin': None,
}

# %% ../../nbs/01_data.loader.ipynb 17
def _supported_backends():
    return [back for back, dl_cls in backend2dataloader.items() if dl_cls is not None ]

# %% ../../nbs/01_data.loader.ipynb 18
def _dispatch_dataloader(
    backend: str # dataloader backend
) -> BaseDataLoader:
    """Return Dataloader class based on given `backend`"""
    dataloader_backends = backend2dataloader.keys()
    if not backend in dataloader_backends:
        raise ValueError(f"backend=`{backend}` is an invalid backend for dataloader. "
            f"Should be one of {dataloader_backends}.")
    
    dataloader_cls = backend2dataloader[backend]
    if dataloader_cls is None:
        raise NotImplementedError(f'backend=`{backend}` is not supported yet.')
    return dataloader_cls


# %% ../../nbs/01_data.loader.ipynb 20
class DataLoader(BaseDataLoader):
    """Main Dataloader class to load Numpy data batches"""
    def __init__(
        self,
        dataset: Dataset,
        backend: str, # Dataloader backend; Currently supports `['jax', 'pytorch']`
        batch_size: int = 1,  # batch size
        shuffle: bool = False,  # if true, dataloader shuffles before sampling each batch
        num_workers: int = 0, # number of workers
        drop_last: bool = False, # drop last batches or not
        **kwargs
    ):
        self.__class__ = _dispatch_dataloader(backend)
        self.__init__(
            dataset=dataset, 
            backend=backend, 
            batch_size=batch_size, 
            shuffle=shuffle, 
            num_workers=num_workers,
            drop_last=drop_last,
            **kwargs
        )
