#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""
Module contains build objects for makeproject.

@package makeprojects.build_objects
"""

# pylint: disable=consider-using-f-string
# pylint: disable=useless-object-inheritance

from __future__ import absolute_import, print_function, unicode_literals

import os
import sys
from burger import run_command

########################################


class BuildError(object):
    """
    Error message generated by builders.

    When a builder completes, a BuildError class is created and appended
    to the ``results`` list for logging.

    Attributes:
        error: Integer error code.
        filename: File name that generated the error.
        configuration: Build configuration.
        msg: Error message.
    """

    def __init__(self, error, filename, configuration=None, msg=None):
        """
        Initializers for an BuildError.

        Args:
            error: Integer error code, zero if not error
            filename: File that generated the error
            configuration: If applicable, configuration that was compiled
            msg: Error message test, if available

        """

        # Init all the values
        self.error = error
        self.filename = filename
        self.configuration = configuration
        self.msg = msg

    ########################################

    def get_error_code(self):
        """
        Return the integer error code.

        Returns:
            The integer error code.
        """
        return self.error

    ########################################

    def __repr__(self):
        """
        Convert the error into a string.

        Returns:
            A full error string.
        """

        if self.error:
            result = "Error #{} in file {}".format(self.error, self.filename)
        else:
            result = "No error in file {}".format(self.filename)

        if self.configuration:
            result += " Configuration \"{}\"".format(self.configuration)
        if self.msg:
            result += " \"{}\"".format(self.msg)

        return result

    def __str__(self):
        """
        Convert the error into a string.

        Returns:
            A full error string.
        """

        return self.__repr__()

########################################


class BuildObject(object):
    """
    Object describing something to build.

    When the directory is parsed, a list of BuildObjects is
    generated and then sorted by priority and then built.

    Attributes:
        file_name: Name of file to build.
        priority: Numeric priorty in ascending order.
        configuration: Configuration if applicable
    """

    def __init__(self, file_name, priority=None,
                 configuration=None):
        """
        Initializers for an BuildObject.

        Args:
            file_name: Name of the file to build.
            priority: Integer priority, lower will be built first.
            configuration: Configuration to build

        Raises:
            TypeError, ValueError
        """

        # Sanity check

        # Make sure a priority is assigned
        if priority is None:
            priority = 50

        # Ensure the pathname is an absolute path
        file_name = os.path.abspath(file_name)

        self.file_name = file_name

        # Ensure it's a number
        self.priority = int(priority)
        self.configuration = configuration

    ########################################

    def build(self):
        """
        Perform the build operation

        Returns:
            BuildError object as Unimplemented build.
        """

        return BuildError(10, self.file_name, self.configuration,
                          msg="Unimplemented build")

    ########################################

    def clean(self):
        """
        Delete temporary files.
        This function is called by ``cleanme`` to remove temporary files.

        On exit, return 0 for no error, or a non zero error code if there was an
        error to report. None if not implemented or not applicable.

        Returns:
            None if not implemented, otherwise an integer error code.
        """
        return BuildError(10, self.file_name, self.configuration,
                          msg="Unimplemented clean")

    ########################################

    def run_command(self, cmd, verbose):
        """
        Issue a command and return the generated BuildError

        Args:
            cmd: command line to execute
            verbose: True if verbose output is required
        Returns:
            BuildError object with error condition, if any.
        """

        # Perform the command
        try:
            error_code = run_command(
                cmd, working_dir=os.path.dirname(self.file_name),
                quiet=not verbose)[0]
            msg = None
        except OSError as error:
            error_code = getattr(error, "winerror", error.errno)
            msg = str(error)
            print(msg, file=sys.stderr)

        # Return the results
        return BuildError(error_code, self.file_name,
                          configuration=self.configuration, msg=msg)

    ########################################

    def __repr__(self):
        """
        Convert the object into a string.

        Returns:
            A full string.
        """

        result = (
            "{} for file \"{}\" with priority {}").format(
                type(self).__name__,
                self.file_name,
                self.priority)

        if self.configuration:
            result += " configuration \"{}\"".format(self.configuration)

        return result

    def __str__(self):
        """
        Convert the object into a string.

        Returns:
            A full string.
        """

        return self.__repr__()
