"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.updateChangelog = void 0;
const path_1 = require("path");
const fs_extra_1 = require("fs-extra");
const logging = require("../logging");
const utils = require("../util");
/**
 * Prepends input changelog entry onto project-level changelog.
 *
 * Currently assumes a headerless changelog formatted according to
 * [conventional-changelog](https://github.com/conventional-changelog/conventional-changelog)
 * rules.
 *
 * @param cwd working directory (git repository)
 * @param options options
 */
async function updateChangelog(cwd, options) {
    const inputChangelog = path_1.join(cwd, options.inputChangelog);
    const outputChangelog = path_1.join(cwd, options.outputChangelog);
    const versionFile = path_1.join(cwd, options.versionFile);
    let version = (await utils.tryReadFile(versionFile)).trim();
    if (!version) {
        throw new Error(`Unable to determine version from ${versionFile}. Cannot proceed with changelog update. Did you run 'bump'?`);
    }
    const inputChangelogContent = await fs_extra_1.readFile(inputChangelog, 'utf-8');
    const changelogVersionSearchPattern = `[${version}]`;
    if (!inputChangelogContent.includes(changelogVersionSearchPattern)) {
        throw new Error(`Supplied version ${version} was not found in input changelog. You may want to check it's content.`);
    }
    const outputChangelogContent = await fs_extra_1.readFile(outputChangelog, 'utf-8');
    if (outputChangelogContent.indexOf(changelogVersionSearchPattern) > -1) {
        logging.info(`Changelog already contains an entry for ${version}. Skipping changelog update.`);
        return;
    }
    const newChangelog = inputChangelogContent.trimEnd() + '\n\n' + outputChangelogContent.trimStart();
    await fs_extra_1.writeFile(outputChangelog, newChangelog);
    utils.exec(`git add ${outputChangelog} && git commit -m "chore(release): ${version}"`, { cwd });
}
exports.updateChangelog = updateChangelog;
//# sourceMappingURL=data:application/json;base64,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