"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Release = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const component_1 = require("../component");
const github_1 = require("../github");
const workflows_model_1 = require("../github/workflows-model");
const version_1 = require("../version");
const publisher_1 = require("./publisher");
const release_trigger_1 = require("./release-trigger");
const BUILD_JOBID = 'release';
const GIT_REMOTE_STEPID = 'git_remote';
const LATEST_COMMIT_OUTPUT = 'latest_commit';
/**
 * (experimental) Manages releases (currently through GitHub workflows).
 *
 * By default, no branches are released. To add branches, call `addBranch()`.
 *
 * @experimental
 */
class Release extends component_1.Component {
    /**
     * @experimental
     */
    constructor(project, options) {
        var _b, _c, _d, _e, _f, _g, _h, _j, _k;
        super(project);
        this._branches = new Array();
        this.jobs = {};
        if (Array.isArray(options.releaseBranches)) {
            throw new Error('"releaseBranches" is no longer an array. See type annotations');
        }
        this.github = project.github;
        this.buildTask = options.task;
        this.preBuildSteps = (_b = options.releaseWorkflowSetupSteps) !== null && _b !== void 0 ? _b : [];
        this.postBuildSteps = (_c = options.postBuildSteps) !== null && _c !== void 0 ? _c : [];
        this.antitamper = (_d = options.antitamper) !== null && _d !== void 0 ? _d : true;
        this.artifactsDirectory = (_e = options.artifactsDirectory) !== null && _e !== void 0 ? _e : 'dist';
        this.versionFile = options.versionFile;
        this.releaseTrigger = (_f = options.releaseTrigger) !== null && _f !== void 0 ? _f : release_trigger_1.ReleaseTrigger.continuous();
        this.containerImage = options.workflowContainerImage;
        this.workflowRunsOn = options.workflowRunsOn;
        /**
         * Use manual releases with no changelog if releaseEveryCommit is explicitly
         * disabled and no other trigger is set.
         *
         * TODO: Remove this when releaseEveryCommit and releaseSchedule are removed
         */
        if (!(((_g = options.releaseEveryCommit) !== null && _g !== void 0 ? _g : true) || options.releaseSchedule || options.releaseTrigger)) {
            this.releaseTrigger = release_trigger_1.ReleaseTrigger.manual({ changelog: false });
        }
        if (options.releaseSchedule) {
            this.releaseTrigger = release_trigger_1.ReleaseTrigger.scheduled({ schedule: options.releaseSchedule });
        }
        this.version = new version_1.Version(project, {
            versionInputFile: this.versionFile,
            artifactsDirectory: this.artifactsDirectory,
            versionrcOptions: options.versionrcOptions,
        });
        this.publisher = new publisher_1.Publisher(project, {
            artifactName: this.artifactsDirectory,
            condition: `needs.${BUILD_JOBID}.outputs.${LATEST_COMMIT_OUTPUT} == github.sha`,
            buildJobId: BUILD_JOBID,
            jsiiReleaseVersion: options.jsiiReleaseVersion,
            failureIssue: options.releaseFailureIssue,
            failureIssueLabel: options.releaseFailureIssueLabel,
            workflowRunsOn: options.workflowRunsOn,
            publishTasks: options.publishTasks,
        });
        const githubRelease = (_h = options.githubRelease) !== null && _h !== void 0 ? _h : true;
        if (githubRelease) {
            this.publisher.publishToGitHubReleases({
                changelogFile: path.posix.join(this.artifactsDirectory, this.version.changelogFileName),
                versionFile: path.posix.join(this.artifactsDirectory, this.version.versionFileName),
                releaseTagFile: path.posix.join(this.artifactsDirectory, this.version.releaseTagFileName),
            });
        }
        // add the default branch (we need the internal method which does not require majorVersion)
        this.defaultBranch = this._addBranch(options.branch, {
            prerelease: options.prerelease,
            majorVersion: options.majorVersion,
            workflowName: (_j = options.releaseWorkflowName) !== null && _j !== void 0 ? _j : 'release',
            tagPrefix: options.releaseTagPrefix,
            npmDistTag: options.npmDistTag,
        });
        for (const [name, opts] of Object.entries((_k = options.releaseBranches) !== null && _k !== void 0 ? _k : {})) {
            this.addBranch(name, opts);
        }
    }
    /**
     * (experimental) Adds a release branch.
     *
     * It is a git branch from which releases are published. If a project has more than one release
     * branch, we require that `majorVersion` is also specified for the primary branch in order to
     * ensure branches always release the correct version.
     *
     * @param branch The branch to monitor (e.g. `main`, `v2.x`).
     * @param options Branch definition.
     * @experimental
     */
    addBranch(branch, options) {
        this._addBranch(branch, options);
    }
    /**
     * Adds a release branch.
     *
     * It is a git branch from which releases are published. If a project has more than one release
     * branch, we require that `majorVersion` is also specified for the primary branch in order to
     * ensure branches always release the correct version.
     *
     * @param branch The branch to monitor (e.g. `main`, `v2.x`)
     * @param options Branch definition
     */
    _addBranch(branch, options) {
        if (this._branches.find(b => b.name === branch)) {
            throw new Error(`The release branch ${branch} is already defined`);
        }
        // if we add a branch, we require that the default branch will also define a
        // major version.
        if (this.defaultBranch && options.majorVersion && this.defaultBranch.majorVersion === undefined) {
            throw new Error('you must specify "majorVersion" for the default branch when adding multiple release branches');
        }
        const releaseBranch = {
            name: branch,
            ...options,
            workflow: this.createWorkflow(branch, options),
        };
        this._branches.push(releaseBranch);
        return releaseBranch;
    }
    /**
     * (experimental) Called before synthesis.
     *
     * @experimental
     */
    preSynthesize() {
        for (const branch of this._branches) {
            if (!branch.workflow) {
                continue;
            }
            branch.workflow.addJobs(this.publisher._renderJobsForBranch(branch.name, branch));
            branch.workflow.addJobs(this.jobs);
        }
    }
    /**
     * (experimental) Adds jobs to all release workflows.
     *
     * @param jobs The jobs to add (name => job).
     * @experimental
     */
    addJobs(jobs) {
        for (const [name, job] of Object.entries(jobs)) {
            this.jobs[name] = job;
        }
    }
    /**
     * (experimental) Retrieve all release branch names.
     *
     * @experimental
     */
    get branches() {
        return this._branches.map(b => b.name);
    }
    /**
     * @returns a workflow or `undefined` if github integration is disabled.
     */
    createWorkflow(branchName, branch) {
        var _b;
        const workflowName = (_b = branch.workflowName) !== null && _b !== void 0 ? _b : `release-${branchName}`;
        // to avoid race conditions between two commits trying to release the same
        // version, we check if the head sha is identical to the remote sha. if
        // not, we will skip the release and just finish the build.
        const noNewCommits = `\${{ steps.${GIT_REMOTE_STEPID}.outputs.${LATEST_COMMIT_OUTPUT} == github.sha }}`;
        // The arrays are being cloned to avoid accumulating values from previous branches
        const preBuildSteps = [...this.preBuildSteps];
        const env = {
            RELEASE: 'true',
        };
        if (branch.majorVersion !== undefined) {
            env.MAJOR = branch.majorVersion.toString();
        }
        if (branch.prerelease) {
            env.PRERELEASE = branch.prerelease;
        }
        if (branch.tagPrefix) {
            env.RELEASE_TAG_PREFIX = branch.tagPrefix;
        }
        // the "release" task prepares a release but does not publish anything. the
        // output of the release task is: `dist`, `.version.txt`, and
        // `.changelog.md`. this is what publish tasks expect.
        // if this is the release for "main" or "master", just call it "release".
        // otherwise, "release:BRANCH"
        const releaseTaskName = (branchName === 'main' || branchName === 'master') ? 'release' : `release:${branchName}`;
        const releaseTask = this.project.addTask(releaseTaskName, {
            description: `Prepare a release from "${branchName}" branch`,
            env,
        });
        releaseTask.exec(`rm -fr ${this.artifactsDirectory}`);
        releaseTask.spawn(this.version.bumpTask);
        releaseTask.spawn(this.buildTask);
        releaseTask.spawn(this.version.unbumpTask);
        if (this.releaseTrigger.isManual) {
            const publishTask = this.publisher.publishToGit({
                changelogFile: path.posix.join(this.artifactsDirectory, this.version.changelogFileName),
                versionFile: path.posix.join(this.artifactsDirectory, this.version.versionFileName),
                releaseTagFile: path.posix.join(this.artifactsDirectory, this.version.releaseTagFileName),
                projectChangelogFile: this.releaseTrigger.changelogPath,
                gitBranch: branchName,
                gitPushCommand: this.releaseTrigger.gitPushCommand,
            });
            releaseTask.spawn(publishTask);
        }
        // anti-tamper check (fails if there were changes to committed files)
        // this will identify any non-committed files generated during build (e.g. test snapshots)
        if (this.antitamper) {
            releaseTask.exec('git diff --ignore-space-at-eol --exit-code');
        }
        const postBuildSteps = [...this.postBuildSteps];
        // check if new commits were pushed to the repo while we were building.
        // if new commits have been pushed, we will cancel this release
        postBuildSteps.push({
            name: 'Check for new commits',
            id: GIT_REMOTE_STEPID,
            run: `echo ::set-output name=${LATEST_COMMIT_OUTPUT}::"$(git ls-remote origin -h \${{ github.ref }} | cut -f1)"`,
        });
        postBuildSteps.push({
            name: 'Upload artifact',
            if: noNewCommits,
            uses: 'actions/upload-artifact@v2.1.1',
            with: {
                name: this.artifactsDirectory,
                path: this.artifactsDirectory,
            },
        });
        if (this.github && !this.releaseTrigger.isManual) {
            return new github_1.TaskWorkflow(this.github, {
                name: workflowName,
                jobId: BUILD_JOBID,
                outputs: {
                    latest_commit: {
                        stepId: GIT_REMOTE_STEPID,
                        outputName: LATEST_COMMIT_OUTPUT,
                    },
                },
                triggers: {
                    schedule: this.releaseTrigger.schedule ? [{ cron: this.releaseTrigger.schedule }] : undefined,
                    push: this.releaseTrigger.isContinuous ? { branches: [branchName] } : undefined,
                },
                container: this.containerImage ? { image: this.containerImage } : undefined,
                env: {
                    CI: 'true',
                },
                permissions: {
                    contents: workflows_model_1.JobPermission.WRITE,
                },
                checkoutWith: {
                    // we must use 'fetch-depth=0' in order to fetch all tags
                    // otherwise tags are not checked out
                    'fetch-depth': 0,
                },
                preBuildSteps,
                task: releaseTask,
                postBuildSteps,
                runsOn: this.workflowRunsOn,
            });
        }
        else {
            return undefined;
        }
    }
}
exports.Release = Release;
_a = JSII_RTTI_SYMBOL_1;
Release[_a] = { fqn: "projen.release.Release", version: "0.40.1" };
//# sourceMappingURL=data:application/json;base64,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