"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.renderJavaScriptOptions = exports.resolveNewProject = exports.renderProjenNewOptions = void 0;
const inventory = require("../inventory");
const option_hints_1 = require("../option-hints");
const PROJEN_NEW = '__new__';
const TAB = makePadding(2);
/**
 * Renders options as if the project was created via `projen new` (embeds the __new__ field).
 */
function renderProjenNewOptions(fqn, args, comments = option_hints_1.NewProjectOptionHints.NONE) {
    return {
        ...args,
        [PROJEN_NEW]: { fqn, args, comments },
    };
}
exports.renderProjenNewOptions = renderProjenNewOptions;
function resolveNewProject(opts) {
    const f = opts[PROJEN_NEW];
    if (!f) {
        return undefined;
    }
    const type = inventory.resolveProjectType(f.fqn);
    if (!type) {
        throw new Error(`unable to resolve project type for ${f.fqn}`);
    }
    return {
        args: f.args,
        fqn: f.fqn,
        type: type,
        comments: f.comments,
    };
}
exports.resolveNewProject = resolveNewProject;
/**
 * Prints all parameters that can be used in a project type, alongside their descriptions.
 *
 * Parameters in `params` that aren't undefined are rendered as defaults,
 * while all other parameters are rendered as commented out.
 *
 * Returns the printed output and a set of required imports as an object
 * in the form { options, imports }.
 */
function renderJavaScriptOptions(opts) {
    var _a, _b, _c, _d;
    const renders = {};
    const optionsWithDefaults = [];
    const allImports = new Set();
    for (const option of opts.type.options) {
        if (option.deprecated) {
            continue;
        }
        const optionName = option.name;
        if (opts.args[optionName] !== undefined) {
            const arg = opts.args[optionName];
            const { js, imports } = renderArgAsJavaScript(arg, option);
            for (const importStr of imports) {
                allImports.add(importStr);
            }
            renders[optionName] = `${optionName}: ${js},`;
            optionsWithDefaults.push(optionName);
        }
        else {
            const defaultValue = ((_a = option.default) === null || _a === void 0 ? void 0 : _a.startsWith('-')) ? undefined : ((_b = option.default) !== null && _b !== void 0 ? _b : undefined);
            renders[optionName] = `// ${optionName}: ${defaultValue},`;
        }
    }
    const bootstrap = (_c = opts.bootstrap) !== null && _c !== void 0 ? _c : false;
    if (bootstrap) {
        for (const arg of ((_d = opts.omitFromBootstrap) !== null && _d !== void 0 ? _d : [])) {
            delete opts.args[arg];
        }
        renders[PROJEN_NEW] = `${PROJEN_NEW}: ${JSON.stringify({ args: opts.args, fqn: opts.type.fqn, comments: opts.comments })},`;
        optionsWithDefaults.push(PROJEN_NEW);
    }
    // generate rendering
    const result = [];
    result.push('{');
    // render options with defaults
    optionsWithDefaults.sort();
    for (const optionName of optionsWithDefaults) {
        result.push(`${TAB}${renders[optionName]}`);
    }
    if (result.length > 1) {
        result.push('');
    }
    // render options without defaults as comments
    if (opts.comments === option_hints_1.NewProjectOptionHints.ALL) {
        const options = opts.type.options.filter((opt) => !opt.deprecated && opts.args[opt.name] === undefined);
        result.push(...renderCommentedOptionsByModule(renders, options));
    }
    else if (opts.comments === option_hints_1.NewProjectOptionHints.FEATURED) {
        const options = opts.type.options.filter((opt) => !opt.deprecated && opts.args[opt.name] === undefined && opt.featured);
        result.push(...renderCommentedOptionsInOrder(renders, options));
    }
    else if (opts.comments === option_hints_1.NewProjectOptionHints.NONE) {
        // don't render any extra options
    }
    if (result[result.length - 1] === '') {
        result.pop();
    }
    result.push('}');
    return { renderedOptions: result.join('\n'), imports: allImports };
}
exports.renderJavaScriptOptions = renderJavaScriptOptions;
function renderCommentedOptionsByModule(renders, options) {
    var _a;
    const optionsByModule = {};
    for (const option of options) {
        const parentModule = option.parent;
        optionsByModule[parentModule] = (_a = optionsByModule[parentModule]) !== null && _a !== void 0 ? _a : [];
        optionsByModule[parentModule].push(option);
    }
    for (const parentModule in optionsByModule) {
        optionsByModule[parentModule].sort((o1, o2) => o1.name.localeCompare(o2.name));
    }
    const result = [];
    const marginSize = Math.max(...options.map((opt) => renders[opt.name].length));
    for (const [moduleName, optionGroup] of Object.entries(optionsByModule).sort()) {
        result.push(`${TAB}/* ${moduleName} */`);
        for (const option of optionGroup) {
            const paramRender = renders[option.name];
            const docstring = option.docs || 'No documentation found.';
            result.push(`${TAB}${paramRender}${makePadding(marginSize - paramRender.length + 2)}/* ${docstring} */`);
        }
        result.push('');
    }
    return result;
}
function renderCommentedOptionsInOrder(renders, options) {
    const result = [];
    const marginSize = Math.max(...options.map((opt) => renders[opt.name].length));
    for (const option of options) {
        const paramRender = renders[option.name];
        const docstring = option.docs || 'No documentation found.';
        result.push(`${TAB}${paramRender}${makePadding(marginSize - paramRender.length + 2)}/* ${docstring} */`);
    }
    return result;
}
/**
 * Renders a value as a JavaScript value, converting strings to enums where
 * appropriate. The type must be JSON-like (string, number, boolean, array,
 * enum, or JSON object).
 *
 * Returns a JavaScript expression as a string, and the names of any
 * necessary imports.
 */
function renderArgAsJavaScript(arg, option) {
    if (option.kind === 'enum') {
        if (!option.fqn) {
            throw new Error(`fqn field is missing from enum option ${option.name}`);
        }
        const parts = option.fqn.split('.'); // -> ['projen', 'web', 'MyEnum']
        const enumChoice = String(arg).toUpperCase().replace(/-/g, '_'); // custom-value -> CUSTOM_VALUE
        const js = `${parts.slice(1).join('.')}.${enumChoice}`; // -> web.MyEnum.CUSTOM_VALUE
        const importName = parts[1]; // -> web
        return { js, imports: [importName] };
    }
    else if (option.jsonLike) {
        return { js: JSON.stringify(arg), imports: [] };
    }
    else {
        throw new Error(`Unexpected option ${option.name} - cannot render a value for this option because it does not have a JSON-like type.`);
    }
}
function makePadding(paddingLength) {
    return ' '.repeat(paddingLength);
}
//# sourceMappingURL=data:application/json;base64,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