"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.readJsiiManifest = exports.resolveProjectType = exports.discover = void 0;
const path = require("path");
const case_1 = require("case");
const fs = require("fs-extra");
const PROJEN_MODULE_ROOT = path.join(__dirname, '..');
const PROJECT_BASE_FQN = 'projen.Project';
/**
 * Returns a list of project types exported the modules defined in `moduleDirs`.
 * This list will always also include the built-in projen project types.
 * Modules without a .jsii manifest are skipped.
 *
 * @param moduleDirs A list of npm module directories
 */
function discover(...moduleDirs) {
    const jsii = discoverJsiiTypes(...moduleDirs);
    const result = new Array();
    for (const fqn of Object.keys(jsii)) {
        if (isProjectType(jsii, fqn)) {
            const p = toProjectType(jsii, fqn);
            result.push(p);
        }
    }
    return result.sort((r1, r2) => r1.pjid.localeCompare(r2.pjid));
}
exports.discover = discover;
/**
 * Resolve all jsii types from @modulesDirs.
 * When a jsii module is found it will recusively list the types from the dependant module as well
 *
 * @param moduleDirs
 * @returns
 */
function discoverJsiiTypes(...moduleDirs) {
    const jsii = {};
    const discoveredManifests = [];
    const discoverJsii = (dir) => {
        const jsiiFile = path.join(dir, '.jsii');
        if (!fs.existsSync(jsiiFile)) {
            return;
        } // no jsii manifest
        const manifest = fs.readJsonSync(jsiiFile);
        if (discoveredManifests.includes(manifest.fingerprint)) {
            return;
        }
        discoveredManifests.push(manifest.fingerprint);
        for (const [fqn, type] of Object.entries(manifest.types)) {
            jsii[fqn] = {
                ...type,
            };
        }
        // Also search recursively in nested project dependencies. If the requested module is an external module
        // this will also end-up in the projen module and add the projen types
        if (manifest.dependencies) {
            for (const dependency of Object.keys(manifest.dependencies)) {
                const nestedDependencyFolder = path.dirname(require.resolve(`${dependency}/package.json`, {
                    paths: [dir],
                }));
                if (fs.existsSync(nestedDependencyFolder)) {
                    discoverJsii(nestedDependencyFolder);
                }
            }
        }
    };
    // read all .jsii manifests from all requested modules and merge
    // them all into a single map of fqn->type.
    for (const dir of [...moduleDirs, PROJEN_MODULE_ROOT]) {
        discoverJsii(dir);
        // Read from scoped packages
        if (dir.includes('@') && fs.lstatSync(dir).isDirectory()) {
            const childDirs = fs.readdirSync(dir).map(file => path.join(dir, file));
            for (const child of childDirs) {
                discoverJsii(child);
            }
        }
    }
    return jsii;
}
function resolveProjectType(projectFqn) {
    let [moduleName] = projectFqn.split('.');
    if (moduleName === 'projen') {
        moduleName = PROJEN_MODULE_ROOT;
    }
    // try picking the manifest. We only need the base folder but this is directly a nice check if we request from a valid jsii package
    const jsiiManifestFile = require.resolve(`${moduleName}/.jsii`, {
        paths: [
            process.cwd(),
        ],
    });
    const moduleFolder = path.dirname(jsiiManifestFile);
    // Read all jsii types that can be loaded from this project type
    const jsii = discoverJsiiTypes(moduleFolder);
    return toProjectType(jsii, projectFqn);
}
exports.resolveProjectType = resolveProjectType;
function toProjectType(jsii, fqn) {
    var _a, _b, _c, _d;
    if (!isProjectType(jsii, fqn)) {
        throw new Error(`Fully qualified name "${fqn}" is not a valid project type.`);
    }
    const typeinfo = jsii[fqn];
    // projen.web.ReactProject -> web.ReactProject
    const typename = fqn.substring(fqn.indexOf('.') + 1);
    const docsurl = `https://github.com/projen/projen/blob/main/API.md#projen-${typename.toLocaleLowerCase().replace(/\./g, '-')}`;
    let pjid = (_c = (_b = (_a = typeinfo.docs) === null || _a === void 0 ? void 0 : _a.custom) === null || _b === void 0 ? void 0 : _b.pjid) !== null && _c !== void 0 ? _c : case_1.snake(typename).replace(/_project$/, '');
    return {
        moduleName: typeinfo.assembly,
        typename,
        pjid,
        fqn,
        options: discoverOptions(jsii, fqn).sort((o1, o2) => o1.name.localeCompare(o2.name)),
        docs: (_d = typeinfo.docs) === null || _d === void 0 ? void 0 : _d.summary,
        docsurl,
    };
}
function readJsiiManifest(jsiiFqn) {
    let [moduleName] = jsiiFqn.split('.');
    if (moduleName === 'projen') {
        moduleName = PROJEN_MODULE_ROOT;
    }
    const jsiiManifestFile = require.resolve(`${moduleName}/.jsii`);
    return fs.readJsonSync(jsiiManifestFile);
}
exports.readJsiiManifest = readJsiiManifest;
function discoverOptions(jsii, fqn) {
    var _a, _b, _c, _d;
    const options = {};
    const params = (_c = (_b = (_a = jsii[fqn]) === null || _a === void 0 ? void 0 : _a.initializer) === null || _b === void 0 ? void 0 : _b.parameters) !== null && _c !== void 0 ? _c : [];
    const optionsParam = params[0];
    const optionsTypeFqn = (_d = optionsParam === null || optionsParam === void 0 ? void 0 : optionsParam.type) === null || _d === void 0 ? void 0 : _d.fqn;
    if (params.length > 1 || (params.length === 1 && (optionsParam === null || optionsParam === void 0 ? void 0 : optionsParam.name) !== 'options')) {
        throw new Error(`constructor for project ${fqn} must have a single "options" argument of a struct type. got ${JSON.stringify(params)}`);
    }
    addOptions(optionsTypeFqn);
    const opts = Object.values(options);
    return opts.sort((a, b) => a.switch.localeCompare(b.switch));
    function addOptions(ofqn, basePath = [], optional = false) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k;
        if (!ofqn) {
            return;
        }
        const struct = jsii[ofqn];
        if (!struct) {
            throw new Error(`unable to find options type ${ofqn} for project ${fqn}`);
        }
        for (const prop of (_a = struct.properties) !== null && _a !== void 0 ? _a : []) {
            const propPath = [...basePath, prop.name];
            // protect against double-booking
            if (prop.name in options) {
                throw new Error(`duplicate option "${prop.name}" in ${fqn} (already declared in ${options[prop.name].parent})`);
            }
            let jsiiKind;
            if ((_b = prop.type) === null || _b === void 0 ? void 0 : _b.fqn) {
                jsiiKind = jsii[(_c = prop.type) === null || _c === void 0 ? void 0 : _c.fqn].kind; // e.g. 'class', 'interface', 'enum'
            }
            const isOptional = optional || prop.optional;
            let defaultValue = (_d = prop.docs) === null || _d === void 0 ? void 0 : _d.default;
            if (defaultValue === 'undefined') {
                defaultValue = undefined;
            }
            // if this is a mandatory option and we have a default value, it has to be JSON-parsable to the correct type
            if (!isOptional && defaultValue) {
                if (!((_e = prop.type) === null || _e === void 0 ? void 0 : _e.primitive)) {
                    throw new Error(`required option "${prop.name}" with a @default must use primitive types (string, number or boolean). type found is: ${JSON.stringify(prop.type)}`);
                }
                checkDefaultIsParsable(prop.name, defaultValue, (_f = prop.type) === null || _f === void 0 ? void 0 : _f.primitive);
            }
            options[prop.name] = filterUndefined({
                path: propPath,
                parent: struct.name,
                name: prop.name,
                fqn: (_g = prop.type) === null || _g === void 0 ? void 0 : _g.fqn,
                docs: prop.docs.summary,
                simpleType: prop.type ? getSimpleTypeName(prop.type) : 'unknown',
                fullType: prop.type,
                kind: jsiiKind,
                jsonLike: prop.type ? isJsonLike(jsii, prop.type) : undefined,
                switch: propPath.map(p => case_1.snake(p).replace(/_/g, '-')).join('-'),
                default: defaultValue,
                optional: isOptional,
                featured: ((_j = (_h = prop.docs) === null || _h === void 0 ? void 0 : _h.custom) === null || _j === void 0 ? void 0 : _j.featured) === 'true',
                deprecated: prop.docs.stability === 'deprecated' ? true : undefined,
            });
        }
        for (const ifc of (_k = struct.interfaces) !== null && _k !== void 0 ? _k : []) {
            addOptions(ifc);
        }
    }
}
function getSimpleTypeName(type) {
    if (type === null || type === void 0 ? void 0 : type.primitive) {
        return type.primitive; // e.g. 'string', 'boolean', 'number'
    }
    else if (type === null || type === void 0 ? void 0 : type.fqn) {
        return type.fqn.split('.').pop(); // projen.NodeProjectOptions -> NodeProjectOptions
    }
    else { // any other types such as collection types
        return 'unknown';
    }
}
/**
 * Whether a value of this type is serializable into JSON.
 */
function isJsonLike(jsii, type) {
    if (type.primitive) { // string, boolean, number, any
        return true;
    }
    else if (type.fqn) {
        const kind = jsii[type.fqn].kind;
        if (['interface', 'enum'].includes(kind)) { // not 'class'
            return true;
        }
    }
    else if (type.collection) {
        return isJsonLike(jsii, type.collection.elementtype);
    }
    return false;
}
function filterUndefined(obj) {
    const ret = {};
    for (const [k, v] of Object.entries(obj)) {
        if (v !== undefined) {
            ret[k] = v;
        }
    }
    return ret;
}
function isProjectType(jsii, fqn) {
    var _a;
    const type = jsii[fqn];
    if (!type) {
        throw new Error(`Could not find project type with fqn "${fqn}" in  .jsii file.`);
    }
    if (type.kind !== 'class') {
        return false;
    }
    if (type.abstract) {
        return false;
    }
    if ((_a = type.docs) === null || _a === void 0 ? void 0 : _a.deprecated) {
        return false;
    }
    let curr = type;
    while (true) {
        if (curr.fqn === PROJECT_BASE_FQN) {
            return true;
        }
        if (!curr.base) {
            return false;
        }
        curr = jsii[curr.base];
        if (!curr) {
            return false;
        }
    }
}
function checkDefaultIsParsable(prop, value, type) {
    // macros are pass-through
    if (value.startsWith('$')) {
        return;
    }
    try {
        const parsed = JSON.parse(value);
        if (typeof (parsed) !== type) {
            throw new Error(`cannot parse @default value for mandatory option ${prop} as a ${type}: ${parsed}`);
        }
    }
    catch (e) {
        throw new Error(`unable to JSON.parse() value "${value}" specified as @default for mandatory option "${prop}": ${e.message}`);
    }
}
//# sourceMappingURL=data:application/json;base64,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